% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/snsf.R
\name{snsf}
\alias{snsf}
\title{Stochastic Frontier Model with a Skew-Normally Distributed Error Term}
\usage{
snsf(
  formula,
  data,
  subset,
  distribution = "e",
  prod = TRUE,
  start.val = NULL,
  init.sk = NULL,
  ln.var.u = NULL,
  ln.var.v = NULL,
  skew.v = NULL,
  mean.u = NULL,
  technique = c("nr"),
  vcetype = c("aim"),
  optim.method = "bfgs",
  optim.report = 1,
  optim.trace = 1,
  reltol = 1e-12,
  optim.reltol = 1e-12,
  lmtol = 1e-05,
  maxit = 199,
  print.level = 0,
  threads = 1,
  only.data = FALSE,
  digits = 4,
  ...
)
}
\arguments{
\item{formula}{an object of class \code{formula} specifying the frontier:
a typical model is \code{y ~ x1 + ...}, where \code{y} is the
log of output (or total cost), and \code{x}'s are inputs (or outputs and
input prices, in logs). See \strong{Details}.}

\item{data}{an optional \code{data.frame} containing the variables in \code{formula}.
If not found in \code{data}, variables are taken from \code{environment(formula)}.}

\item{subset}{an optional logical or numeric vector specifying a subset of observations
for which the model is estimated and efficiencies are computed.}

\item{distribution}{character scalar specifying the distribution of the inefficiency term: default \code{"e"} (exponential).
\code{"h"} (half-normal) and \code{"t"} (truncated normal) to be implemented.}

\item{prod}{logical. If \code{TRUE}, estimates correspond to a stochastic \emph{production}
frontier and technical efficiencies are returned; if \code{FALSE}, estimates
correspond to a stochastic \emph{cost} frontier and cost efficiencies are returned.
Default is \code{TRUE}.}

\item{start.val}{optional numeric vector of starting values for the optimizer.}

\item{init.sk}{numeric. Initial value for the skewness parameter of the noise component;
default is \code{0.5}.}

\item{ln.var.u}{optional one-sided formula; e.g. \code{ln.var.u = ~ z3 + z4}. Specifies exogenous
variables entering the (log) variance of the inefficiency component. If
\code{NULL}, the inefficiency variance is homoskedastic, i.e.,
\eqn{\sigma_{u0}^2 = \exp(\gamma_{u0}[0])}.}

\item{ln.var.v}{optional one-sided formula; e.g. \code{ln.var.v = ~ z1 + z2}. Specifies exogenous
variables entering the (log) variance of the random noise component. If
\code{NULL}, the noise variance is homoskedastic, i.e.,
\eqn{\sigma_{v0}^2 = \exp(\gamma_{v0}[0])}.}

\item{skew.v}{optional one-sided formula; e.g. \code{skew.v = ~ z5 + z6}. Allows the skewness
of the noise to depend linearly on exogenous variables. If \code{NULL}, the
skewness is constant across units.}

\item{mean.u}{optional one-sided formula; e.g. \code{mean.u = ~ z7 + z8}. Specifies whether the
mean of the pre-truncated normal distribution of the inefficiency term is a
linear function of exogenous variables. In cross-sectional models, used only
when \code{distribution = "t"}. If \code{NULL}, the mean is constant across units. To be implemented.}

\item{technique}{Optimization technique to use.}

\item{vcetype}{Type of variance-covariance matrix estimation.}

\item{optim.method}{character. Method passed to \code{stats::optim} when \code{optim = TRUE}.
Default is \code{"bfgs"}.}

\item{optim.report}{integer. Verbosity level for reporting during optimization (if implemented).
Default is \code{1}.}

\item{optim.trace}{integer. Trace level for optimization (if implemented). Default is \code{1}.}

\item{reltol}{numeric. Relative convergence tolerance used when maximizing the log-likelihood.}

\item{optim.reltol}{numeric. Relative tolerance specifically for \code{optim}; default \code{1e-8}.}

\item{lmtol}{numeric. Convergence tolerance based on the scaled gradient (when applicable).
Default is \code{1e-5}.}

\item{maxit}{numeric. Maximum number of iterations for the optimizer. Default is \code{199}.}

\item{print.level}{integer. Printing level: \code{1}—estimation results;
\code{2}—optimization details; \code{3}—summary of (cost/technical)
efficiencies; \code{4}—unit-specific point and interval estimates of
efficiencies. Default is \code{0}.}

\item{threads}{Number of threads for parallel computation.}

\item{only.data}{Logical; if TRUE, return only processed data.}

\item{digits}{integer. Number of digits for displaying estimates and efficiencies. Default is \code{4}.}

\item{...}{Additional arguments (currently unused).}

\item{optim}{logical. If \code{TRUE}, estimation proceeds via \code{stats::optim}; if
\code{FALSE}, an internal routine (if provided) would be used. Default is \code{FALSE}.}

\item{report}{Reporting level for optimization progress.}

\item{trace}{Logical; if TRUE, trace optimization progress.}
}
\value{
An object of class \code{"snreg"} with maximum-likelihood estimates and diagnostics:

\describe{

  \item{\code{par}}{Numeric vector of ML parameter estimates at the optimum.}
  \item{\code{coef}}{Named numeric vector equal to \code{par}.}
  \item{\code{vcov}}{Variance–covariance matrix of the estimates.}
  \item{\code{sds}}{Standard errors, \code{sqrt(diag(vcov))}.}
  \item{\code{ctab}}{Coefficient table with columns
        \code{Coef.}, \code{SE}, \code{z}, \code{P>|z|}.}

  \item{\code{ll}}{Maximized log-likelihood value.}
  \item{\code{hessian}}{(When computed) Observed Hessian or OPG used to form \code{vcov}.}
  \item{\code{value}}{(Optim-only, before aliasing) Objective value from \code{optim}.}
  \item{\code{counts}}{(Optim-only) Iteration and evaluation counts from \code{optim}.}
  \item{\code{convergence}}{Convergence code).}
  \item{\code{message}}{(Optim-only) Message returned by \code{optim}, if any.}
  \item{\code{gradient}}{(NR-only) Gradient at the solution.}
  \item{\code{gg}}{(NR-only) Gradient-related diagnostic.}
  \item{\code{gHg}}{(NR-only) Newton-step diagnostic.}
  \item{\code{theta_rel_ch}}{(NR-only) Relative parameter change metric across iterations.}

  \item{\code{resid}}{Regression residuals.}
  \item{\code{RSS}}{Residual sum of squares \code{crossprod(resid)}.}
  \item{\code{shat2}}{Residual variance estimate \code{var(resid)}.}
  \item{\code{shat}}{Residual standard deviation \code{sqrt(shat2)}.}
  \item{\code{aic}}{Akaike Information Criterion.}
  \item{\code{bic}}{Bayesian Information Criterion.}
  \item{\code{Mallows}}{Mallows' \eqn{C_p}-like statistic.}

  \item{\code{u}}{Estimated inefficiency term (vector). Returned for models with
        an inefficiency component (e.g., exponential).}
  \item{\code{eff}}{Efficiency scores \code{exp(-u)} (technical or cost, depending on \code{prod}).}
  \item{\code{sv}}{Estimated (possibly unit-specific) standard deviation of the noise term.}
  \item{\code{su}}{Estimated (possibly unit-specific) standard deviation or scale of the
        inefficiency term. For exponential models.}
  \item{\code{skewness}}{Estimated skewness index (e.g., from the skewness equation).}

  \item{\code{esample}}{Logical vector marking observations used in estimation.}
  \item{\code{n}}{Number of observations used.}
}

The returned object has class \code{"snreg"}.
}
\description{
\code{snsf} performs maximum likelihood estimation of the parameters and
technical or cost efficiencies in a Stochastic Frontier Model with a
skew-normally distributed error term.
}
\details{
Stochastic Frontier Model with a Skew-Normally Distributed Error Term


Models for \code{snsf} are specified symbolically. A typical model has the form
\code{y ~ x1 + ...}, where \code{y} represents the logarithm of outputs or total
costs and \code{\{x1, ...\}} is a set of inputs (for production) or outputs and
input prices (for cost), all typically in logs.

Options \code{ln.var.u} and \code{ln.var.v} allow for multiplicative
heteroskedasticity in the inefficiency and/or noise components; i.e., their
variances can be modeled as exponential functions of exogenous variables
(including an intercept), as in Caudill et al. (1995).
}
\examples{

\donttest{

library(snreg)

data("banks07")

# Translog cost function specification

myprod <- FALSE

spe.tl <- log(TC) ~ (log(Y1) + log(Y2) + log(W1) + log(W2))^2 +
  I(0.5 * log(Y1)^2) + I(0.5 * log(Y2)^2) +
  I(0.5 * log(W1)^2) + I(0.5 * log(W2)^2)

# Specification 1: homoskedastic noise, skewness, inefficiency

formSV <- NULL   # variance equation; constant variance
formSK <- NULL   # skewness equation; constant skewness
formSU <- NULL   # inefficiency variance equation; constant variance

m1 <- snsf(
  formula  = spe.tl,
  data     = banks07,
  prod     = myprod,
  ln.var.v = formSV,
  skew.v   = formSK,
  ln.var.u = formSU
)

# Specification 2: heteroskedastic

formSV <- ~ log(TA)      # variance equation; heteroskedastic noise (variance depends on TA)
formSK <- ~ ER           # skewness equation; with determinants (skewness is determined by ER)
formSU <- ~ LA + ER      # inefficiency variance equation; heteroskedastic noise of inefficiency 
# ([variance of] inefficiency depends on LA and ER)#' 
m2 <- snsf(
  formula  = spe.tl,
  data     = banks07,
  prod     = myprod,
  ln.var.v = formSV,
  skew.v   = formSK,
  ln.var.u = formSU
)

}

}
\references{
Badunenko, O., & Henderson, D. J. (2023).
\emph{Production analysis with asymmetric noise}.
Journal of Productivity Analysis, \bold{61}(1), 1–18.
https://doi.org/10.1007/s11123-023-00680-5

Caudill, S. B., Ford, J. M., & Gropper, D. M. (1995).
\emph{Frontier estimation and firm-specific inefficiency measures in the presence of heteroskedasticity}.
Journal of Business & Economic Statistics, \bold{13}(1), 105–111.
}
\seealso{
\code{\link[npsf]{sf}}, \code{\link{snreg}}, \code{\link{lm.mle}}
}
\author{
Oleg Badunenko <Oleg.Badunenko.brunel.ac.uk>
}
\keyword{"Stochastic}
\keyword{"efficiency}
\keyword{Analysis"}
\keyword{Frontier}
\keyword{Heteroskedasticity}
\keyword{Parametric}
\keyword{analysis"}
