#' @title Omniphobicity Data Analysis
#' @name omniphobicity_analysis
#'
#' @description
#' This function classifies surface wetting behavior based on water contact angle (WCA)
#' and oil contact angle (OCA) values, then visualizes the results using scatter plots,
#' bar plots, and Principal Component Analysis (PCA). PCA plots include group ellipses
#' to show clustering trends, with the ellipse confidence level set by the 'ellipse_level'
#' parameter (default = 0.68, corresponding to approximately 1 standard deviation).
#'
#' @param contact_data A data frame containing at least two numeric columns: 'WCA' and 'OCA'.
#' @param ellipse_level A numeric value between 0 and 1 specifying the confidence level for the ellipse in the PCA plot. Default is 0.68.
#'
#' @import ggplot2
#' @import stats
#' @import multcompView
#' @import factoextra
#' @import grid
#'
#' @importFrom dplyr %>% mutate summarise group_by n select
#' @importFrom ggplot2 ggplot aes geom_bar labs theme_minimal scale_fill_brewer
#' @importFrom ggplot2 theme element_text element_blank element_rect
#' @importFrom ggplot2 geom_point geom_hline geom_vline scale_color_brewer
#' @importFrom ggplot2 geom_boxplot geom_text annotate geom_segment
#' @importFrom stats prcomp aov sd na.omit TukeyHSD aggregate
#' @importFrom multcompView multcompLetters4
#' @importFrom factoextra fviz_pca_ind
#' @importFrom grid arrow unit
#'
#' @return A list containing the results of the omniphobicity analysis, including:
#' \itemize{
#'   \item \strong{Classification table} summarizing surface categories based on water and oil contact angles.
#'   \item \strong{Contact angle summary statistics} including minimum, maximum, mean, and standard deviation of WCA and OCA.
#'   \item \strong{Bar plot} visualizing the distribution of surface classifications.
#'   \item \strong{Scatter plot} showing the relationship between water and oil contact angles across samples.
#'   \item \strong{PCA plot} with ellipses and sample scores to visualize clustering among surface types.
#'   \item \strong{PCA loadings plot} showing contributions of WCA and OCA to the principal components.
#'   \item \strong{Boxplots} of PC1 and PC2 scores grouped by surface type.
#'   \item \strong{ANOVA and Tukey HSD test results} for PC1 and PC2 scores across surface types.
#'   \item \strong{Compact Letter Displays (CLDs)} indicating statistically significant groupings based on post-hoc tests.
#' }
#'
#' @examples
#' # EXAMPLE 1:
#' WCA = c(151.3, 154.1, 144.5, 143.0, 147.6, 85.2, 60.4, 152.1, 149.8, 148.7, 140.0, 88.0, 65.0,
#'         152.4, 70.0, 144.33, 150.50, 147.05, 149.35, 145.66, 141.2, 60.1, 92.7, 75.3, 155.0)
#' OCA = c(145.1, 143.4, 147.6, 139.0, 151.7, 80.3, 49.5, 143.8, 141.1, 143.8, 147.6, 72.5, 40.0,
#'         143.6, 65.2, 140.10, 141.65, 142.17, 138.84, 138.23, 137.8, 45.6, 91.3, 60.2, 150.2)
#' contact_data <- data.frame(WCA, OCA)
#' omniphobicity_analysis(contact_data) # ellipse_level = 0.68 (default unless specified)
#' omniphobicity_analysis(contact_data, ellipse_level = 0.95)
#'
#' # EXAMPLE 2:
#' WCA = c(65.2, 72.1, 58.7, 75.4, 68.9, 70.1, 66.5, 122.3, 135.5,
#'         140.8, 125.6, 130.2, 115.0, 138.3, 143.0, 127.6, 119.7,
#'         72.4, 85.3, 79.6, 68.5, 82.9, 76.1, 88.0, 84.2, 110.5,
#'         97.6, 104.2, 111.3, 120.7, 99.4)
#' OCA = c(72.6, 75.1, 60.8, 69.4, 80.2, 77.5, 68.3, 120.4, 130.7,
#'         133.2, 127.5, 123.3, 115.6, 134.1, 125.9, 117.4, 129.0,
#'         110.2, 103.8, 112.5, 108.4, 115.9, 111.3, 120.0, 106.6,
#'         72.1, 85.2, 70.3, 78.4, 88.6, 66.0)
#' contact_data <- data.frame(WCA, OCA)
#' omniphobicity_analysis(contact_data) # ellipse_level = 0.68 (default unless specified)
#' omniphobicity_analysis(contact_data, ellipse_level = 0.95)
#'
#' @author Paul Angelo C. Manlapaz
#' @export

utils::globalVariables(c("WCA", "OCA", "type", "WCA_class", "OCA_class", "Freq", "PC1", "PC2", "varname"))

omniphobicity_analysis <- function(contact_data, ellipse_level = 0.68) {

  # Check for required columns
  required_cols <- c("WCA", "OCA")
  if (!all(required_cols %in% colnames(contact_data))) {
    stop("Input data must contain columns: WCA (Water Contact Angle) and OCA (Oil Contact Angle).")
  }

  # Remove NA values
  contact_data <- stats::na.omit(contact_data)

 # Classification based on angle thresholds
  contact_data <- dplyr::mutate(contact_data,
                                WCA_class = ifelse(WCA > 90, "Hydrophobic", "Hydrophilic"),
                                OCA_class = ifelse(OCA > 90, "Oleophobic", "Oleophilic"),
                                type = paste(WCA_class, "&", OCA_class))

  # Summary statistics
  summary_stats <- dplyr::summarise(contact_data,
                                    WCA_mean = mean(WCA),
                                    WCA_sd = stats::sd(WCA),
                                    OCA_mean = mean(OCA),
                                    OCA_sd = stats::sd(OCA),
                                    WCA_min = min(WCA),
                                    WCA_max = max(WCA),
                                    OCA_min = min(OCA),
                                    OCA_max = max(OCA))

  # Barplot for classification
  classification_counts <- contact_data |>
    dplyr::group_by(type) |>
    dplyr::summarise(Freq = dplyr::n(), .groups = "drop")

  p1 <- ggplot2::ggplot(classification_counts, ggplot2::aes(x = type, y = Freq, fill = type)) +
    ggplot2::geom_bar(stat = "identity") +
    ggplot2::labs(title = "Surface Classification Based on Contact Angles",
                  x = "Surface Type", y = "Frequency") +
    ggplot2::theme_minimal() +
    ggplot2::scale_fill_brewer(palette = "Dark2") +
    ggplot2::theme(axis.text.x = ggplot2::element_text(angle = 45, hjust = 1),
                   panel.grid = ggplot2::element_blank(),
                   panel.border = ggplot2::element_rect(color = "black", fill = NA, linewidth = 1),
                   plot.title = ggplot2::element_text(hjust = 0.5))


  # Scatter plot of WCA vs OCA
  p2 <- ggplot2::ggplot(contact_data, ggplot2::aes(x = WCA, y = OCA, color = type)) +
    ggplot2::geom_point(size = 3) +
    ggplot2::geom_hline(yintercept = 90, linetype = "dashed", color = "gray") +
    ggplot2::geom_vline(xintercept = 90, linetype = "dashed", color = "gray") +
    ggplot2::labs(title = "Water vs Oil Contact Angles",
                  x = "Water Contact Angle (degree)", y = "Oil Contact Angle (degree)") +
    ggplot2::theme_minimal() +
    ggplot2::scale_color_brewer(palette = "Dark2", name = "Surface Type") +
    ggplot2::theme(panel.grid = ggplot2::element_blank(),
                   panel.border = ggplot2::element_rect(color = "black", fill = NA, linewidth = 1),
                   plot.title = ggplot2::element_text(hjust = 0.5))


  # PCA with confidence ellipses and loadings
  pca_result <- stats::prcomp(dplyr::select(contact_data, WCA, OCA), center = TRUE, scale. = TRUE)
  pca_scores <- as.data.frame(pca_result$x)
  pca_scores$type <- contact_data$type

  # Calculate variance explained for axis label
  pc_var <- pca_result$sdev^2 / sum(pca_result$sdev^2)
  pc1_var <- round(pc_var[1] * 100, 1)
  pc2_var <- round(pc_var[2] * 100, 1)

  # Compute PCA loadings (scaled arrows)
  loadings <- as.data.frame(pca_result$rotation)
  loadings$varname <- rownames(loadings)

  # PCA Plot
  # Define offset so arrows do not overlap axis lines
  offset <- 0.05

  # Define arrow length
  arrow_x <- 0.3
  arrow_y <- 0.3

  p3 <- factoextra::fviz_pca_ind(pca_result,
                                 geom.ind = "point",
                                 col.ind = contact_data$type,
                                 palette = "Dark2",
                                 addEllipses = TRUE,
                                 ellipse.level = ellipse_level,
                                 legend.title = "Surface Type") +
    ggplot2::labs(title = "Principal Component Analysis of Contact Angles",
                  x = paste0("PC1 (", pc1_var, "%)"),
                  y = paste0("PC2 (", pc2_var, "%)")) +
    ggplot2::theme_minimal() +
    ggplot2::theme(plot.title = ggplot2::element_text(hjust = 0.5),
                   panel.border = ggplot2::element_rect(color = "black", fill = NA, linewidth = 1),
                   legend.position = "right",
                   panel.grid = ggplot2::element_blank()) +
    ggplot2::geom_segment(data = data.frame(1),
                          ggplot2::aes(x = offset, y = 0, xend = offset + arrow_x, yend = 0),
                          arrow = grid::arrow(length = grid::unit(0.25, "cm"), type = "open"),
                          linetype = "solid", linewidth = 1, color = "maroon", inherit.aes = FALSE) +
    ggplot2::geom_segment(data = data.frame(1),
                          ggplot2::aes(x = 0, y = 0, xend = 0, yend = 0 + arrow_y),
                          arrow = grid::arrow(length = grid::unit(0.25, "cm"), type = "open"),
                          linetype = "solid", linewidth = 1, color = "purple", inherit.aes = FALSE) +
    ggplot2::annotate("text", x = offset + arrow_x + 0.05, y = 0.05, label = "WCA", color = "maroon", hjust = 0, size = 4.2) +
    ggplot2::annotate("text", x = 0.05, y = offset + arrow_y + 0.05, label = "OCA", color = "purple", hjust = 0, size = 4.2)


  # Perform ANOVA for PC1 and PC2
  aov_PC1 <- stats::aov(PC1 ~ type, data = pca_scores)
  aov_PC2 <- stats::aov(PC2 ~ type, data = pca_scores)

  # Extract p-values correctly
  pval_PC1 <- summary(aov_PC1)[[1]][["Pr(>F)"]][1]
  pval_PC2 <- summary(aov_PC2)[[1]][["Pr(>F)"]][1]

  # Tukey HSD if ANOVA is significant
  tukey_PC1 <- if (pval_PC1 < 0.05) stats::TukeyHSD(aov_PC1) else NULL
  tukey_PC2 <- if (pval_PC2 < 0.05) stats::TukeyHSD(aov_PC2) else NULL

  # Tukey CLD Plots
  cld_plot_PC1 <- NULL
  cld_plot_PC2 <- NULL

  if (!is.null(tukey_PC1)) {
    cld_PC1 <- multcompView::multcompLetters4(aov_PC1, tukey_PC1)
    group_labels <- data.frame(type = names(cld_PC1$type), letters = sapply(cld_PC1$type, paste0, collapse = ""))
    means_PC1 <- stats::aggregate(PC1 ~ type, data = pca_scores, mean)
    merged_PC1 <- merge(means_PC1, group_labels, by = "type")

    cld_plot_PC1 <- ggplot2::ggplot(pca_scores, ggplot2::aes(x = type, y = PC1, fill = type)) +
      ggplot2::geom_boxplot() +
      ggplot2::geom_text(data = merged_PC1, ggplot2::aes(x = type, y = PC1 + 0.1, label = letters), vjust = 0) +
      ggplot2::labs(title = "Tukey HSD on PC1") +
      ggplot2::theme_minimal() +
      ggplot2::scale_fill_brewer(palette = "Dark2") +
      ggplot2::theme(axis.text.x = ggplot2::element_text(angle = 45, hjust = 1),
                     panel.grid = ggplot2::element_blank(),
                     panel.border = ggplot2::element_rect(color = "black", fill = NA, size = 0.8),
                     plot.title = ggplot2::element_text(hjust = 0.5))
  }

  if (!is.null(tukey_PC2)) {
    cld_PC2 <- multcompView::multcompLetters4(aov_PC2, tukey_PC2)
    group_labels <- data.frame(type = names(cld_PC2$type), letters = sapply(cld_PC2$type, paste0, collapse = ""))
    means_PC2 <- stats::aggregate(PC2 ~ type, data = pca_scores, mean)
    merged_PC2 <- merge(means_PC2, group_labels, by = "type")

    cld_plot_PC2 <- ggplot2::ggplot(pca_scores, ggplot2::aes(x = type, y = PC2, fill = type)) +
      ggplot2::geom_boxplot() +
      ggplot2::geom_text(data = merged_PC2, ggplot2::aes(x = type, y = PC2 + 0.1, label = letters), vjust = 0) +
      ggplot2::labs(title = "Tukey HSD on PC2") +
      ggplot2::theme_minimal() +
      ggplot2::scale_fill_brewer(palette = "Dark2") +
      ggplot2::theme(axis.text.x = ggplot2::element_text(angle = 45, hjust = 1),
                     panel.grid = ggplot2::element_blank(),
                     panel.border = ggplot2::element_rect(color = "black", fill = NA, size = 0.8),
                     plot.title = ggplot2::element_text(hjust = 0.5))
  }


  # Return
  return(list(
    summary_stats = summary_stats,
    classification_counts = classification_counts,
    contact_data = contact_data,
    pca = pca_result,
    classification_plot = p1,
    scatter_plot = p2,
    pca_plot = p3,
    anova_PC1 = aov_PC1,
    anova_PC2 = aov_PC2,
    tukey_PC1 = tukey_PC1,
    tukey_PC2 = tukey_PC2,
    tukey_plot_PC1 = cld_plot_PC1,
    tukey_plot_PC2 = cld_plot_PC2
  ))
}
