% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tar_tangle.R
\name{tar_tangle}
\alias{tar_tangle}
\title{Convert Quarto or R Markdown to a pipeline}
\usage{
tar_tangle(path)
}
\arguments{
\item{path}{File path to the literate programming source file.
The file can be a Quarto or R Markdown document.}
}
\value{
A list of new target definition objects.
See the "Target definition objects" section for background.
}
\description{
Convert a literate programming source file
into a \code{targets} pipeline.
}
\details{
The word "tangle" comes from the early days of literate
programming (see Knuth 1984).
To "tangle" means to convert a literate programming source document
into pure code: code that a human may find cryptic
but a machine can run.
For example, \code{knitr::knit(tangle = TRUE)} (and \code{\link[knitr:knit]{knitr::purl()}})
accepts an \code{.Rmd} file and returns an R script with all the
R code chunks pasted together.

\code{tar_tangle()} is similar, but for a \code{targets} pipeline.
It accepts a Quarto or R Markdown source file as input,
and it returns a list of target definition objects.
Each target definition object comes from evaluating
\code{\link[targets:tar_target]{targets::tar_target()}} on the each of the assignment statements
in each R code chunk in the file.

For example, consider the following code chunk:

\if{html}{\out{<div class="sourceCode">}}\preformatted{  ```\{r, deployment = "main"\}
  #| pattern: map(data)
  #| format: qs
  #| cue: tar_cue(mode = "always")
  data <- get_data(data)
  ```
}\if{html}{\out{</div>}}

\code{tar_tangle()} converts this chunk into:

\if{html}{\out{<div class="sourceCode">}}\preformatted{  tar_target(
    name = target_name,
    command = command_to_run(data),
    pattern = map(data),
    format = "qs",
    deployment = "main",
    cue = tar_cue(mode = "always")
  )
}\if{html}{\out{</div>}}

To put it all together, suppose our \verb{_targets.R} script
for the pipeline looks like this:

\if{html}{\out{<div class="sourceCode">}}\preformatted{  library(targets)
  tar_source()
  list(
    tar_tangle("example.qmd"),
    tar_target(model, fit_model(data))
  )
}\if{html}{\out{</div>}}

The pipeline above is equivalent to:

\if{html}{\out{<div class="sourceCode">}}\preformatted{  library(targets)
  tar_source()
  list(
    tar_target(
      name = target_name,
      command = command_to_run(data),
      pattern = map(data),
      format = "qs",
      deployment = "main",
      cue = tar_cue(mode = "always")
    ),
    tar_target(model, fit_model(data))
  )
}\if{html}{\out{</div>}}

This pattern is a nice compromise between
interactivity and automation:
you can run the whole pipeline with \code{\link[targets:tar_make]{targets::tar_make()}},
or you can explore individual code chunks in the report
using an IDE like RStudio, Positron, or VSCode.
However, there is a chance other tools like
Quarto or \code{pkgdown} may automatically detect the report and
inappropriately try to run the whole thing from end to end.
To prevent this, you may wish to write
\code{knitr::opts_chunk$set(eval = FALSE)}
in a code chunk at the top of the report.

See the "Examples" section in this help file for a
runnable demonstration with multiple code chunks.

Each code chunk can have more than one top-level assignment statement
(with \verb{<-}, \code{=}, or \verb{->}), and each assignment statement gets
converted into its own target. Non-assignment statements such as
\code{library(dplyr)} are ignored.
It is always good practice to check your pipeline with
\code{\link[targets:tar_manifest]{targets::tar_manifest()}} and \code{\link[targets:tar_visnetwork]{targets::tar_visnetwork()}}
before running it with \code{\link[targets:tar_make]{targets::tar_make()}}.
}
\examples{
if (identical(Sys.getenv("TAR_LONG_EXAMPLES"), "true")) {
targets::tar_dir({  # tar_dir() runs code from a temporary directory.
write.csv(airquality, "data.csv")
lines <- c(
  "---",
  "title: \"Example pipeline\"",
  "---",
  "",
  "```{r}",
  "knitr::opts_chunk$set(eval = FALSE)",
  "```",
  "",
  "```{r}",
  "#| format: file",
  "file <- \"data.csv\"",
  "```",
  "",
  "```{r}",
  "#| memory: persistent",
  "#| packages: [dplyr, readr]",
  "data <- read_csv(file, col_types = cols()) |>",
  "  filter(!is.na(Ozone))",
  "```",
  "",
  "```{r}",
  "#| format: qs",
  "#| cue: tar_cue(mode = \"never\")",
  "model <- lm(Ozone ~ Temp, data) |>",
  "  coefficients()",
  "```",
  "",
  "```{r}",
  "#| deployment: main",
  "#| packages: ggplot2",
  "plot <- ggplot(data) +",
  "  geom_point(aes(x = Temp, y = Ozone)) +",
  "  geom_abline(intercept = model[1], slope = model[2]) +",
  "  theme_gray(24)",
  "```"
)
writeLines(lines, "pipeline.qmd")
targets::tar_script(tarchetypes::tar_tangle("pipeline.qmd"))
targets::tar_make()
targets::tar_read(plot)
})
}
}
\references{
\itemize{
\item Knuth, Donald E. (1984). "Literate Programming".
The Computer Journal. 27 (2). British Computer Society: 97-11.
\url{doi:10.1093/comjnl/27.2.97}.
}
}
\seealso{
Other Domain-specific languages for pipeline construction: 
\code{\link{tar_assign}()}
}
\concept{Domain-specific languages for pipeline construction}
