% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sts-functions.R
\name{sts_fit_with_hmc}
\alias{sts_fit_with_hmc}
\title{Draw posterior samples using Hamiltonian Monte Carlo (HMC)}
\usage{
sts_fit_with_hmc(
  observed_time_series,
  model,
  num_results = 100,
  num_warmup_steps = 50,
  num_leapfrog_steps = 15,
  initial_state = NULL,
  initial_step_size = NULL,
  chain_batch_shape = list(),
  num_variational_steps = 150,
  variational_optimizer = NULL,
  variational_sample_size = 5,
  seed = NULL,
  name = NULL
)
}
\arguments{
\item{observed_time_series}{\code{float} \code{tensor} of shape
\verb{concat([sample_shape, model.batch_shape, [num_timesteps, 1]])} where
\code{sample_shape} corresponds to i.i.d. observations, and the trailing \verb{[1]}
dimension may (optionally) be omitted if \code{num_timesteps > 1}. May
optionally be an instance of \code{sts_masked_time_series}, which includes
a mask \code{tensor} to specify timesteps with missing observations.}

\item{model}{An instance of \code{StructuralTimeSeries} representing a
time-series model. This represents a joint distribution over
time-series and their parameters with batch shape \verb{[b1, ..., bN]}.}

\item{num_results}{Integer number of Markov chain draws. Default value: \code{100}.}

\item{num_warmup_steps}{Integer number of steps to take before starting to
collect results. The warmup steps are also used to adapt the step size
towards a target acceptance rate of 0.75. Default value: \code{50}.}

\item{num_leapfrog_steps}{Integer number of steps to run the leapfrog integrator
for. Total progress per HMC step is roughly proportional to \code{step_size * num_leapfrog_steps}.
Default value: \code{15}.}

\item{initial_state}{Optional Python \code{list} of \code{Tensor}s, one for each model
parameter, representing the initial state(s) of the Markov chain(s). These
should have shape \code{tf$concat(list(chain_batch_shape, param$prior$batch_shape, param$prior$event_shape))}.
If \code{NULL}, the initial state is set automatically using a sample from a variational posterior.
Default value: \code{NULL}.}

\item{initial_step_size}{\code{list} of \code{tensor}s, one for each model parameter,
representing the step size for the leapfrog integrator. Must
broadcast with the shape of \code{initial_state}. Larger step sizes lead to
faster progress, but too-large step sizes make rejection exponentially
more likely. If \code{NULL}, the step size is set automatically using the
standard deviation of a variational posterior. Default value: \code{NULL}.}

\item{chain_batch_shape}{Batch shape (\code{list} or \code{int}) of chains to run in parallel.
Default value: \code{list()} (i.e., a single chain).}

\item{num_variational_steps}{\code{int} number of steps to run the variational
optimization to determine the initial state and step sizes. Default value: \code{150}.}

\item{variational_optimizer}{Optional \code{tf$train$Optimizer} instance to use in
the variational optimization. If \code{NULL}, defaults to \code{tf$train$AdamOptimizer(0.1)}.
Default value: \code{NULL}.}

\item{variational_sample_size}{integer number of Monte Carlo samples to use
in estimating the variational divergence. Larger values may stabilize
the optimization, but at higher cost per step in time and memory.
Default value: \code{1}.}

\item{seed}{integer to seed the random number generator.}

\item{name}{name prefixed to ops created by this function. Default value: \code{NULL} (i.e., 'fit_with_hmc').}
}
\value{
list of:
\itemize{
\item samples: \code{list} of \code{Tensors} representing posterior samples of model
parameters, with shapes \verb{[concat([[num_results], chain_batch_shape, param.prior.batch_shape, param.prior.event_shape]) for param in model.parameters]}.
\item kernel_results: A (possibly nested) \code{list} of \code{Tensor}s representing
internal calculations made within the HMC sampler.
}
}
\description{
Markov chain Monte Carlo (MCMC) methods are considered the gold standard of
Bayesian inference; under suitable conditions and in the limit of infinitely
many draws they generate samples from the true posterior distribution. HMC (Neal, 2011)
uses gradients of the model's log-density function to propose samples,
allowing it to exploit posterior geometry. However, it is computationally more
expensive than variational inference and relatively sensitive to tuning.
}
\details{
This method attempts to provide a sensible default approach for fitting
StructuralTimeSeries models using HMC. It first runs variational inference as
a fast posterior approximation, and initializes the HMC sampler from the
variational posterior, using the posterior standard deviations to set
per-variable step sizes (equivalently, a diagonal mass matrix). During the
warmup phase, it adapts the step size to target an acceptance rate of 0.75,
which is thought to be in the desirable range for optimal mixing (Betancourt et al., 2014).
}
\section{References}{

\itemize{
\item \href{https://arxiv.org/abs/1206.1901}{Radford Neal. MCMC Using Hamiltonian Dynamics. \emph{Handbook of Markov Chain Monte Carlo}, 2011.}
\item \href{https://arxiv.org/abs/1411.6669}{M.J. Betancourt, Simon Byrne, and Mark Girolami. Optimizing The Integrator Step Size for Hamiltonian Monte Carlo.}
}
}

\examples{
\dontrun{
observed_time_series <-
  rep(c(3.5, 4.1, 4.5, 3.9, 2.4, 2.1, 1.2), 5) +
  rep(c(1.1, 1.5, 2.4, 3.1, 4.0), each = 7) \%>\%
  tensorflow::tf$convert_to_tensor(dtype = tensorflow::tf$float64)
day_of_week <- observed_time_series \%>\% sts_seasonal(num_seasons = 7)
local_linear_trend <- observed_time_series \%>\% sts_local_linear_trend()
model <- observed_time_series \%>\%
  sts_sum(components = list(day_of_week, local_linear_trend))
states_and_results <- observed_time_series \%>\%
  sts_fit_with_hmc(
    model,
    num_results = 10,
    num_warmup_steps = 5,
    num_variational_steps = 15)
}

}
\seealso{
Other sts-functions: 
\code{\link{sts_build_factored_surrogate_posterior}()},
\code{\link{sts_build_factored_variational_loss}()},
\code{\link{sts_decompose_by_component}()},
\code{\link{sts_decompose_forecast_by_component}()},
\code{\link{sts_forecast}()},
\code{\link{sts_one_step_predictive}()},
\code{\link{sts_sample_uniform_initial_state}()}
}
\concept{sts-functions}
