
# Function to calculate vegetation indices

vegetation_indices <- function(data){

  # Convert column names to uppercase to handle case insensitivity

  colnames(data) <- toupper(colnames(data))


  # Check if necessary columns exist (in uppercase)

  required_columns <- c("SERIAL_NO","RED", "GREEN", "NIR", "BLUE", "L", "SWIR2")

  if (!all(required_columns %in% colnames(data))) {

    stop("The input data must contain the following column names with valid data:SERIAL_NO, RED, GREEN, NIR, BLUE, L, SWIR2")

  }

  # Check if necessary columns are numeric
  if (!all(sapply(data[c("RED", "GREEN", "NIR", "BLUE", "L", "SWIR2")], is.numeric))) {

    stop("Columns RED, GREEN, NIR, BLUE, L, and SWIR2 must contain numeric data.")
  }



  #### Calculate vegetation indices


  #Normalized Difference Vegetation Index (NDVI)

  data$NDVI <- (data$NIR - data$RED) / (data$NIR + data$RED)

  #Green Normalized Difference Vegetation Index (GNDVI)

  data$GNDVI <- (data$NIR - data$GREEN) / (data$NIR + data$GREEN)

  #Normalized Difference Infrared Index (NDII)

  data$NDII <- (data$NIR - data$SWIR2) / (data$NIR + data$SWIR2)

  #Enhanced Vegetation Index (EVI)

  data$EVI <- (2.5 * (data$NIR - data$RED)) / (data$NIR + (6 * data$RED) - (7.5 * data$BLUE)+1)

  #Soil-Adjusted Vegetation Index (SAVI)

  data$SAVI <- ((data$NIR - data$RED)* (1 + data$L)) / (data$NIR + data$RED + data$L)

  #Difference Vegetation Index (DVI)

  data$DVI <- data$NIR - data$RED

  #Green Difference Vegetation Index (GDVI)

  data$GDVI <- data$NIR - data$GREEN

  #NIR Difference Index (NDI)

  data$NDI <- data$NIR - data$RED

  #Visible Atmospherically Resistant Vegetation Index (VARI)

  data$VARI <- (data$GREEN - data$RED) / (data$GREEN + data$RED - data$BLUE)

  #Corrected Normalized Difference Vegetation Index (CNDVI)

  data$CNDVI <- (data$NIR - data$RED) / (data$NIR + data$RED + 0.2)

  #Optimized Soil-Adjusted Vegetation Index (OSAVI)

  data$OSAVI <- ((data$NIR - data$RED)* (1 + 0.16)) / (data$NIR + data$RED + 0.16)

  #Chlorophyll Absorption Index (CAI)

  data$CAI <- (data$RED - data$GREEN) / (data$RED + data$GREEN)

  #Burn Area Index (BAI)

  data$BAI <- 1 / ((0.1 - data$RED)^2 + (0.06 - data$NIR)^2)

  #Chlorophyll Index - Green (CIg)

  data$CIg <- (data$NIR / data$GREEN) - 1

  #Global Environmental Monitoring Index (GEMI)

  ETA <- (2 * (data$NIR^2 - data$RED^2) + (1.5 * data$NIR) + (0.5 * data$RED)) / (data$NIR + data$RED + 0.5)

  data$GEMI <- (ETA * (1 - 0.25 * ETA)) - ((data$RED - 0.125) / (1 - data$RED))

  #Modified Soil Adjusted Vegetation Index (MSAVI)

  data$MSAVI <- (1/2) * (2 * (data$NIR + 1) - sqrt((2 * data$NIR + 1)^2 - (8 * (data$NIR - data$RED))))

  #Modified Triangular Vegetation Index (MTVI2)

  data$MTVI2 <- (1.5 * ((1.2 * (data$NIR - data$GREEN)) - (2.5 * (data$RED - data$GREEN)))) * (sqrt(((2 * data$NIR) + 1)^2 - ((6 * data$NIR) - (5 * sqrt(data$RED))) - 0.5))

  #Atmospherically Resistant Vegetation Index (ARVI)

  data$ARVI <- (data$NIR - (2 * data$RED) + data$BLUE) / (data$NIR + (2 * data$RED) + data$BLUE)

  #Structure Intensive Pigment Vegetation Index (SIPI)

  data$SIPI <- (data$NIR - data$BLUE) / (data$NIR - data$RED)

  #Normalized Burn Ratio (NBR)

  data$NBR <- (data$NIR - data$SWIR2) / (data$NIR + data$SWIR2)

  #Green Chlorophyll Vegetation Index (GCI)

  data$GCI <-  (data$NIR / data$GREEN) - 1

  #Red-Edge Chlorophyll Vegetation Index (RECI)

  data$RECI <- (data$NIR / data$RED) - 1



  # Return the updated data frame with calculated indices
  return(data)
}

