% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/weightedCox.R
\name{cox_rhogamma}
\alias{cox_rhogamma}
\title{Weighted Cox Model with Rho-Gamma Weights}
\usage{
cox_rhogamma(
  dfcount,
  scheme = "fh",
  scheme_params = list(rho = 0, gamma = 0.5),
  draws = 0,
  alpha = 0.05,
  verbose = FALSE,
  lr.digits = 4
)
}
\arguments{
\item{dfcount}{List; output from \code{\link{df_counting}} containing counting
process data including time, delta, z, w_hat, survP_all, survG_all, etc.}

\item{scheme}{Character; weighting scheme. See \code{\link{wt.rg.S}} for options.
Default: "fh".}

\item{scheme_params}{List; parameters for the selected scheme. Default:
\code{list(rho = 0, gamma = 0.5)}. Required parameters depend on scheme:
\itemize{
\item "fh": \code{rho}, \code{gamma}
\item "MB": \code{mb_tstar}
\item "custom_time": \code{t.tau}, \code{w0.tau}, \code{w1.tau}
}}

\item{draws}{Integer; number of resampling draws for variance estimation and
bias correction. If 0, only asymptotic inference is performed. Default: 0.}

\item{alpha}{Numeric; significance level for confidence intervals. Default: 0.05.}

\item{verbose}{Logical; whether to print detailed output. Default: FALSE.}

\item{lr.digits}{Integer; number of decimal places for formatted output. Default: 4.}
}
\value{
A list containing:
\describe{
\item{fit}{List with fitted model components:
\itemize{
\item \code{bhat}: Estimated log hazard ratio
\item \code{sig_bhat_asy}: Asymptotic standard error
\item \code{u.zero}: Score statistic at beta=0 (log-rank)
\item \code{z.score}: Standardized score statistic
\item \code{sig2_score}: Variance of score statistic
\item \code{wt_rg}: Vector of time-dependent weights
\item \code{bhat_debiased}: Bias-corrected estimate (if draws > 0)
\item \code{sig_bhat_star}: Resampling-based standard error (if draws > 0)
}
}
\item{hr_ci_asy}{Data frame with asymptotic HR and CI}
\item{hr_ci_star}{Data frame with resampling-based HR and CI (if draws > 0)}
\item{cox_text_asy}{Formatted string with HR and asymptotic CI}
\item{cox_text_star}{Formatted string with HR and resampling CI (if draws > 0)}
\item{z.score_debiased}{Bias-corrected score statistic (if draws > 0)}
\item{zlogrank_text}{Formatted log-rank test result}
}
}
\description{
Fits a weighted Cox proportional hazards model using flexible time-dependent
weights (e.g., Fleming-Harrington, Magirr-Burman). Supports resampling-based
inference for variance estimation and bias correction.
}
\details{
This function solves the weighted Cox partial likelihood score equation:
U(beta) = sum_i w_i K_i (dN_0/Y_0 - dN_1/Y_1) = 0

where K_i are time-dependent weights and Y_j, dN_j are risk sets
and event counts for group j.

When \code{draws > 0}, the function performs resampling to:
\itemize{
\item Estimate finite-sample variance (more accurate than asymptotic)
\item Compute bias correction for \eqn{\hat{\beta}}
\item Provide improved confidence intervals for small samples
}

The score test at \eqn{\beta=0} corresponds to the weighted log-rank test.
}
\note{
The treatment variable in \code{dfcount} must be coded as 0=control, 1=experimental.
}
\examples{
# First get counting process data
library(survival)
str(veteran)
veteran$treat <- as.numeric(veteran$trt) - 1

dfcount <- df_counting(
  df = veteran,
  tte.name = "time",
  event.name = "status",
  treat.name = "treat"
)

# Fit weighted Cox model with FH(0,0.5) weights
fit <- cox_rhogamma(
  dfcount = dfcount,
  scheme = "fh",
  scheme_params = list(rho = 0, gamma = 0.5),
  draws = 1000,
  verbose = TRUE
)

print(fit$cox_text_star)  # Resampling-based CI
print(fit$zlogrank_text)  # Weighted log-rank test

# Compare asymptotic and resampling CIs
print(fit$hr_ci_asy)
print(fit$hr_ci_star)

}
\references{
Magirr, D. and Burman, C. F. (2019). Modestly weighted logrank tests.
Statistics in Medicine, 38(20), 3782-3790.
}
\seealso{
\code{\link{df_counting}} for preprocessing
\code{\link{wt.rg.S}} for weighting schemes
\code{\link{cox_score_rhogamma}} for score function

Other survival_analysis: 
\code{\link{KM_diff}()},
\code{\link{df_counting}()},
\code{\link{wt.rg.S}()}

Other weighted_tests: 
\code{\link{df_counting}()},
\code{\link{wt.rg.S}()}
}
\concept{survival_analysis}
\concept{weighted_tests}
