#' Coerce a generic list of models (and subgroups) to a predxhaz_list
#'
#' @param models Either:
#'   (A) a named list of objects coercible to \code{predxhaz} (one subgroup per model), or
#'   (B) a named list of models, where each model is a list of subgroups
#'       (e.g., \code{list(grp0 = <pred>, grp1 = <pred>, ...)}), each coercible to \code{predxhaz}.
#' @param group_labels Optional named character vector mapping subgroup ids (names) to display labels.
#'   Missing names get their id as label.
#' @return An object of class \code{predxhaz_list}.
#'
#' @export
as.predxhaz_list <- function(models, group_labels = NULL) {
  stopifnot(is.list(models), length(models) > 0, !is.null(names(models)))

  .px <- function(x) if (inherits(x, "predxhaz")) x else as.predxhaz(x)

  is_flat <- all(vapply(models, function(m) !is.list(m) || inherits(m, "predxhaz"), logical(1)))

  if (is_flat) {
    items <- lapply(models, function(m) list(subgroup = .px(m)))
    group_ids <- "subgroup"
  } else {
    group_ids <- sort(unique(unlist(lapply(models, names))))
    if (!length(group_ids)) stop("No subgroup names detected in 'models'.")
    items <- lapply(models, function(m) {
      stopifnot(is.list(m))
      out <- list()
      for (g in group_ids) if (!is.null(m[[g]])) out[[g]] <- .px(m[[g]])
      out
    })
  }


  if (is.null(group_labels)) {
    group_labels <- setNames(group_ids, group_ids)
  } else {
    stopifnot(is.character(group_labels), length(names(group_labels)) > 0)
    missing <- setdiff(group_ids, names(group_labels))
    if (length(missing)) group_labels <- c(group_labels, setNames(missing, missing))
    group_labels <- group_labels[group_ids]
  }

  structure(
    list(
      items       = items,
      model_names = names(items),
      group_ids   = group_ids,
      group_labels= group_labels
    ),
    class = "predxhaz_list"
  )
}
