# The anova function for a coxph object
anova.coxph <- function (object, ...,  test = 'Chisq') {
    if (!inherits(object, "coxph"))
        stop ("argument must be a cox model")

    # All the ... args need to be coxph or coxme fits.  If any of them
    #  have a name attached, e.g., 'charlie=T' we assume a priori
    #  that they are illegal
    #
    dotargs <- list(...)
    named <- if (is.null(names(dotargs))) 
	           rep(FALSE, length(dotargs))
             else (names(dotargs) != "")
    if (any(named)) 
        warning(paste("The following arguments to anova.coxph(..)", 
            "are invalid and dropped:", paste(deparse(dotargs[named]), 
                collapse = ", ")))
    dotargs <- dotargs[!named]

    if (length(dotargs) >0) {
        # Check that they are all cox or coxme models
        is.coxmodel <-unlist(lapply(dotargs, function(x) inherits(x, "coxph")))
        is.coxme <- unlist(lapply(dotargs, function(x) inherits(x, "coxme")))
        if (!all(is.coxmodel | is.coxme))
            stop("All arguments must be Cox models")
        
        if (any(is.coxme))
            return(anova.coxmelist(c(list(object), dotargs), test = test))
        else return(anova.coxphlist(c(list(object), dotargs), test = test))
    }

    #
    # The argument is a single Cox model 
    # Show the nested list of models generated by this model.
    # Missings are the real nuisance here, we want all the models to be
    #  evaluated on the same subset of observations.
    # Create a data set that contains all the needed variables, and then collapse
    #  it down using na.omit.  We could use model.frame instead, but has many
    #  wrong variable names, e.g. a variable names "offset(zed") if the variable
    #  zed were an offset.  
    if (length(object$rscore)>0)
        stop("Can't do anova tables with robust variances")

    varlist <- unique(c(all.vars(object$formula), all.vars(object$call$subset),
                        all.vars(object$call$weights)))
    lcall <- parse(text=paste("data.frame(", paste(varlist, collapse=", "), ")"))
    if (is.null(object$call$data))
        newdata <- eval(lcall, environment(object$formula))
    else newdata<- eval(lcall, eval(object$call$data), 
                        environment(object$formula))
    newdata <- na.omit(newdata)

    # get the list of terms to drop, one by one
    termlist<-attr(object$terms,"term.labels") #variables in the formula
    specials <- attr(object$terms, "specials")
    if (!is.null(specials$strata)) {
        # The strata term is not dropped
        termlist <- termlist[-(specials$strata -1)]
        }
    
    # Create variables to hold the output
    nmodel <- length(termlist)
    df <- integer(nmodel+1)  #this will hold the df vector
    loglik <- double(nmodel+1) #and the loglike vector
    df[nmodel+1] <- sum(!is.na(object$coefficients))
    loglik[nmodel+1] <- object$loglik[2]
    df[1] <- 0
    loglik[1] <- object$loglik[1]

    fit <- object
    if (nmodel > 1) for (i in nmodel:2) {
        fit <- update(fit, paste(".~. -", termlist[i]), data=newdata)
        df[i] <- sum(!is.na(coef(fit)))
        loglik[i] <- fit$loglik[2]
        }

    table <- data.frame(loglik=loglik, Chisq=c(NA, 2*diff(loglik)), 
                        Df=c(NA, diff(df))) 

    if (nmodel == 0) #failsafe for a NULL model
             table <- table[1, , drop = FALSE]

    if (length(test) >0 && test[1]=='Chisq') {
        table[['Pr(>|Chi|)']] <- 1- pchisq(table$Chisq, table$Df)
        }
    row.names(table) <- c('NULL', termlist)

    title <- paste("Analysis of Deviance Table\n Cox model: response is ",
		   deparse(object$terms[[2]]),
		   "\nTerms added sequentially (first to last)\n", 
		   sep = "")
    if (is.R())
	    structure(table, heading = title, class = c("anova", "data.frame"))
    else    structure(table, heading= title, class='anova')
}
