#' AsianOption: Asian Option Pricing with Price Impact
#'
#' Implements binomial tree pricing for geometric and arithmetic Asian options
#' incorporating market price impact from hedging activities. Uses the
#' Cox-Ross-Rubinstein (CRR) model with the replicating portfolio method.
#'
#' @section Main Functions:
#' \itemize{
#'   \item \code{\link{price_geometric_asian}}: Exact pricing for geometric Asian calls
#'   \item \code{\link{arithmetic_asian_bounds}}: Bounds for arithmetic Asian calls
#'   \item \code{\link{compute_p_adj}}: Compute adjusted risk-neutral probability
#'   \item \code{\link{compute_adjusted_factors}}: Compute modified up/down factors
#'   \item \code{\link{check_no_arbitrage}}: Validate no-arbitrage condition
#' }
#'
#' @section Price Impact Mechanism:
#' When market makers hedge options, their trading volume causes price movements
#' following the Kyle (1985) linear impact model. This modifies the binomial tree
#' dynamics through adjusted up/down factors and risk-neutral probability.
#'
#' @section Mathematical Framework:
#' The package uses the replicating portfolio approach where the option value
#' equals the cost of constructing a portfolio that replicates its payoff,
#' leading to risk-neutral pricing.
#'
#' For geometric Asian options, the geometric average of stock prices is used.
#' For arithmetic Asian options, rigorous upper and lower bounds are computed
#' using Jensen's inequality. A tighter path-specific bound is available via
#' exact path enumeration.
#'
#' @section No-Arbitrage Condition:
#' For valid pricing, the model requires that the adjusted down factor is less
#' than the risk-free rate, which is less than the adjusted up factor. This
#' ensures a valid risk-neutral probability. All pricing functions validate
#' this condition automatically.
#'
#' For detailed mathematical formulations, see the package vignettes and
#' the reference paper.
#'
#' @section Computational Complexity:
#' The implementation enumerates all \eqn{2^n} possible price paths:
#' \itemize{
#'   \item \eqn{n \leq 15}: Fast (< 1 second)
#'   \item \eqn{n = 20}: ~1 million paths (~10 seconds)
#'   \item \eqn{n > 20}: Warning issued automatically
#' }
#'
#' @references
#' Tiwari, P., & Majumdar, S. (2025). Asian option valuation under price impact.
#' \emph{arXiv preprint}. \doi{10.48550/arXiv.2512.07154}
#'
#' @examples
#' # Price geometric Asian option with price impact
#' price_geometric_asian(
#'   S0 = 100, K = 100, r = 1.05,
#'   u = 1.2, d = 0.8,
#'   lambda = 0.1, v_u = 1, v_d = 1,
#'   n = 3
#' )
#'
#' # Compute bounds for arithmetic Asian option
#' bounds <- arithmetic_asian_bounds(
#'   S0 = 100, K = 100, r = 1.05,
#'   u = 1.2, d = 0.8,
#'   lambda = 0.1, v_u = 1, v_d = 1,
#'   n = 3
#' )
#' print(bounds)
#'
#' # Check no-arbitrage condition
#' check_no_arbitrage(r = 1.05, u = 1.2, d = 0.8,
#'                    lambda = 0.1, v_u = 1, v_d = 1)
#'
#' # Compute adjusted factors
#' factors <- compute_adjusted_factors(u = 1.2, d = 0.8,
#'                                     lambda = 0.1, v_u = 1, v_d = 1)
#' print(factors)
#' @keywords internal
"_PACKAGE"

## usethis namespace: start
#' @useDynLib AsianOption, .registration = TRUE
#' @importFrom Rcpp sourceCpp
#' @importFrom stats pnorm
## usethis namespace: end
NULL
