#' Wavelet-based NARDL Model
#'
#' This function implements the Wavelet-based Nonlinear Autoregressive Distributed Lag (WNARDL) model using wavelet transform.
#'
#' @param ts A time series object (numeric vector) for the dependent variable.
#' @param Filter Wavelet filter to use (default is "haar").
#' @param Wvlevels Number of wavelet decomposition levels. Default is calculated based on the length of `ts`.
#' @param Exo A time series object (numeric vector) for the exogenous variable.
#' @param MaxLag Maximum number of lags to consider. Default is 3.
#' @param Trend Boolean to include trend in the model. Default is TRUE.
#'
#' @return A list containing:
#' \item{Coefficients}{Model coefficients (short and long run).}
#' \item{AsymTest}{Wald test statistics and p-values.}
#' \item{IC}{Information criteria (AIC, BIC, Log-likelihood).}
#'
#' @references
#' Jammazi, R., Lahiani, A., & Nguyen, D. K. (2015). A wavelet-based nonlinear ARDL model for assessing the exchange rate pass-through to crude oil prices. *Journal of International Financial Markets, Institutions and Money, 34*, 173-187. https://doi.org/10.1016/j.intfin.2014.11.011
#'
#' @examples
#' ts <- rnorm(100)
#' Exo <- rnorm(100)
#' Results <- DWNARDL(ts, Filter = "haar", Exo = Exo, MaxLag = 3)
DWNARDL <- function(ts, Filter = "haar", Wvlevels = NULL, Exo, MaxLag = 3, Trend = TRUE) {
  Exo <- as.ts(Exo)
  ts <- (as.matrix(ts))

  if (is.null(Wvlevels)) {
    Wvlevels <- round(log2(length(ts)))
  } else {
    Wvlevels <- Wvlevels
  }

  mraout <- modwt(as.vector(ts), filter = Filter, n.levels = Wvlevels)
  WaveletSeries <- cbind(do.call(cbind, mraout@W), mraout@V[[Wvlevels]])

  Data1 <- as.ts(WaveletSeries[, ncol(WaveletSeries)])
  Data2 <- as.ts(Exo)

  if (Trend == TRUE) {
    t = 5
  } else if (Trend == FALSE) {
    t = 3
  }

  Data_ARDL <- cbind.data.frame(Data1, Data2)
  colnames(Data_ARDL) <- c("y", "x")

  NARDL <- nardl(y ~ x, Data_ARDL, ic = "bic", maxlag = MaxLag, graph = TRUE, case = t)
  Res <- rbind(NARDL$sels$coefficients, NARDL$lres)

  Wald <- rbind.data.frame(NARDL$wldsr, NARDL$wldq)
  colnames(Wald) <- c("Statistic", "pvalue")
  rownames(Wald) <- c("Short Run", "Long Run")

  IC <- cbind.data.frame(AIC = AIC(NARDL$fits), BIC = BIC(NARDL$fits), LogLik = logLik(NARDL$fits)[1])

  Results <- list(Coefficients = Res, AsymTest = Wald, IC = IC)
  return(Results)
}
