% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/residuals.R
\name{residuals.maxlogL}
\alias{residuals.maxlogL}
\title{Extract Residuals from \code{maxlogL} model.}
\usage{
\method{residuals}{maxlogL}(object, parameter = NULL, type = "rqres", routine, ...)
}
\arguments{
\item{object}{an object of \code{\link{maxlogL}} class obtained by fitting a
model with \code{\link{maxlogLreg}}.}

\item{parameter}{a character which specifies residuals for a specific parameter.}

\item{type}{a character with the type of residuals to be computed.
The default value is \code{type = "rqres"}, which is used to
compute the normalized randomized quantile residuals.}

\item{routine}{a character specifying the integration routine.
\code{integrate} and \code{gauss_quad} are available. Custom
routines can be defined but they must be compatible with the
\code{\link{integration}} API.}

\item{...}{further arguments for the integration routine.}
}
\value{
a vector with the specified residuals of a \code{maxlogLreg} model.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}

\code{residuals.maxlogL} is the \code{maxlogLreg} specific method for the
generic function residuals which extracts the residuals from a fitted model.
}
\details{
For \code{type = "deviance"}, the residuals are computed using the following
expression

\deqn{r^D_i = \mbox{sign}(y_i - \hat{\mu}_i) d_i^{1/2},}

where \eqn{d_i} is the residual deviance of each data point. In this context,
\eqn{\hat{\mu}} is the estimated mean, computed as the expected value using
the estimated parameters of a fitted \code{maxlogLreg} model.

On the other hand, for \code{type = "response"} the computation is simpler

\deqn{r^R_i = (y_i - \hat{\mu}_i).}
}
\examples{
library(EstimationTools)

#----------------------------------------------------------------------------
# Example 1: Test deviance residuals
set.seed(123)
n <- 500
x <- runif(n = n, min = 0, max = 1)
y <- rweibull(n = n, shape = 1, scale = exp(4.5 + 0.5*x))
status <- rep(1, n) # sample(0:1, size = n, replace = TRUE)

distribution <- Surv(y, status) ~ dweibull

formulas <- list(
  scale.fo = ~ x
)

fixed <- list(shape = 1)

links <- list(
  over = "scale",
  fun = "log_link"
)

model <- maxlogLreg(
  formulas = formulas,
  y_dist = distribution,
  fixed = fixed,
  link = links
)

# Using `residuals` method
cox_snell_residuals_test <- residuals(model, type = "cox-snell")
martingale_residuals_test <- residuals(model, type = "martingale")
deviance_residuals_test <- residuals(model, type = "right-censored-deviance")

# From scratch
cox_snell_residuals_ref <- -pweibull(
  q = y,
  shape = 1,
  scale = exp(cbind(rep(1, n), x) \%*\% cbind(coef(model))),
  lower.tail = FALSE,
  log.p = TRUE
)
martingale_residuals_ref <- status - cox_snell_residuals_ref
deviance_residuals_ref <- sign(martingale_residuals_ref) * (
  -2 * (martingale_residuals_ref + status*log(status - martingale_residuals_ref))
)^ 0.5


plot(cox_snell_residuals_test, cox_snell_residuals_ref)
plot(martingale_residuals_test, martingale_residuals_ref)
plot(deviance_residuals_test, deviance_residuals_ref)


#----------------------------------------------------------------------------

}
\author{
Jaime Mosquera Gutiérrez, \email{jmosquerag@unal.edu.co}
}
