#' Gives the true win probability, mean ventilator days, and advanced respiratory days for a set of simulation parameters.
#' @param Prob_Death Probability of death within 28 days for control patients.
#' @param P_Vent_28 Probability of being ventilated for 28 days or more in control participants who do not die in 28 days.
#' @param P_Vent_0 Probability of being ventilated for 0 days in control participants who do not die in 28 days.
#' @param CONTROL_VENT_DENSITY_NON_0_28 Nonparametric density estimate of ventilator days between 1 and 27 days for control participants who do not die in 28 days.
#' @param P_ADVANCED_0_NO_VENT Probability of 0 advanced respiratory days in control participants who did not die in 28 days and are never ventilated.
#' @param CONTROL_ADVANCED_DISTRIBUTION_NO_VENT Nonparametric density estimate of advanced respiratory days between 1 and 28 days for control participants who do not die in 28 days and are never ventilated.
#' @param P_ADVANCED_0_VENT Probability of 0 advanced respiratory days in control participants who did not die in 28 days and are ventilated.
#' @param CONTROL_ADVANCED_DISTRIBUTION_VENT Nonparametric density estimate of advanced respiratory days between 1 and 28 days for control participants who do not die in 28 days and are never ventilated.
#' @param Prob_Death_Treatment Probability of death within 28 days for treatment patients.
#' @param P_Vent_28_Treatment Probability of being ventilated for 28 days or more in treatment participants who do not die in 28 days.
#' @param P_Vent_0_Treatment Probability of being ventilated for 0 days in treatment participants who do not die in 28 days.
#' @param CHANGE_MEAN_VENTILATION_TREATMENT Shift in the nonparametric density estimate of ventilator days between 1 and 27 for treatment patients relative to controls in those who did not die in 28 days.
#' @param P_ADVANCED_0_NO_VENT_Treatment Probability of 0 advanced respiratory days in treatment participants who did not die in 28 days and are never ventilated.
#' @param CHANGE_MEAN_ADVANCED_TREATMENT_NO_VENT Shift in the nonparametric density estimate of advanced respiratory days between 1 and 28 for treatment patients relative to controls in those who do not die in 28 days and are never ventilated.
#' @param P_ADVANCED_0_VENT_Treatment Probability of 0 advanced respiratory days in treatment participants who did not die in 28 days and are ventilated.
#' @param CHANGE_MEAN_ADVANCED_TREATMENT_VENT Shift in the nonparametric density estimate of advanced respiratory days between 1 and 28 for treatment patients relative to controls in those who do not die in 28 days and are ventilated.
#' @return A vector containing the true win probability of a scenario. It also returns the implied mean ventilator days in treatment and control participants and the implied mean advanced respiratory days.
#' @importFrom DFBA dfba_mann_whitney
#' @importFrom stats rbinom sample
#' @examples
#' ##Load in the nonparametric densities
#' ##load("Control_Ventilator_Non0_28_Density.rda")
#' ##load("Control_Advanced_Distribution_No_Ventilated.rda")
#' ##load("Control_Advanced_Distribution_Ventilated.rda")
#' library(DFBA)
#' ###Control Parameters
#' Prob_Death=.25
#' P_Vent_0 = .48
#' P_Vent_28 = .05
#' P_ADVANCED_0_NO_VENT = .31
#' P_ADVANCED_0_VENT = .62
#' ###Treatment Parameters
#' Prob_Death_Treatment = Prob_Death-.05
#' P_Vent_28_Treatment = P_Vent_28-.01
#' P_Vent_0_Treatment = P_Vent_0 + .1
#' CHANGE_MEAN_VENTILATION_TREATMENT = -2
#' P_ADVANCED_0_NO_VENT_Treatment = P_ADVANCED_0_NO_VENT + .1
#' CHANGE_MEAN_ADVANCED_TREATMENT_NO_VENT = -2
#' P_ADVANCED_0_VENT_Treatment = P_ADVANCED_0_VENT + .1
#' CHANGE_MEAN_ADVANCED_TREATMENT_VENT = -2
#' Get_True_Win_Probability(Prob_Death,P_Vent_28,P_Vent_0,CONTROL_VENT_DENSITY_NON_0_28,
#' P_ADVANCED_0_NO_VENT,CONTROL_ADVANCED_DISTRIBUTION_NO_VENT,
#' P_ADVANCED_0_VENT,CONTROL_ADVANCED_DISTRIBUTION_VENT,
#' Prob_Death_Treatment,P_Vent_28_Treatment,P_Vent_0_Treatment,CHANGE_MEAN_VENTILATION_TREATMENT,
#' P_ADVANCED_0_NO_VENT_Treatment,CHANGE_MEAN_ADVANCED_TREATMENT_NO_VENT,
#' P_ADVANCED_0_VENT_Treatment,CHANGE_MEAN_ADVANCED_TREATMENT_VENT)
#'@export
Get_True_Win_Probability = function(
    ##Control Parameters
  Prob_Death,
  P_Vent_28,
  P_Vent_0,
  CONTROL_VENT_DENSITY_NON_0_28,
  P_ADVANCED_0_NO_VENT,
  CONTROL_ADVANCED_DISTRIBUTION_NO_VENT,
  P_ADVANCED_0_VENT,
  CONTROL_ADVANCED_DISTRIBUTION_VENT,
  ###Treatment Parameters
  Prob_Death_Treatment,
  P_Vent_28_Treatment,
  P_Vent_0_Treatment,
  CHANGE_MEAN_VENTILATION_TREATMENT,
  P_ADVANCED_0_NO_VENT_Treatment,
  CHANGE_MEAN_ADVANCED_TREATMENT_NO_VENT,
  P_ADVANCED_0_VENT_Treatment,
  CHANGE_MEAN_ADVANCED_TREATMENT_VENT
){

  ###Check that nonparametric densities have been fed in
  DENSITY_NAMES = c("x","y","bw","n",
                    "old.coords","call","data.name",  "has.na")
  if(sum(names(CONTROL_VENT_DENSITY_NON_0_28) %in% DENSITY_NAMES)<length(DENSITY_NAMES) |
     sum(names(CONTROL_ADVANCED_DISTRIBUTION_VENT) %in% DENSITY_NAMES)<length(DENSITY_NAMES) |
     sum(names(CONTROL_ADVANCED_DISTRIBUTION_NO_VENT) %in% DENSITY_NAMES)<length(DENSITY_NAMES)){

    warning("At least one of the nonparametric densities is not a density() object.

These will be replaced with estimated distributions from the SPARC Database")


    if(sum(names(CONTROL_VENT_DENSITY_NON_0_28) %in% DENSITY_NAMES)<length(DENSITY_NAMES)){

      load("Control_Ventilator_Non0_28_Density.rda")

    }

    if(sum(names(CONTROL_ADVANCED_DISTRIBUTION_NO_VENT) %in% DENSITY_NAMES)<length(DENSITY_NAMES)){

      load("Control_Advanced_Distribution_No_Ventilated.rda")
    }


    if(sum(names(CONTROL_ADVANCED_DISTRIBUTION_VENT) %in% DENSITY_NAMES)<length(DENSITY_NAMES)){

      load("Control_Advanced_Distribution_Ventilated.rda")

    }

  }else{

    DEN_ALL = CONTROL_VENT_DENSITY_NON_0_28
    ADVANCED_DISTRIBUTION_NO_VENT = CONTROL_ADVANCED_DISTRIBUTION_NO_VENT
    ADVANCED_DISTRIBUTION_VENT =  CONTROL_ADVANCED_DISTRIBUTION_VENT
    ###GEt the win probability

    N=20000


    DEATH = rep(NA,N)
    TRT = rep(NA,N)
    DAYS_VENTILATION = rep(NA,N)
    DAYS_ADVANCED = rep(NA,N)

    ##Generate TREATMENT
    TRT = rbinom(N,1,.5)



    ###Generate outcomes for control

    DEATH[which(TRT==0)]=rbinom(sum(TRT==0),1,Prob_Death)
    ##VFD and Advanced respiratory support


    ###use the mixture distribution to generate death
    ###We'll fill in things later, lets make a blank vector
    VENT = rep(NA,sum(TRT==0 & DEATH==0))






    for(i in 1:length(VENT)){

      MIXTURE = sample(c(0,"M",28),1,prob=c(P_Vent_0,1-P_Vent_0 - P_Vent_28,P_Vent_28))
      if(MIXTURE==0){
        VENT[i]=0
      }
      if(MIXTURE==28){
        VENT[i]=28
      }
      if(MIXTURE=="M"){
        VENT[i]=round(pmax(pmin(sample(DEN_ALL$x,1,prob=DEN_ALL$y),27),1)) ###make a whole number

      }

    }




    DAYS_VENTILATION[which(TRT==0 & DEATH==0)]=VENT


    ###Generate Advanced Respiratory support from a nonparametric gaussian kernal density, witha  point mass at 0
    for(i in which(TRT==0 & DEATH==0 & DAYS_VENTILATION<28 )){

      if(DAYS_VENTILATION[i]==0){
        ###HAs the mixture distribution from patients with NO ventilation
        BIN = rbinom(1,1,P_ADVANCED_0_NO_VENT)
        if(BIN==1){
          ###0 days
          DAYS_ADVANCED[i]=0
        }else{
          DAYS_ADVANCED[i]=pmax(pmin(ceiling(sample(ADVANCED_DISTRIBUTION_NO_VENT$x,1,prob=ADVANCED_DISTRIBUTION_NO_VENT$y)),28),1)
        }


      }else{
        ###HAs the mixture distribution from patients with SOME ventilation
        BIN = rbinom(1,1,P_ADVANCED_0_VENT)
        if(BIN==1){
          ###0 days
          DAYS_ADVANCED[i]=0
        }else{
          DAYS_ADVANCED[i]= pmin(pmax(1,ceiling(sample(ADVANCED_DISTRIBUTION_VENT$x,1,prob=ADVANCED_DISTRIBUTION_NO_VENT$y))), 28-DAYS_VENTILATION[i])
        }

      }


    }










    ###



    ###Generate outcomes for treatment
    DEATH[which(TRT==1)]=rbinom(sum(TRT==1),1,Prob_Death_Treatment)
    ##VFD and Advanced respiratory support


    VENT = rep(NA,sum(TRT==1 & DEATH==0))




    for(i in 1:length(VENT)){

      MIXTURE = sample(c(0,"M",28),1,prob=c(P_Vent_0_Treatment,1-P_Vent_0_Treatment - P_Vent_28_Treatment,P_Vent_28_Treatment))
      if(MIXTURE==0){
        VENT[i]=0
      }
      if(MIXTURE==28){
        VENT[i]=28
      }
      if(MIXTURE=="M"){
        VENT[i]=round(pmax(pmin(sample(DEN_ALL$x+CHANGE_MEAN_VENTILATION_TREATMENT,1,prob=DEN_ALL$y),27),1)) ###make a whole number

      }

    }




    DAYS_VENTILATION[which(TRT==1 & DEATH==0)]=VENT


    ###Generate Advanced Respiratory Support frommixture nonparametric density
    for(i in which(TRT==1 & DEATH==0 & DAYS_VENTILATION<28 )){

      if(DAYS_VENTILATION[i]==0){
        ###HAs the mixture distribution from patients with NO ventilation
        BIN = rbinom(1,1,P_ADVANCED_0_NO_VENT_Treatment)
        if(BIN==1){
          ###0 days
          DAYS_ADVANCED[i]=0
        }else{
          DAYS_ADVANCED[i]=pmax(pmin(ceiling(sample(ADVANCED_DISTRIBUTION_NO_VENT$x + CHANGE_MEAN_ADVANCED_TREATMENT_NO_VENT,1,prob=ADVANCED_DISTRIBUTION_NO_VENT$y)),28),1)
        }


      }else{
        ###HAs the mixture distribution from patients with SOME ventilation
        BIN = rbinom(1,1,P_ADVANCED_0_VENT_Treatment)
        if(BIN==1){
          ###0 days
          DAYS_ADVANCED[i]=0
        }else{
          DAYS_ADVANCED[i]= pmin(pmax(1,ceiling(sample(ADVANCED_DISTRIBUTION_VENT$x + CHANGE_MEAN_ADVANCED_TREATMENT_VENT,1,prob=ADVANCED_DISTRIBUTION_NO_VENT$y))), 28-DAYS_VENTILATION[i])
        }

      }


    }



    ##Get the composite endpoint






    COMPOSITE_RANKING = rep(NA,length(TRT))
    #1. The most unfavorable outcome is death at any time. Any patient who dies within the first 28 days
    #is given the worst score.
    COMPOSITE_RANKING[which(DEATH==1)]=-1 ##Set to -1 just to avoid any issues with the ranks after this
    #2. The best outcome is 0 days on invasive mechanical ventilation or ECMO and no more than 1 day
    #on advanced respiratory support.
    COMPOSITE_RANKING[which(DAYS_VENTILATION==0 & DAYS_ADVANCED<=1 & is.na(COMPOSITE_RANKING))]=N


    #3. First, subjects are ranked according to the number of days on invasive mechanical ventilation or
    #ECMO.
    #4. Within the categories set in #3, subjects are then ranked according to number of days on
    #advanced respiratory support (nested ranking).

    which_no_best_worst_outcome_composite = which(is.na(COMPOSITE_RANKING))
    VENTILATOR_RANK = rank(28-DAYS_VENTILATION[which_no_best_worst_outcome_composite],ties.method = "average")

    COMPOSITE_RANKING_NOT_BEST_WORST = VENTILATOR_RANK ###Storing this
    ##Break ventilator rank ties
    UN_VENTILATOR_RANK = unique(VENTILATOR_RANK)
    for(i in 1:length(UN_VENTILATOR_RANK)){
      which1 = which(VENTILATOR_RANK==UN_VENTILATOR_RANK[i])
      if(length(which1)>1){
        ###these are all tied, can we re-rank based on Advanced support?
        TIE_BREAK_DAYS_ADVANCED =   sort(unique(DAYS_ADVANCED[which_no_best_worst_outcome_composite[which1]]))

        GRID =  .001*(1:length(TIE_BREAK_DAYS_ADVANCED))/length(TIE_BREAK_DAYS_ADVANCED)

        ###Creates a grid for tie-breaks
        ##We will subtract each value of the grid
        ##in order of days on advanced respiratory support
        ##Goal is to get the composite ranking order

        for(j in 1:length(TIE_BREAK_DAYS_ADVANCED)){
          ###Look through the order of unique tie break days in order, subtract the corresponding grid value from the ventilator score
          which2 = which(DAYS_ADVANCED[which_no_best_worst_outcome_composite[which1]] == TIE_BREAK_DAYS_ADVANCED[j])

          ###All these patients had this # of days on advanced support
          VENTILATOR_RANK[which1[which2]]=VENTILATOR_RANK[which1[which2]]-GRID[j]
        }

      }
    }




    COMPOSITE_RANKING[which_no_best_worst_outcome_composite]=VENTILATOR_RANK


    ###Now we can re-rank again.
    COMPOSITE_RANKING = rank(COMPOSITE_RANKING,ties.method = "min")


    MEAN_VENT_DAYS_TREATMENT=round(mean(VENT[TRT==1],na.rm=TRUE),1)
    MEAN_VENT_DAYS_CONTROL=round(mean(VENT[TRT==0],na.rm=TRUE),1)
    MEAN_ADV_DAYS_TREATMENT = round(mean(DAYS_VENTILATION[TRT==1],na.rm=TRUE),1)
    MEAN_ADV_DAYS_CONTROL = round(mean(DAYS_VENTILATION[TRT==0],na.rm=TRUE),1)


    m1 =   dfba_mann_whitney(COMPOSITE_RANKING[TRT==1], COMPOSITE_RANKING[TRT==0],hide_progress = FALSE)


    WIN_PROBABILITY=round(m1$omegabar,3)*100

    RESULTS = c(WIN_PROBABILITY,MEAN_VENT_DAYS_TREATMENT,MEAN_VENT_DAYS_CONTROL,MEAN_ADV_DAYS_CONTROL,MEAN_ADV_DAYS_TREATMENT)
    names(RESULTS)=c("Win Probability",
                     "Treatment Mean Ventilator Days",
                     "Control Mean Ventilator Days",
                     "Treatment Mean Advanced Days",
                     "Control Mean Advanced Days")

    return(RESULTS)

  }



}
