#' Provides operating characteristics for a group sequential design using the I-SPY ARDS hierarchical composite ranked endpoint and gating strategy with gating strategy and Bayesian Wilcoxon Rank-Sum test.
#' @param CONTROL_TRUTH List containing control simulation parameters for (1) Mortality, (2) Ventilator day distributions in survivors, and (3) advanced respiratory days in survivors.
#' @param CONTROL_VENT_DENSITY_NON_0_28 Nonparametric density estimate of ventilator days between 1 and 27 days for control participants who do not die in 28 days.
#' @param CONTROL_ADVANCED_DISTRIBUTION_NO_VENT Nonparametric density estimate of advanced respiratory days between 1 and 28 days for control participants who do not die in 28 days and are never ventilated.
#' @param CONTROL_ADVANCED_DISTRIBUTION_VENT Nonparametric density estimate of advanced respiratory days between 1 and 28 days for control participants who do not die in 28 days and are never ventilated.
#' @param TREATMENT_TRUTH List containing treatment simulation parameters for (1) Mortality, (2) Ventilator day distributions in survivors, and (3) advanced respiratory days in survivors.
#' @param N_LOOK Sample size at interim looks
#' @param cut_Wilcox Superiority cutoffs for Bayesian Wilcoxon.
#' @param cut_gating Superiority cutoffs for Frequentist 1-sided Fisher test.
#' @param SAFETY_BOUND Safety threshold for a mortality increase in treatment participants.
#' @param Futility_BOUND Bounds for stopping for futility based on Bayesian Wilcoxon.
#' @param cut_AR Windows for AR truncation. Randomization probabilities are truncated between .5-cut_AR and .5+cut_AR.
#' @param Num_Sims Number of Simulations to run.
#' @return A list containing the true simulation scenario, the design parameters, and the operating characteristics of the design. Operating characteristics include the graduation probability (i.e. power), the sample size distribution, and the joint distribution of graduation and sample size.
#' @importFrom DFBA dfba_mann_whitney
#' @importFrom stats fisher.test rbinom sd wilcox.test
#' @examples
#' ####Set up simulation scenario example
#' ##These are example nonparametric densities for ventilator days
#' ##and advanced respiratory day distributions conditional on ventilation
#' ## load("CONTROL_ADVANCED_DISTRIBUTION_NO_VENT.rda")
#' ## load("CONTROL_ADVANCED_DISTRIBUTION_VENT.rda")
#' ## load("CONTROL_VENT_DENSITY_NON_0_28.rda")
#' library(DFBA)
#' ###Additional simulation parameters in control arm
#' Prob_Death=.25
#' P_Vent_0 = .48
#' P_Vent_28 = .05
#' P_ADVANCED_0_NO_VENT = .31
#' P_ADVANCED_0_VENT = .62
#' ###Treatment Parameters
#' Prob_Death_Treatment = Prob_Death-.05
#' P_Vent_28_Treatment = P_Vent_28-.01
#' P_Vent_0_Treatment = P_Vent_0 + .1
#' CHANGE_MEAN_VENTILATION_TREATMENT = -2
#' P_ADVANCED_0_NO_VENT_Treatment = P_ADVANCED_0_NO_VENT + .1
#' CHANGE_MEAN_ADVANCED_TREATMENT_NO_VENT = -2
#' P_ADVANCED_0_VENT_Treatment = P_ADVANCED_0_VENT + .1
#' CHANGE_MEAN_ADVANCED_TREATMENT_VENT = -2
#' ###Put all the parameters in a list
#' CONTROL_TRUTH = as.list(rep(NA,8))
#' names(CONTROL_TRUTH)=c("Probability of Death","Prob Vent at day 28 in survivors",
#' "Prob Vent for 0 days in survivors",
#' "Nonparametric density estimate of day 1-27 ventilation day in survivors",
#' "Prob of 0 day advanced respiratory support in
#'   non-ventilated surviving participants",
#' "Nonparametric density estimate of day 1-28 advanced respiratory
#'  days in non-ventilated surviving participants",
#' "Prob of 0 day advanced respiratory support in
#'  ventilated surviving participants",
#' "Nonparametric density estimate of day 1-28 advanced
#' respiratory days in ventilated surviving participants")
#' CONTROL_TRUTH[[1]] = Prob_Death
#' CONTROL_TRUTH[[2]] = P_Vent_28
#' CONTROL_TRUTH[[3]] = P_Vent_0
#' CONTROL_TRUTH[[4]] = CONTROL_VENT_DENSITY_NON_0_28
#' CONTROL_TRUTH[[5]] = P_ADVANCED_0_NO_VENT
#' CONTROL_TRUTH[[6]] = CONTROL_ADVANCED_DISTRIBUTION_NO_VENT
#' CONTROL_TRUTH[[7]] = P_ADVANCED_0_VENT
#' CONTROL_TRUTH[[8]] = CONTROL_ADVANCED_DISTRIBUTION_VENT
#' ###Put treatment simulation truth in a matrix
#' TREATMENT_TRUTH = as.list(rep(NA,8))
#' names(TREATMENT_TRUTH)=c("Probability of Death",
#' "Prob Vent at day 28 in survivors",
#' "Prob Vent for 0 days in survivors",
#' "Shift in Nonparametric density estimate of
#'  day 1-27 ventilation day in survivors",
#' "Prob of 0 day advanced respiratory support
#'  in non-ventilated surviving participants",
#' "Shift in Nonparametric density estimate of day 1-28
#' advanced respiratory days in non-ventilated surviving participants",
#' "Prob of 0 day advanced respiratory support in
#'  ventilated surviving participants",
#' "Shift in Nonparametric density estimate of day 1-28
#'  advanced respiratory days in ventilated surviving participants")
#' TREATMENT_TRUTH[[1]] = Prob_Death_Treatment
#' TREATMENT_TRUTH[[2]] = P_Vent_28_Treatment
#' TREATMENT_TRUTH[[3]] = P_Vent_0_Treatment
#' TREATMENT_TRUTH[[4]] = CHANGE_MEAN_VENTILATION_TREATMENT
#' TREATMENT_TRUTH[[5]] = P_ADVANCED_0_NO_VENT_Treatment
#' TREATMENT_TRUTH[[6]] = CHANGE_MEAN_ADVANCED_TREATMENT_NO_VENT
#' TREATMENT_TRUTH[[7]] = P_ADVANCED_0_VENT_Treatment
#' TREATMENT_TRUTH[[8]] = CHANGE_MEAN_ADVANCED_TREATMENT_VENT
#'
#' ###Setting up decision boundaries
#' #If NA, this means no interim decision at that look
#' N_LOOK = c(200,400,600,800)     ##Sample size at interim looks
#' cut_Wilcox = c(NA,.98,.95,.9)   ##Superiority cutoffs for Bayesian Wilcoxon
#' cut_gating = c(NA,.02,.02,.05)  ##Superiority cutoffs for Frequentist 1-sided Fisher test
#' SAFETY_BOUND = c(.1,NA,NA,NA)   ##Safety threshold for a mortality increase
#' Futility_BOUND = c(.5,.6,.7,NA) ##Bounds for stopping for futility based on Bayesian Wilcoxon
#' cut_AR = c(.2,.2,.2,.2)         ##Windows for AR truncation
#' Sim_ARDS_Win_Probability(CONTROL_TRUTH,CONTROL_VENT_DENSITY_NON_0_28,
#' CONTROL_ADVANCED_DISTRIBUTION_NO_VENT,
#' CONTROL_ADVANCED_DISTRIBUTION_VENT,
#'  TREATMENT_TRUTH,N_LOOK,
#' cut_Wilcox,cut_gating,SAFETY_BOUND, Futility_BOUND,cut_AR,  50)
#'@export
Sim_ARDS_Win_Probability =  function(CONTROL_TRUTH, ##List containing control simulation parameters for (1) Mortality, (2) Ventilator day distributions in survivors, and (3) advanced respiratory days in survivors.
                                     CONTROL_VENT_DENSITY_NON_0_28,
                                     CONTROL_ADVANCED_DISTRIBUTION_NO_VENT,
                                     CONTROL_ADVANCED_DISTRIBUTION_VENT,
                                     TREATMENT_TRUTH, ##List containing treatment simulation parameters for (1) Mortality, (2) Ventilator day distributions in survivors, and (3) advanced respiratory days in survivors.
                                       N_LOOK, ##Sample size at interim looks
                                       cut_Wilcox, ##Superiority cutoffs for Bayesian Wilcoxon
                                       cut_gating, ##Superiority cutoffs for Frequentist 1-sided Fisher test
                                       SAFETY_BOUND, ##Safety threshold for a mortality increase in treatment participants
                                       Futility_BOUND, ##Bounds for stopping for futility based on Bayesian Wilcoxon
                                       cut_AR, ##Windows for AR truncation
                                        Num_Sims){ ##Number of simulation replications



    ###Pull out parameters
    Prob_Death = CONTROL_TRUTH[[1]]
    P_Vent_28 = CONTROL_TRUTH[[2]]
    P_Vent_0 = CONTROL_TRUTH[[3]]
    DEN_ALL = CONTROL_TRUTH[[4]]
    P_ADVANCED_0_NO_VENT =  CONTROL_TRUTH[[5]]
    ADVANCED_DISTRIBUTION_NO_VENT = CONTROL_TRUTH[[6]]
    P_ADVANCED_0_VENT = CONTROL_TRUTH[[7]]
    ADVANCED_DISTRIBUTION_VENT =  CONTROL_TRUTH[[8]]




    Prob_Death_Treatment =  TREATMENT_TRUTH[[1]]
    P_Vent_28_Treatment = TREATMENT_TRUTH[[2]]
    P_Vent_0_Treatment = TREATMENT_TRUTH[[3]]
    CHANGE_MEAN_VENTILATION_TREATMENT =  TREATMENT_TRUTH[[4]]
    P_ADVANCED_0_NO_VENT_Treatment = TREATMENT_TRUTH[[5]]
    CHANGE_MEAN_ADVANCED_TREATMENT_NO_VENT = TREATMENT_TRUTH[[6]]
      P_ADVANCED_0_VENT_Treatment = TREATMENT_TRUTH[[7]]
    CHANGE_MEAN_ADVANCED_TREATMENT_VENT = TREATMENT_TRUTH[[8]]



      ###First get the true win probability
  VEC =    Get_True_Win_Probability(Prob_Death,
        P_Vent_28,
        P_Vent_0,
        CONTROL_VENT_DENSITY_NON_0_28,
        P_ADVANCED_0_NO_VENT,
        CONTROL_ADVANCED_DISTRIBUTION_NO_VENT,
        P_ADVANCED_0_VENT,
        CONTROL_ADVANCED_DISTRIBUTION_VENT,
        ###Treatment Parameters
        Prob_Death_Treatment,
        P_Vent_28_Treatment,
        P_Vent_0_Treatment,
        CHANGE_MEAN_VENTILATION_TREATMENT,
        P_ADVANCED_0_NO_VENT_Treatment,
        CHANGE_MEAN_ADVANCED_TREATMENT_NO_VENT,
        P_ADVANCED_0_VENT_Treatment,
        CHANGE_MEAN_ADVANCED_TREATMENT_VENT)



  B = Num_Sims
  POWER = rep(0,B)
  SAMPLE_SIZE = rep(max(N_LOOK),B) #start with the maximum sample sizes
  STORE_P = matrix(nrow=B,ncol=length(N_LOOK))
  colnames(STORE_P)=N_LOOK

  for(b in 1:B){
    rand_prob=.5

    look_count=1
    N=N_LOOK[1]
    DEATH = rep(NA,N)
    TRT = rep(NA,N)
    DAYS_VENTILATION = rep(NA,N)
    DAYS_ADVANCED = rep(NA,N)

    ##Generate TREATMENT
    TRT = rbinom(N,1,.5)



    ###Generate outcomes for control

    DEATH[which(TRT==0)]=rbinom(sum(TRT==0),1,Prob_Death)
    ##VFD and Advanced respiratory support


    ###use the mixture distribution to generate death
    ###We'll fill in things later, lets make a blank vector
    VENT = rep(NA,sum(TRT==0 & DEATH==0))


    for(i in 1:length(VENT)){

      MIXTURE = sample(c(0,"M",28),1,prob=c(P_Vent_0,1-P_Vent_0 - P_Vent_28,P_Vent_28))
      if(MIXTURE==0){
        VENT[i]=0
      }
      if(MIXTURE==28){
        VENT[i]=28
      }
      if(MIXTURE=="M"){
        VENT[i]=round(pmax(pmin(sample(DEN_ALL$x,1,prob=DEN_ALL$y),27),1)) ###make a whole number

      }

    }




    DAYS_VENTILATION[which(TRT==0 & DEATH==0)]=VENT


    ###Generate Advanced Respiratory support from a nonparametric gaussian kernal density, witha  point mass at 0
    for(i in which(TRT==0 & DEATH==0 & DAYS_VENTILATION<28 )){

      if(DAYS_VENTILATION[i]==0){
        ###HAs the mixture distribution from patients with NO ventilation
        BIN = rbinom(1,1,P_ADVANCED_0_NO_VENT)
        if(BIN==1){
          ###0 days
          DAYS_ADVANCED[i]=0
        }else{
          DAYS_ADVANCED[i]=pmax(pmin(ceiling(sample(ADVANCED_DISTRIBUTION_NO_VENT$x,1,prob=ADVANCED_DISTRIBUTION_NO_VENT$y)),28),1)
        }


      }else{
        ###HAs the mixture distribution from patients with SOME ventilation
        BIN = rbinom(1,1,P_ADVANCED_0_VENT)
        if(BIN==1){
          ###0 days
          DAYS_ADVANCED[i]=0
        }else{
          DAYS_ADVANCED[i]= pmin(pmax(1,ceiling(sample(ADVANCED_DISTRIBUTION_VENT$x,1,prob=ADVANCED_DISTRIBUTION_NO_VENT$y))), 28-DAYS_VENTILATION[i])
        }

      }


    }










    ###



    ###Generate outcomes for treatment
    DEATH[which(TRT==1)]=rbinom(sum(TRT==1),1,Prob_Death_Treatment)
    ##VFD and Advanced respiratory support


    VENT = rep(NA,sum(TRT==1 & DEATH==0))




    for(i in 1:length(VENT)){

      MIXTURE = sample(c(0,"M",28),1,prob=c(P_Vent_0_Treatment,1-P_Vent_0_Treatment - P_Vent_28_Treatment,P_Vent_28_Treatment))
      if(MIXTURE==0){
        VENT[i]=0
      }
      if(MIXTURE==28){
        VENT[i]=28
      }
      if(MIXTURE=="M"){
        VENT[i]=round(pmax(pmin(sample(DEN_ALL$x+CHANGE_MEAN_VENTILATION_TREATMENT,1,prob=DEN_ALL$y),27),1)) ###make a whole number

      }

    }




    DAYS_VENTILATION[which(TRT==1 & DEATH==0)]=VENT


    ###Generate Advanced Respiratory Support frommixture nonparametric density
    for(i in which(TRT==1 & DEATH==0 & DAYS_VENTILATION<28 )){

      if(DAYS_VENTILATION[i]==0){
        ###HAs the mixture distribution from patients with NO ventilation
        BIN = rbinom(1,1,P_ADVANCED_0_NO_VENT_Treatment)
        if(BIN==1){
          ###0 days
          DAYS_ADVANCED[i]=0
        }else{
          DAYS_ADVANCED[i]=pmax(pmin(ceiling(sample(ADVANCED_DISTRIBUTION_NO_VENT$x + CHANGE_MEAN_ADVANCED_TREATMENT_NO_VENT,1,prob=ADVANCED_DISTRIBUTION_NO_VENT$y)),28),1)
        }


      }else{
        ###HAs the mixture distribution from patients with SOME ventilation
        BIN = rbinom(1,1,P_ADVANCED_0_VENT_Treatment)
        if(BIN==1){
          ###0 days
          DAYS_ADVANCED[i]=0
        }else{
          DAYS_ADVANCED[i]= pmin(pmax(1,ceiling(sample(ADVANCED_DISTRIBUTION_VENT$x + CHANGE_MEAN_ADVANCED_TREATMENT_VENT,1,prob=ADVANCED_DISTRIBUTION_NO_VENT$y))), 28-DAYS_VENTILATION[i])
        }

      }


    }







    SAFETY_STOP=0
    FUTILE=0
    POWER[b]=0
    ###Is there a safety look?
    if(!is.na(SAFETY_BOUND[look_count])){
      ###Check for a safety stop
      ###This code does not allow checking in groups
      SAFETY_STOP = SAFETY_STOP + (mean(DEATH[TRT==1])> (mean(DEATH[TRT==0])+SAFETY_BOUND[look_count]))
    }


    ##Get the composite endpoint if there is
    ##A futility or safety look prescribed here
    if(!is.na(cut_Wilcox[look_count]) |
       !is.na(cut_gating[look_count])|
       !is.na(SAFETY_BOUND[look_count]) |
       cut_AR[look_count]>0){

      ###Asking for one of the interim looks
      ###
      COMPOSITE_RANKING = rep(NA,length(TRT))
      #1. The most unfavorable outcome is death at any time. Any patient who dies within the first 28 days
      #is given the worst score.
      COMPOSITE_RANKING[which(DEATH==1)]=-1 ##Set to -1 just to avoid any issues with the ranks after this
      #2. The best outcome is 0 days on invasive mechanical ventilation or ECMO and no more than 1 day
      #on advanced respiratory support.
      COMPOSITE_RANKING[which(DAYS_VENTILATION==0 & DAYS_ADVANCED<=1 & is.na(COMPOSITE_RANKING))]=length(TRT)


      #3. First, subjects are ranked according to the number of days on invasive mechanical ventilation or
      #ECMO.
      #4. Within the categories set in #3, subjects are then ranked according to number of days on
      #advanced respiratory support (nested ranking).

      which_no_best_worst_outcome_composite = which(is.na(COMPOSITE_RANKING))
      VENTILATOR_RANK = rank(28-DAYS_VENTILATION[which_no_best_worst_outcome_composite],ties.method = "average")

      COMPOSITE_RANKING_NOT_BEST_WORST = VENTILATOR_RANK ###Storing this
      ##Break ventilator rank ties
      UN_VENTILATOR_RANK = unique(VENTILATOR_RANK)
      for(i in 1:length(UN_VENTILATOR_RANK)){
        which1 = which(VENTILATOR_RANK==UN_VENTILATOR_RANK[i])
        if(length(which1)>1){
          ###these are all tied, can we re-rank based on Advanced support?
          TIE_BREAK_DAYS_ADVANCED =   sort(unique(DAYS_ADVANCED[which_no_best_worst_outcome_composite[which1]]))

          GRID =  .001*(1:length(TIE_BREAK_DAYS_ADVANCED))/length(TIE_BREAK_DAYS_ADVANCED)

          ###Creates a grid for tie-breaks
          ##We will subtract each value of the grid
          ##in order of days on advanced respiratory support
          ##Goal is to get the composite ranking order

          for(j in 1:length(TIE_BREAK_DAYS_ADVANCED)){
            ###Look through the order of unique tie break days in order, subtract the corresponding grid value from the ventilator score
            which2 = which(DAYS_ADVANCED[which_no_best_worst_outcome_composite[which1]] == TIE_BREAK_DAYS_ADVANCED[j])

            ###All these patients had this # of days on advanced support
            VENTILATOR_RANK[which1[which2]]=VENTILATOR_RANK[which1[which2]]-GRID[j]
          }

        }
      }




      COMPOSITE_RANKING[which_no_best_worst_outcome_composite]=VENTILATOR_RANK


      ###Now we can re-rank again.
      COMPOSITE_RANKING = rank(COMPOSITE_RANKING,ties.method = "min")

      m1 =   dfba_mann_whitney(COMPOSITE_RANKING[TRT==1], COMPOSITE_RANKING[TRT==0],hide_progress = FALSE)
      STORE_P[b,look_count]=m1$prH1
      ###Do we have a superiority look for the Bayesian Wilcox?
      if(!is.na(cut_Wilcox[look_count])){
        POWER[b]=m1$prH1>cut_Wilcox[look_count]
      }

      ###Do we have a superiority look for mortality?
      if(!is.na(cut_gating[look_count])){
        X1 = table(TRT,DEATH)   ###Gating based on decreased death rates
        if(POWER[b]==0){
          POWER[b] = fisher.test(X1,alternative="less")$p.value< cut_gating[look_count]
        }
      }


      ###Do we have a futility look?
      if(!is.na(Futility_BOUND[look_count])){
        FUTILE=m1$prH1<Futility_BOUND[look_count]
      }

      rand_prob = max(min(m1$prH1,.5+cut_AR[look_count]),.5-cut_AR[look_count])

      if(is.na(POWER[b])){
        POWER[b]=0
      }

    }


    if(SAFETY_STOP==0 & FUTILE==0 & POWER[b]==0){
      ###Now we move to the next group sequential look


      look_count=2


      while((POWER[b]==0 & FUTILE <1 & SAFETY_STOP<1) & look_count<=length(N_LOOK)){

        ###Placeholders for this
        TRT_HOLD = TRT
        DEATH_HOLD = DEATH
        DAYS_VENTILATION_HOLD = DAYS_VENTILATION
        DAYS_ADVANCED_HOLD = DAYS_ADVANCED

        ###Generate the new data
        N=N_LOOK[look_count]-N_LOOK[look_count-1]


        ###Generate the new data
        DEATH = rep(NA,N)
        TRT = rep(NA,N)
        DAYS_VENTILATION = rep(NA,N)
        DAYS_ADVANCED = rep(NA,N)

        ##Generate TREATMENT
        TRT = rbinom(N,1,rand_prob)



        ###Generate outcomes for control and treatment
        ###Generate outcomes for control

        DEATH[which(TRT==0)]=rbinom(sum(TRT==0),1,Prob_Death)
        ##VFD and Advanced respiratory support


        ###use the mixture distribution to generate death
        ###We'll fill in things later, lets make a blank vector
        VENT = rep(NA,sum(TRT==0 & DEATH==0))






        for(i in 1:length(VENT)){

          MIXTURE = sample(c(0,"M",28),1,prob=c(P_Vent_0,1-P_Vent_0 - P_Vent_28,P_Vent_28))
          if(MIXTURE==0){
            VENT[i]=0
          }
          if(MIXTURE==28){
            VENT[i]=28
          }
          if(MIXTURE=="M"){
            VENT[i]=round(pmax(pmin(sample(DEN_ALL$x,1,prob=DEN_ALL$y),27),1)) ###make a whole number

          }

        }




        DAYS_VENTILATION[which(TRT==0 & DEATH==0)]=VENT




        ###Generate Advanced Respiratory support from a nonparametric gaussian kernal density, witha  point mass at 0
        for(i in which(TRT==0 & DEATH==0 & DAYS_VENTILATION<28 )){

          if(DAYS_VENTILATION[i]==0){
            ###HAs the mixture distribution from patients with NO ventilation
            BIN = rbinom(1,1,P_ADVANCED_0_NO_VENT)
            if(BIN==1){
              ###0 days
              DAYS_ADVANCED[i]=0
            }else{
              DAYS_ADVANCED[i]=pmax(pmin(ceiling(sample(ADVANCED_DISTRIBUTION_NO_VENT$x,1,prob=ADVANCED_DISTRIBUTION_NO_VENT$y)),28),1)
            }


          }else{
            ###HAs the mixture distribution from patients with SOME ventilation
            BIN = rbinom(1,1,P_ADVANCED_0_VENT)
            if(BIN==1){
              ###0 days
              DAYS_ADVANCED[i]=0
            }else{
              DAYS_ADVANCED[i]= pmin(pmax(1,ceiling(sample(ADVANCED_DISTRIBUTION_VENT$x,1,prob=ADVANCED_DISTRIBUTION_NO_VENT$y))), 28-DAYS_VENTILATION[i])
            }

          }


        }





        ###Generate outcomes for treatment
        DEATH[which(TRT==1)]=rbinom(sum(TRT==1),1,Prob_Death_Treatment)
        ##VFD and Advanced respiratory support


        VENT = rep(NA,sum(TRT==1 & DEATH==0))




        for(i in 1:length(VENT)){

          MIXTURE = sample(c(0,"M",28),1,prob=c(P_Vent_0_Treatment,1-P_Vent_0_Treatment - P_Vent_28_Treatment,P_Vent_28_Treatment))
          if(MIXTURE==0){
            VENT[i]=0
          }
          if(MIXTURE==28){
            VENT[i]=28
          }
          if(MIXTURE=="M"){
            VENT[i]=round(pmax(pmin(sample(DEN_ALL$x+CHANGE_MEAN_VENTILATION_TREATMENT,1,prob=DEN_ALL$y),27),1)) ###make a whole number

          }

        }




        DAYS_VENTILATION[which(TRT==1 & DEATH==0)]=VENT



        ###Generate Advanced Respiratory Support frommixture nonparametric density
        for(i in which(TRT==1 & DEATH==0 & DAYS_VENTILATION<28 )){

          if(DAYS_VENTILATION[i]==0){
            ###HAs the mixture distribution from patients with NO ventilation
            BIN = rbinom(1,1,P_ADVANCED_0_NO_VENT_Treatment)
            if(BIN==1){
              ###0 days
              DAYS_ADVANCED[i]=0
            }else{
              DAYS_ADVANCED[i]=pmax(pmin(ceiling(sample(ADVANCED_DISTRIBUTION_NO_VENT$x + CHANGE_MEAN_ADVANCED_TREATMENT_NO_VENT,1,prob=ADVANCED_DISTRIBUTION_NO_VENT$y)),28),1)
            }


          }else{
            ###HAs the mixture distribution from patients with SOME ventilation
            BIN = rbinom(1,1,P_ADVANCED_0_VENT_Treatment)
            if(BIN==1){
              ###0 days
              DAYS_ADVANCED[i]=0
            }else{
              DAYS_ADVANCED[i]= pmin(pmax(1,ceiling(sample(ADVANCED_DISTRIBUTION_VENT$x + CHANGE_MEAN_ADVANCED_TREATMENT_VENT,1,prob=ADVANCED_DISTRIBUTION_NO_VENT$y))), 28-DAYS_VENTILATION[i])
            }

          }


        }









        ##Get the composite endpoint, first combine the old data and the new data
        TRT = c(TRT_HOLD,TRT)
        DAYS_ADVANCED = c(DAYS_ADVANCED_HOLD,DAYS_ADVANCED)
        DAYS_VENTILATION = c(DAYS_VENTILATION_HOLD,DAYS_VENTILATION)
        DEATH = c(DEATH_HOLD,DEATH)

        ###Is there a safety look?
        if(!is.na(SAFETY_BOUND[look_count])){
          ###Check for a safety stop
          ###This code does not allow checking in groups
          SAFETY_STOP = SAFETY_STOP + (mean(DEATH[TRT==1])> (mean(DEATH[TRT==0])+SAFETY_BOUND[look_count]))
        }


        ##Get the composite endpoint if there is
        ##A futility or safety look prescribed here
        if(!is.na(cut_Wilcox[look_count]) |
           !is.na(cut_gating[look_count])|
           !is.na(SAFETY_BOUND[look_count])){

          ###Asking for one of the interim looks
          ###
          COMPOSITE_RANKING = rep(NA,length(TRT))
          #1. The most unfavorable outcome is death at any time. Any patient who dies within the first 28 days
          #is given the worst score.
          COMPOSITE_RANKING[which(DEATH==1)]=-1 ##Set to -1 just to avoid any issues with the ranks after this
          #2. The best outcome is 0 days on invasive mechanical ventilation or ECMO and no more than 1 day
          #on advanced respiratory support.
          COMPOSITE_RANKING[which(DAYS_VENTILATION==0 & DAYS_ADVANCED<=1 & is.na(COMPOSITE_RANKING))]=length(TRT)


          #3. First, subjects are ranked according to the number of days on invasive mechanical ventilation or
          #ECMO.
          #4. Within the categories set in #3, subjects are then ranked according to number of days on
          #advanced respiratory support (nested ranking).

          which_no_best_worst_outcome_composite = which(is.na(COMPOSITE_RANKING))
          VENTILATOR_RANK = rank(28-DAYS_VENTILATION[which_no_best_worst_outcome_composite],ties.method = "average")

          COMPOSITE_RANKING_NOT_BEST_WORST = VENTILATOR_RANK ###Storing this
          ##Break ventilator rank ties
          UN_VENTILATOR_RANK = unique(VENTILATOR_RANK)
          for(i in 1:length(UN_VENTILATOR_RANK)){
            which1 = which(VENTILATOR_RANK==UN_VENTILATOR_RANK[i])
            if(length(which1)>1){
              ###these are all tied, can we re-rank based on Advanced support?
              TIE_BREAK_DAYS_ADVANCED =   sort(unique(DAYS_ADVANCED[which_no_best_worst_outcome_composite[which1]]))

              GRID =  .001*(1:length(TIE_BREAK_DAYS_ADVANCED))/length(TIE_BREAK_DAYS_ADVANCED)

              ###Creates a grid for tie-breaks
              ##We will subtract each value of the grid
              ##in order of days on advanced respiratory support
              ##Goal is to get the composite ranking order

              for(j in 1:length(TIE_BREAK_DAYS_ADVANCED)){
                ###Look through the order of unique tie break days in order, subtract the corresponding grid value from the ventilator score
                which2 = which(DAYS_ADVANCED[which_no_best_worst_outcome_composite[which1]] == TIE_BREAK_DAYS_ADVANCED[j])

                ###All these patients had this # of days on advanced support
                VENTILATOR_RANK[which1[which2]]=VENTILATOR_RANK[which1[which2]]-GRID[j]
              }

            }
          }




          COMPOSITE_RANKING[which_no_best_worst_outcome_composite]=VENTILATOR_RANK


          ###Now we can re-rank again.
          COMPOSITE_RANKING = rank(COMPOSITE_RANKING,ties.method = "min")

          m1 =   dfba_mann_whitney(COMPOSITE_RANKING[TRT==1], COMPOSITE_RANKING[TRT==0],hide_progress = FALSE)
          STORE_P[b,look_count]=m1$prH1
          ###Do we have a superiority look for the Bayesian Wilcox?
          if(!is.na(cut_Wilcox[look_count])){
            POWER[b]=m1$prH1>cut_Wilcox[look_count]
          }

          ###Do we have a superiority look for mortality?
          if(!is.na(cut_gating[look_count])){
            X1 = table(TRT,DEATH)   ###Gating based on decreased death rates
            if(POWER[b]==0){
              POWER[b] = fisher.test(X1,alternative="less")$p.value< cut_gating[look_count]
            }
          }


          ###Do we have a futility look?
          if(!is.na(Futility_BOUND[look_count])){
            FUTILE=m1$prH1<Futility_BOUND[look_count]
          }



        }





        if(POWER[b]==1 | FUTILE == 1 | SAFETY_STOP==1){
          SAMPLE_SIZE[b]=length(TRT)
        }else{
          rand_prob = max(min(m1$prH1,.5+cut_AR[look_count]),.5-cut_AR[look_count])

          look_count=look_count+1
        }

      }



    }else{
      SAMPLE_SIZE[b]=length(TRT)

    }


  } ###End of simulations loop



  DEATH_VEC = c(round(Prob_Death_Treatment,3)*100,round(Prob_Death,3)*100)
  names(DEATH_VEC)=c("Treatment 28-Day Mortality","Control 28-Day Mortality")
 # VEC = c(VEC[1],DEATH_VEC, VEC[2:length(VEC)])

  TRUE_MAT=matrix(nrow=3, ncol=2)
  rownames(TRUE_MAT)=c("28-Day Mortality",
                       "Average Ventilator Days",
                       "Average Advanced Respiratory Days")
  colnames(TRUE_MAT)=c("Treatment","Control")

  ###28-day mortality
  TRUE_MAT[1,1]=DEATH_VEC[1]
  TRUE_MAT[1,2]=DEATH_VEC[2]
  ###Ventilator Days
  TRUE_MAT[2,1]=VEC[2]
  TRUE_MAT[2,2]=VEC[3]
  ###Advanced Days
  TRUE_MAT[3,1]=VEC[4]
  TRUE_MAT[3,2]=VEC[5]


  RES1 = as.list(rep(NA,2))
  names(RES1)=c("Win Probability","True Parameters")
  RES1[[1]]=VEC[1]
  RES1[[2]]=TRUE_MAT


  RESULT_LIST = as.list(rep(NA,3))
  names(RESULT_LIST)=  c("True Scenario","Design Parameters" ,"Operating Characteristics")
  RESULT_LIST[[1]]=RES1

  DESIGN_PARAMETERS =rbind( cut_Wilcox,
                            cut_gating,
                            SAFETY_BOUND,
                            Futility_BOUND,
                            cut_AR)
  rownames(DESIGN_PARAMETERS)=c("Bayesian Wilcoxon Superiority Boundary",
                                "One-sided Fisher Exact test on Mortality Superiority Boundary",
                                "Safety Boundary for Increases in Mortality in Treatment Group",
                                "Bayesian Wilcoxon Futility Boundary",
                                "Adaptive Randomization Window")
  colnames(DESIGN_PARAMETERS)=N_LOOK
  RESULT_LIST[[2]]=DESIGN_PARAMETERS


  OCS = as.list(rep(NA,3))
  names(OCS)=c("Graduation Probability","Sample Size Distribution","Graduation and Sample Size")
  OCS[[1]]=round(mean(POWER),3)*100
  SS = as.list(rep(NA,2))
  names(SS)=c("Average","Distribution")
  SS[[1]]=round(mean(SAMPLE_SIZE),2)*10
  PCT = rep(NA,length(N_LOOK))
  for(j in 1:length(N_LOOK)){
    PCT[j]=round(mean(SAMPLE_SIZE==N_LOOK[j]),3)*100
  }
  MAT1= rbind(N_LOOK,PCT)
  rownames(MAT1)=c("Sample Size","% Trial Stopped")
  OCS[[2]]= MAT1


  ###Lastly the power/% distribution
  PCT = matrix(nrow=2,ncol=length(N_LOOK))
  for(j in 1:length(N_LOOK)){
    PCT[1,j]=round(mean(SAMPLE_SIZE==N_LOOK[j] & POWER==0),3)*100
    PCT[2,j]=round(mean(SAMPLE_SIZE==N_LOOK[j] & POWER==1),3)*100
  }
  rownames(PCT)=c("No Graduation","Graduation")
  colnames(PCT)=N_LOOK

  OCS[[3]]=PCT


RESULT_LIST[[3]]=OCS

  return(RESULT_LIST)
  # print(c(mean(POWER),mean(SAMPLE_SIZE)))

  # SCENARIOS$Gated_Power[scenario_num] =mean(POWER)
  # SCENARIOS$Sample_Size[scenario_num] =mean(SAMPLE_SIZE)



}


