#' Perform Longitudinal Analysis with Linear Mixed Models (LMM)
#'
#' This function automates the analysis of longitudinal clinical data using linear mixed models.
#' It models clinical variables and returns a weighted matrix of model coefficient scores.
#'
#' @param clinical_data Dataframe containing clinical and metadata for participants, including identifier as `participant_id`.
#' @param variables_to_scale Character vector of variable names to be analyzed.
#' @param random_effects A character string specifying the random effects formula (default: "(1 | participant_id)").
#' @return A matrix of model coefficient scores, where rows represent dependent variables and columns represent independent variables.
#' @import dplyr lme4 qgraph
#' @importFrom stats as.formula
#' @export
lmm_analysis <- function(clinical_data, variables_to_scale, random_effects = "(1 | participant_id)") {
  # Ensure all selected variables exist in the dataset
  variables_to_scale <- intersect(variables_to_scale, colnames(clinical_data))

  if (length(variables_to_scale) == 0) {
    stop("No valid variables provided for scaling.")
  }

  # Scale selected variables
  clinical_data <- clinical_data %>%
    mutate(across(all_of(variables_to_scale), scale))

  # Fit LMM models
  models <- list()
  variables_string <- paste(variables_to_scale, collapse = " + ")

  for (dependent_variable in variables_to_scale) {
    formula <- as.formula(paste(dependent_variable, "~", variables_string, "+", random_effects))

    tryCatch({
      # Suppress warnings from lmer()
      model <- suppressWarnings(lmer(formula, data = clinical_data))
      models[[dependent_variable]] <- summary(model)$coefficients[, 3]
    }, error = function(e) {
      warning(paste("Error fitting model for", dependent_variable, ":", e$message))
    })
  }

  # Convert model results to a matrix
  if (length(models) == 0) {
    stop("No models were successfully fitted.")
  }

  score <- do.call(rbind, models)
  colnames(score) <- variables_to_scale

  score_matrix <- score_matrix(score, shift_matrix(score))

  # No directed edges are assumed and mean of both sides is calculated
  symm_score <- (score_matrix + t(score_matrix))/2

  return(symm_score)
}
