#' Plot PCO axes
#'
#' `plot()` visualizes the relationship between a PCO axis and the vertebra or between pairs of PCO axes.
#'
#' @param x a `regions_pco` object; the output of a call to [svdPCO()].
#' @param pco_y,pco_x number; PCO score indices for the y- and x-axes, respectively. `pco_x` can be `NULL`.
#' @param ... arguments passed to [ggplot2::geom_text()] when `pco_x` is not `NULL`. If `scores` is supplied as an argument, it will replace `pco_y` if unspecified.
#'
#' @return A `ggplot` object.
#'
#' @details
#' When `pco_x` is `NULL` (the default), `plot()` will display a scatterplot of the PCO axis identified by `pco_y` and vertebra position using [ggplot2::geom_point()]. This plot is similar to that generated by [plotsegreg()]. Otherwise, `plot()` uses [ggplot2::geom_text()] to identify vertebrae positions in the space corresponding to the requested PCOs.
#'
#' @seealso
#'
#' [svdPCO()] for generating the PCO scores. [plot.regions_sim()] for plotting PCO scores against vertebra position for simulated PCOs. [plotsegreg()] for plotting PCO scores against vertebra position after selecting breakpoints for a segmented regression.
#'
#' @example man/examples/example-svdPCO.R

#' @exportS3Method plot regions_pco
plot.regions_pco <- function(x, pco_y = 1, pco_x = NULL, ...) {

  if (identical(pco_y, formals()$pco_y)) {
    dotnames <- ...names()
    if ("scores" %in% dotnames) {
      scores <- ...elt(which(dotnames == "scores")[1])
      chk::chk_whole_numeric(scores)
      if (!is.null(pco_x) && length(pco_y) > 1) {
        chk::err("`scores` can only have length 1 when `pco_x` is specified")
      }
      chk::chk_range(scores, c(1, ncol(x[["scores"]])))
      pco_y <- scores
    }
  }

  pos <- .get_pos(x)
  specimen <- attr(x, "specimen")

  if (is.null(pco_x)) {
    chk::chk_whole_numeric(pco_y)
    chk::chk_range(pco_y, c(1, ncol(x[["scores"]])))

    pco_y <- sort(pco_y)

    Yvar <- x[["scores"]][, pco_y, drop = FALSE]

    # Plot PCO against vertebrae using plotsegreg() machinery
    return(.plotreg_internal(Xvar = pos, Yvar = Yvar, lines = FALSE,
                             scores = pco_y, specimen = specimen))
  }

  chk::chk_whole_numeric(pco_y)
  if (length(pco_y) > 1) {
    chk::err("`pco_y` can only have length 1 when `pco_x` is specified")
  }
  chk::chk_range(pco_y, c(1, ncol(x[["scores"]])))

  Yvar <- x[["scores"]][, pco_y]

  chk::chk_whole_number(pco_x)
  chk::chk_range(pco_x, c(1, ncol(x[["scores"]])))

  Xvar <- x[["scores"]][, pco_x]

  p <- ggplot(mapping = aes(x = Xvar,
                            y = Yvar,
                            label = pos,
                            color = if (nlevels(specimen) > 1) specimen)) +
    geom_text(...) +
    geom_point(alpha = 0) +
    theme_bw() +
    guides(color = guide_legend(override.aes = aes(label = "", alpha = 1))) +
    labs(title = "Principal coordinates analysis",
         x = paste("PCO", pco_x),
         y = paste("PCO", pco_y),
         color = NULL)

  p
}
