#===============================================================================
#                      MDT Test Functions
#===============================================================================

#' @title Multivariate Dependence Trend Test
#'
#' @description Computes the test statistic for the multivariate dependence trend
#'  (MDT) test and the p-value.
#'
#'@param data Numeric matrix, a two dimensional data set where each column is a
#'  variable.
#' @param width Numeric value, the width to be used for the *moving window* algorithm.
#'  Should be bigger than 1 and smaller than the number of observations, see **details**.
#' @param alpha Numeric value, significance level for the test. Must be between 0 and 1, default is 0.05
#' @param Nbs Positive Integer, number of samples to be used for bootstrapping. Default is 1000.
#'
#' @return Named list containing:
#'
#'  - `statistic`, the MDT test statistic.
#'  - `p.value`, computed *p-value* computed using bootstrap.
#'  - `result`, the result of the test. If `TRUE` the trend is considered significant.
#'
#' @details
#' The multivariate dependence trend (MDT) test allows to check for trend in the data series dependence
#' structure.
#' The moving window technique has been employed in order to take into account the dependence evolution,
#' represented by Kendall's \eqn{\tau}, according to time.
#'
#' The test proposes a null hypothesis of no trend, against an alternative hypothesis of a monotonic trend in the dependence structure.
#' Goutali and Chebana (2024) propose the following test statistic:
#'
#' Let \eqn{X = (x_1, x_2, ..., x_n)} and \eqn{Y = (y_1, y_2, ..., y_n)}
#' be two variables of a data series, the test statistic of the MDT test is given by:
#'
#' \deqn{T_{MDT}=\tau_n(\tau_{nw}(X,Y),T')}
#'
#' where \eqn{\tau_n} is the empirical version of bivariate Kendall's \eqn{\tau}. The
#' series of the Kendall's \eqn{\tau} obtained through moving window with width \eqn{w}
#' is \eqn{\tau_{nw}(X, Y)}, this series has length \eqn{q = n - w +1}. \eqn{T' = (1, 2, ..., q)} is the time order of the rolling window series.
#'
#' The choice of `width` is a trade-off. A small \eqn{w} increases the number of
#' rolling windows for reliable analysis, while a large \eqn{w} is necessary to have
#' sufficient values to identify the dependence structure. The *p-value* is computed using a bootstrap procedure.
#'
#' @references
#' Goutali, D., and Chebana, F. (2024). Multivariate overall and dependence trend tests, applied to hydrology,
#' Environmental Modelling & Software, 179, \doi{10.1016/j.envsoft.2024.106090}
#'
#' @seealso
#' \itemize{
#'  \item `kendall.tau` : Function from the package `VGAM` used for computing the bivariate Kendall's \eqn{\tau}.
#'  \item `rollapply` : Function from the package `zoo` used to apply the moving window technique.
#'  \item `samp.bootstrap`: Function from the `resample` package, used to generate the samples
#'    necessary to perform bootstrapping.
#' }
#'
#' @examples
#'
#' \donttest{
#' # CASE 1: Only trend in the dependence structure
#' # Sample data:
#' DependenceStructure <- generate_data("dependenceStructure")
#'
#' width <- 10
#'
#' # Perform the mdt test:
#' mdt <- MDT_test(DependenceStructure, width, alpha = 0.05, Nbs = 1000)
#' print(mdt)
#'
#'
#' # CASE 2: Only trend in the marginal distributions
#' # Sample data:
#' MarginalTrend <- generate_data("marginalTrend")
#'
#' # Perform the mdt test:
#' mdt <- MDT_test(MarginalTrend, width)
#' print(mdt)
#'
#'
#' # CASE 3: No trend
#' # Sample data:
#' NoTrend <- generate_data("noTrend")
#'
#' # Perform the mdt test:
#' mdt <- MDT_test(NoTrend, width)
#' print(mdt)
#' }
#'
#'
#' @importFrom zoo rollapply
#' @importFrom VGAM kendall.tau
#' @importFrom resample samp.bootstrap
#'
#' @export

MDT_test <- function(data,
                     width,
                     alpha = 0.05,
                     Nbs = 1000) {}

MDT_test <- function(data,
                     width,
                     alpha = 0.05,
                     Nbs = 1000) {

  data_ <- as.matrix(data)

  # Input checks:

  if (!is.numeric(data_)) {
    stop("'data' must be numeric")
  }

  if (!is.numeric(width)) {
    stop("Parameter 'width' must be numeric")
  }

  if (!is.numeric(alpha)) {
    stop("Parameter 'alpha' must be numeric")
  }

  if (!is.numeric(Nbs)) {
    stop("Parameter 'Nbs' must be numeric")
  }

  # Series length
  n = nrow(data_)

  if (width < 1 || width > n) {
    stop("Parameter 'width' must be bigger than 1 and smaller than number of observations")
  } else if (width == 1 || width == n) {
    warning("Inadequate value for 'width' parameter, must be bigger than 1 and smaller than number of observations")
  }

  if ((Nbs %% 1) != 0 || Nbs <= 0) {
    stop("Parameter 'Nbs' must be a positive integer.")
  }

  if (alpha <= 0 || alpha >= 1) {
    stop("Parameter 'alpha' must be between 0 and 1")
  }

  results <- MDT_pvalue(data_, width, alpha, Nbs)
  return(results)
}

# ==============================================================================
# MDT TEST STATISTIC

MDT_statistic <- function(data, width){

  n <- dim(data)[1]

  # Variable declaration
  T2 = 1:(n - width + 1)

  # Defining 'kendall.tau' as a function
  kend <- function(x){kendall.tau(x[, 1], x[, 2])}

  # Getting correlation coefficients between 'x' and 'y' in different subsets,
  # moving or rolling window

  roll = rollapply(data, width = width, FUN = kend,
                   by.column = FALSE, align = "right")

  # Computign Kendall's tau
  dep = kendall.tau(as.numeric(roll), T2)

  return(abs(dep))
}

#===============================================================================
# P-VALUE COMPUTATION

MDT_pvalue <- function(data, width, alpha, Nbs){

  n <- dim(data)[1]

  # Test statistic storage
  Sn = rep(0, Nbs)

  # Matrix to be used as indices for the resampling with replacement
  index = samp.bootstrap(n, Nbs)

  # p-value estimation via bootstrapping
  for (i in 1:Nbs) {
    dat_sample = data[index[, i],]
    Sn[i] = MDT_statistic(dat_sample, width)
  }

  thresh <- MDT_statistic(data, width)

  pvalue = sum(Sn > rep(thresh, Nbs)) / Nbs

  if (pvalue < alpha) {
    result = TRUE
  } else {
    result = FALSE
  }

  return(list("statistic" = thresh, "p.value" = pvalue, "result" = result))
}


