# InvEEP.R
#' Inverse Exponentiated Exponential Poisson (IEEP) Distribution
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Inverse Exponentiated Exponential Poisson distribution.
#'
#' The Inverse Exponentiated Exponential Poisson distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Inverse Exponentiated Exponential Poisson distribution has CDF:
#'
#' \deqn{
#' F(x;\,\alpha,\beta,\lambda) = 1 - \frac{1}{{\left( {1 - {e^{ - \lambda }}} \right)}}
#' \left[ {1 - \exp \left\{ { - \lambda {{\left( {1 - {e^{ - \beta /x}}} \right)}^
#' \alpha }} \right\}} \right]\,; \quad x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The implementation includes the following functions:
#' \itemize{
#'   \item \code{dinv.expo.exp.pois()} — Density function
#'   \item \code{pinv.expo.exp.pois()} — Distribution function
#'   \item \code{qinv.expo.exp.pois()} — Quantile function
#'   \item \code{rinv.expo.exp.pois()} — Random generation
#'   \item \code{hinv.expo.exp.pois()} — Hazard function
#' }
#'
#' @usage
#' dinv.expo.exp.pois(x, alpha, beta, lambda, log = FALSE)
#' pinv.expo.exp.pois(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qinv.expo.exp.pois(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rinv.expo.exp.pois(n, alpha, beta, lambda)
#' hinv.expo.exp.pois(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dinv.expo.exp.pois}: numeric vector of (log-)densities
#'   \item \code{pinv.expo.exp.pois}: numeric vector of probabilities
#'   \item \code{qinv.expo.exp.pois}: numeric vector of quantiles
#'   \item \code{rinv.expo.exp.pois}: numeric vector of random variates
#'   \item \code{hinv.expo.exp.pois}: numeric vector of hazard values
#' }
#'
#' @references  
#' Ristic, M.M., & Nadarajah, S.(2014). 
#' A New Lifetime Distribution. 
#' \emph{Journal of Statistical Computation and Simulation}, \bold{84(1)}, 135--150. 
#' \doi{10.1080/00949655.2012.697163}
#' 
#' Telee, L. B. S., & Kumar, V. (2023). 
#' Inverse Exponentiated Exponential Poisson Distribution with Theory and 
#' Applications. \emph{International Journal of Engineering Science Technologies}, 
#' \bold{7(5)}, 17--36.  \doi{10.29121/IJOEST.v7.i5.2023.535}
#'  
#' @examples
#' x <- seq(0.1, 1, 0.1)
#' dinv.expo.exp.pois(x, 1.5, 0.8, 2)
#' pinv.expo.exp.pois(x, 1.5, 0.8, 2)
#' qinv.expo.exp.pois(0.5, 1.5, 0.8, 2)
#' rinv.expo.exp.pois(10, 1.5, 0.8, 2)
#' hinv.expo.exp.pois(x, 1.5, 0.8, 2)
#' 
#' #Data
#' x <- conductors
#' #ML Estimates    
#' params = list(alpha =40.5895, beta=22.7519, lambda=2.9979)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = pinv.expo.exp.pois, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qinv.expo.exp.pois, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' res <- gofic(x, params = params, dfun = dinv.expo.exp.pois, 
#'              pfun = pinv.expo.exp.pois, plot=FALSE)
#' print.gofic(res)
#' 
#' @name InvEEP
#' @aliases dinv.expo.exp.pois pinv.expo.exp.pois qinv.expo.exp.pois rinv.expo.exp.pois hinv.expo.exp.pois
#' @export
NULL

#' @rdname InvEEP
#' @usage NULL
#' @export 
dinv.expo.exp.pois <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
    u <- 1 - exp(-beta / x[valid])
	z <- 1.0/(1 - exp(-lambda))
	w <- x[valid] ^ -2
    pdf[valid] <- alpha * beta * lambda * z * w * exp(-beta/x[valid]) *(u ^(alpha-1)) * exp(-lambda * (u ^ alpha))
    }
	if (log) 
       pdf<- log(pdf)
    return(pdf)   
}

#' @rdname InvEEP
#' @usage NULL
#' @export 
pinv.expo.exp.pois <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {	
    u <- 1 - exp(-beta / q[valid])
	z <- 1.0/(1 - exp(-lambda))
    cdf[valid] <- 1.0 - z * (1.0 - exp(-lambda * (u ^ alpha)))
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname InvEEP
#' @usage NULL
#' @export
qinv.expo.exp.pois <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta' and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {	
	z <- (1.0 - exp(-lambda)) 	
    qtl[valid] <- -(1.0/beta) * log(1.0 - ((-(1.0/lambda)*log(1.0- z *(1-p[valid]))) ^ (1.0/alpha)))
	qtl[valid] <- 1/qtl[valid]
    }
    return(qtl)   
}

#' @rdname InvEEP
#' @usage NULL
#' @export 
rinv.expo.exp.pois <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qinv.expo.exp.pois(u, alpha, beta, lambda)
    return(q)
}

#' @rdname InvEEP
#' @usage NULL
#' @export 
hinv.expo.exp.pois <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")  
    nume <- dinv.expo.exp.pois(x, alpha, beta, lambda, log = FALSE)
	surv <- 1.0 - pinv.expo.exp.pois(x, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
}
