# PoisInvWeib.R
#' Poisson Inverse Weibull Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Poisson Inverse Weibull distribution.
#'
#' The Poisson Inverse Weibull distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Poisson Inverse Weibull distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \quad \frac{1}{\left(1-e^{-\lambda}\right)}\left[1-\exp 
#' \left\{-\lambda \exp \left(-(\alpha / x)^\beta\right)
#' \right\}\right] \quad ;\;x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' @usage
#' dpois.inv.weib(x, alpha, beta, lambda, log = FALSE)
#' ppois.inv.weib(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qpois.inv.weib(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rpois.inv.weib(n, alpha, beta, lambda)
#' hpois.inv.weib(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dpois.inv.weib}: numeric vector of (log-)densities
#'   \item \code{ppois.inv.weib}: numeric vector of probabilities
#'   \item \code{qpois.inv.weib}: numeric vector of quantiles
#'   \item \code{rpois.inv.weib}: numeric vector of random variates
#'   \item \code{hpois.inv.weib}: numeric vector of hazard values
#' }
#'
#' @references 
#' Kus, C. (2007). A new lifetime distribution. 
#' \emph{Computational Statistics and Data Analysis}, \bold{51}, 4497--4509.
#' 
#' Joshi, R. K., & Kumar, V. (2021). 
#' Poisson Inverse Weibull Distribution with Theory and Applications. 
#' \emph{International Journal of Statistics and Systems}, \bold{16(1)}, 1--16. 
#'
#' Rodrigues, G.C., Louzada, F., & Ramos, P.L.(2018). 
#' Poisson–exponential distribution: different methods of estimation. 
#' \emph{Journal of Applied Statistics}, \bold{45(1)}, 128--144.
#'
#' @examples
#' x <- seq(0.1, 10, 0.2)
#' dpois.inv.weib(x, 2.0, 0.5, 0.2)
#' ppois.inv.weib(x, 2.0, 0.5, 0.2)
#' qpois.inv.weib(0.5, 2.0, 0.5, 0.2)
#' rpois.inv.weib(10, 2.0, 0.5, 0.2)
#' hpois.inv.weib(x, 2.0, 0.5, 0.2)
#' 
#' # Data
#' x <- fibers63
#' # ML estimates
#' params = list(alpha=5.5146, beta=1.8811, lambda=16.2341)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = ppois.inv.weib, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qpois.inv.weib, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dpois.inv.weib, pfun=ppois.inv.weib, plot=TRUE)
#' print.gofic(out)
#' 
#' @name PoisInvWeib
#' @aliases dpois.inv.weib ppois.inv.weib qpois.inv.weib rpois.inv.weib hpois.inv.weib
#' @export
NULL

#' @rdname PoisInvWeib
#' @usage NULL
#' @export   
dpois.inv.weib <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
	tmp <- 1/(1 - exp(-lambda))
	w <- (alpha/x[valid])
	z <- exp(-w ^ beta)
    gx <- (beta/alpha) * lambda * (w^(beta+1)) * z * exp(-lambda *z)	
    pdf[valid] <- tmp * gx
	}
    if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname PoisInvWeib
#' @usage NULL
#' @export    
ppois.inv.weib <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	tmp <- 1/(1 - exp(-lambda))
	z <- exp(-(alpha/q[valid]) ^ beta)
    cdf[valid] <- tmp * (1-exp(-lambda * z))
	}
    if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname PoisInvWeib
#' @usage NULL
#' @export   
qpois.inv.weib <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta' and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
	z <- -(1/lambda) * log(1-(1-exp(-lambda))*p[valid])		
    qtl[valid] <- alpha * ((-log(z)) ^ -(1/beta))    
	}
    return(qtl)   
 } 

#' @rdname PoisInvWeib
#' @usage NULL
#' @export   
rpois.inv.weib <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qpois.inv.weib(u, alpha, beta, lambda)
    return(q)
}

#' @rdname PoisInvWeib
#' @usage NULL
#' @export   
hpois.inv.weib <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
		
    nume <- dpois.inv.weib(x, alpha, beta, lambda)
	surv <- 1.0 - ppois.inv.weib(x, alpha, beta, lambda)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
} 
