#' Numerical derivatives of the overlap functional (normal vs. 2-component mixture).
#'
#' Computes finite-difference approximations of the partial derivatives of the
#' overlap coefficient with respect to the parameters in the case where
#' the first group is modeled as a normal distribution and the second group as
#' a two-component Gaussian mixture.
#'
#' This function is intended for internal use (delta-method variance estimation).
#'
#' @param mu1_hat Numeric scalar. Estimated mean for the normal group.
#' @param mu2_hat Numeric vector of length 2. Estimated means for the mixture group.
#' @param sigma1_hat Numeric scalar. Estimated standard deviation for the normal group.
#' @param sigma2_hat Numeric vector of length 2. Estimated standard deviations for the mixture group.
#' @param pi2_hat Either a numeric scalar in [0,1] giving the mixing proportion of the first component in the
#' mixture, or a numeric vector of length 2 with elements summing to 1.
#' @param h Positive numeric scalar. Base step size for finite differences.
#' @param intersec Numeric vector. Intersection points used as cutpoints.
#' @param OVL_mix Function that evaluates the overlap-related expression at cutpoints.
#'
#' @return A list with components \code{deriv1}–\code{deriv7}.
#'
#' @keywords internal

OVL_derivates_1 <- function(mu1_hat, mu2_hat, sigma1_hat, sigma2_hat, pi2_hat, h, intersec, OVL_mix) {

  # Auxiliares: primera evita que se fije un h demasiado pequeño, segunda para controlar que los pesos
  #no se salgan de [0, 1].
  .ovl_scale_h <- function(h, x) {
    sc <- max(1, abs(x), na.rm = TRUE)
    max(h, sqrt(.Machine$double.eps) * sc)
  }
  .ovl_clamp01 <- function(p, eps = sqrt(.Machine$double.eps)) {
    min(max(p, eps), 1 - eps)
  }

  force(OVL_mix)

  #h adaptativo y estabilidad en parámetros
  h_mu1    <- .ovl_scale_h(h, mu1_hat)
  h_sigma1 <- .ovl_scale_h(h, sigma1_hat)
  h_mu2    <- .ovl_scale_h(h, mu2_hat)
  h_sigma2 <- .ovl_scale_h(h, sigma2_hat)
  h_pi2    <- .ovl_scale_h(h, pi2_hat[1])

  # asegura que sigma +/- h siga siendo > 0
  h_sigma1 <- min(h_sigma1, 0.49 * sigma1_hat)
  h_sigma2 <- pmin(h_sigma2, 0.49 * sigma2_hat)

  # asegura pi2[1] en (0,1) y que pi2[1] +/- h siga en (0,1)
  p2 <- .ovl_clamp01(pi2_hat[1])
  h_pi2 <- min(h_pi2, 0.49 * min(p2, 1 - p2))
  pi2_hat <- c(p2, 1 - p2)

  # Perturbaciones para mu1
  mu1_h_1 <- mu1_hat + h_mu1
  mu1_h_2 <- mu1_hat - h_mu1

  # Perturbaciones para mu2
  mu21_h_1 <- c(mu2_hat[1] + h_mu2, mu2_hat[2])
  mu21_h_2 <- c(mu2_hat[1] - h_mu2, mu2_hat[2])
  mu22_h_1 <- c(mu2_hat[1], mu2_hat[2] + h_mu2)
  mu22_h_2 <- c(mu2_hat[1], mu2_hat[2] - h_mu2)

  # Perturbaciones para sigma1
  sigma1_h_1 <- sigma1_hat + h_sigma1
  sigma1_h_2 <- sigma1_hat - h_sigma1

  # Perturbaciones para sigma2
  sigma21_h_1 <- c(sigma2_hat[1] + h_sigma2[1], sigma2_hat[2])
  sigma21_h_2 <- c(sigma2_hat[1] - h_sigma2[1], sigma2_hat[2])
  sigma22_h_1 <- c(sigma2_hat[1], sigma2_hat[2] + h_sigma2[2])
  sigma22_h_2 <- c(sigma2_hat[1], sigma2_hat[2] - h_sigma2[2])

  # Perturbaciones para pi2
  pi2_h_1 <- c(pi2_hat[1] + h_pi2, 1 - (pi2_hat[1] + h_pi2))
  pi2_h_2 <- c(pi2_hat[1] - h_pi2, 1 - (pi2_hat[1] - h_pi2))

  # Aproximación de Taylor de orden 2
  deriv1 <- (OVL_mix(mu1_h_1, mu2_hat, sigma1_hat, sigma2_hat, 1, pi2_hat, intersec) -
               OVL_mix(mu1_h_2, mu2_hat, sigma1_hat, sigma2_hat, 1, pi2_hat, intersec)) / (2 * h_mu1)

  deriv2 <- (OVL_mix(mu1_hat, mu2_hat, sigma1_h_1, sigma2_hat, 1, pi2_hat, intersec) -
               OVL_mix(mu1_hat, mu2_hat, sigma1_h_2, sigma2_hat, 1, pi2_hat, intersec)) / (2 * h_sigma1)

  deriv3 <- (OVL_mix(mu1_hat, mu21_h_1, sigma1_hat, sigma2_hat, 1, pi2_hat, intersec) -
               OVL_mix(mu1_hat, mu21_h_2, sigma1_hat, sigma2_hat, 1, pi2_hat, intersec)) / (2 * h_mu2)

  deriv4 <- (OVL_mix(mu1_hat, mu22_h_1, sigma1_hat, sigma2_hat, 1, pi2_hat, intersec) -
               OVL_mix(mu1_hat, mu22_h_2, sigma1_hat, sigma2_hat, 1, pi2_hat, intersec)) / (2 * h_mu2)

  deriv5 <- (OVL_mix(mu1_hat, mu2_hat, sigma1_hat, sigma21_h_1, 1, pi2_hat, intersec) -
               OVL_mix(mu1_hat, mu2_hat, sigma1_hat, sigma21_h_2, 1, pi2_hat, intersec)) / (2 * h_sigma2[1])

  deriv6 <- (OVL_mix(mu1_hat, mu2_hat, sigma1_hat, sigma22_h_1, 1, pi2_hat, intersec) -
               OVL_mix(mu1_hat, mu2_hat, sigma1_hat, sigma22_h_2, 1, pi2_hat, intersec)) / (2 * h_sigma2[2])

  deriv7 <- (OVL_mix(mu1_hat, mu2_hat, sigma1_hat, sigma2_hat, 1, pi2_h_1, intersec) -
               OVL_mix(mu1_hat, mu2_hat, sigma1_hat, sigma2_hat, 1, pi2_h_2, intersec)) / (2 * h_pi2)

  return(list(deriv1 = deriv1, deriv2 = deriv2, deriv3 = deriv3, deriv4 = deriv4, deriv5 = deriv5, deriv6 = deriv6, deriv7 = deriv7))
}

