% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/probitRE_probitRE.R
\name{probitRE_probitRE}
\alias{probitRE_probitRE}
\title{Panel Sample Selection Model for Binary Outcome}
\usage{
probitRE_probitRE(
  probit1,
  probit2,
  id.name,
  data = NULL,
  par = NULL,
  method = "BFGS",
  rho_off = FALSE,
  tau_off = FALSE,
  H = 10,
  init = c("zero", "unif", "norm", "default")[4],
  rho.init = 0,
  tau.init = 0,
  use.optim = FALSE,
  verbose = 0
)
}
\arguments{
\item{probit1}{Formula for the first-stage probit model with random effects at the individual level}

\item{probit2}{Formula for the second-stage probit model with random effects at the individual level}

\item{id.name}{the name of the id column in data}

\item{data}{Input data, must be a data.frame object}

\item{par}{Starting values for estimates}

\item{method}{Optimization algorithm. Default is BFGS}

\item{rho_off}{A Boolean value indicating whether to turn off the correlation between the random effects of the probit and linear models. Default is FALSE.}

\item{tau_off}{A Boolean value indicating whether to turn off the correlation between the error terms of the probit and linear models. Default is FALSE.}

\item{H}{Number of quadrature points}

\item{init}{Initialization method}

\item{rho.init}{Initial value for the correlation between the random effects of the probit and linear models. Default is 0.}

\item{tau.init}{Initial value for the correlation between the error terms of the probit and linear models. Default is 0.}

\item{use.optim}{A Boolean value indicating whether to use optim instead of maxLik. Default is FALSE.}

\item{verbose}{A integer indicating how much output to display during the estimation process.
\itemize{
\item <0 - No ouput
\item 0 - Basic output (model estimates)
\item 1 - Limited output, providing likelihood of iterations
\item 2 - Moderate output, basic ouput + parameter and likelihood on each call
\item 3 - Extensive output, moderate output + gradient values on each call
}}
}
\value{
A list containing the results of the estimated model, some of which are inherited from the return of maxLik
\itemize{
\item estimates: Model estimates with 95\% confidence intervals
\item estimate or par: Point estimates
\item predict. A list containing the predicted probabilities of responding (respond_prob) and the predicted counterfactual outcome values (outcome_prob), their gradients (gr_respond and gr_outcome), and estimated counterfactual population mean (pop_mean).
\item variance_type: covariance matrix used to calculate standard errors. Either BHHH or Hessian.
\item var: covariance matrix
\item se: standard errors
\item var_bhhh: BHHH covariance matrix, inverse of the outer product of gradient at the maximum
\item se_bhhh: BHHH standard errors
\item gradient: Gradient function at maximum
\item hessian: Hessian matrix at maximum
\item gtHg: \eqn{g'H^-1g}, where H^-1 is simply the covariance matrix. A value close to zero (e.g., <1e-3 or 1e-6) indicates good convergence.
\item LL or maximum: Likelihood
\item AIC: AIC
\item BIC: BIC
\item n_obs: Number of observations
\item n_par: Number of parameters
\item time: Time takes to estimate the model
\item iterations: number of iterations taken to converge
\item message: Message regarding convergence status.
}

Note that the list inherits all the components in the output of maxLik. See the documentation of maxLik for more details.
}
\description{
A panel sample selection model for binary outcome, with selection at both the individual and individual-time levels. The outcome is observed in the second stage only if the first stage outcome is one.\cr\cr
Let \eqn{\mathbf{w}_{it}} and \eqn{\mathbf{x}_{it}} represent the \emph{row} vectors of covariates in the selection and outcome equations, respectively, with \eqn{\boldsymbol{\alpha}} and \eqn{\boldsymbol{\beta}} denoting the corresponding \emph{column} vectors of parameters.\cr\cr
First stage (probitRE):
\deqn{d_{it}=1(\mathbf{w}_{it} \boldsymbol{\alpha}+\delta u_i+\varepsilon_{it}>0)}{d_it = 1(w_it * \alpha + \delta * u_i +\varepsilon_it > 0)}
Second stage (probitRE):
\deqn{y_{it} = 1(\mathbf{x}_{it} \boldsymbol{\beta} + \lambda v_i +\epsilon_{it}>0)}{y_it = 1(x_it * \beta + \gamma * m_i + \lambda * v_i + \epsilon_it > 0)}
Correlation structure:
\eqn{u_i} and \eqn{v_i} are bivariate normally distributed with a correlation of \eqn{\rho}.
\eqn{\varepsilon_{it}} and \eqn{\epsilon_{it}} are bivariate normally distributed with a correlation of \eqn{\tau}. \cr\cr
w and x can be the same set of variables. Identification can be weak if w are not good predictors of d.
}
\examples{
library(PanelSelect)
library(MASS)
N = 150
period = 5
obs = N*period
rho = 0.5
tau = 0
set.seed(100)

re = mvrnorm(N, mu=c(0,0), Sigma=matrix(c(1,rho,rho,1), nrow=2))
u = rep(re[,1], each=period)
v = rep(re[,2], each=period)
e = mvrnorm(obs, mu=c(0,0), Sigma=matrix(c(1,tau,tau,1), nrow=2))
e1 = e[,1]
e2 = e[,2]

t = rep(1:period, N)
id = rep(1:N, each=period)
w = rnorm(obs)
z = rnorm(obs)
x = rnorm(obs)
d = as.numeric(x + w + u + e1 > 0)
y = as.numeric(x + w + v + e2 > 0)
y[d==0] = NA
dt = data.frame(id, t, y, x, w, z, d)

# As N increases, the parameter estimates will be more accurate
m = probitRE_probitRE(d~x+w, y~x+w, 'id', dt, H=10, verbose=-1)
print(m$estimates, digits=4)
}
\references{
Bailey, M., & Peng, J. (2025). A Random Effects Model of Non-Ignorable Nonresponse in Panel Survey Data. Available at SSRN \url{https://www.ssrn.com/abstract=5475626}
}
\seealso{
Other PanelSelect: 
\code{\link{probitRE_PLNRE}()},
\code{\link{probitRE_PoissonRE}()},
\code{\link{probitRE_linearRE}()}
}
\concept{PanelSelect}
