% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/LP_fit.R
\name{LP_fit}
\alias{LP_fit}
\title{Fit a Lincoln-Petersen Model using conditional likelihood}
\usage{
LP_fit(data, p_model = ~..time, p_beta.start = NULL, trace = FALSE)
}
\arguments{
\item{data}{Data frame containing the variables:
\itemize{
\item \strong{cap_hist} Capture history (see details below)
\item \strong{freq} Number of times this capture history was observed
}
plus any other covariates (e.g. discrete strata and/or continuous covariates) to be used
in the model fitting.}

\item{p_model}{Model for the captured probabilities. This can reference
other variables in the data frame, plus a special reserved term \code{..time} to indicate
a time dependence in the capture probabilities. For example, \code{p_model=~1} would indicate
that the capture probabilities are equal across the sampling events;
\code{p_model=~..time} would indicate that the capture probabilities vary by sampling events;
\code{p_model=~sex*..time} would indicate that the capture probabilities vary across
all combination of sampling events (\code{..time}) and a stratification variable (\code{sex}). The \code{sex} variable
also needs to be in the data frame.

For some models (e.g., tag loss models), the \code{..time} variable cannot be used because
the conditional models (on being captured at the second event) end up having
only have one capture probability (e.g., only for event 1) because of the conditioning process.

For the Bailey Binomial model, the \code{..time} variable cannot be used because
the conditional model ends up having only one capture probability (i.e., only for event 2).}

\item{p_beta.start}{Initial values for call to optimization routine for the beta parameters (on the logit scale).
The values will be replicated to match
the number of initial beta parameters needed. Some care is needed here!}

\item{trace}{If trace flag is set in call when estimating functions}
}
\value{
An list object of class \emph{LP_fit} with abundance estimates and other information with the following elements
\itemize{
\item \strong{summary} A data frame with the model for the capture probabilities;
the conditional log-likelihood; the number of parameters; and method used to fit the model
\item \strong{data} A data frame with the raw data used in the fit
\item \strong{fit} Results of the fit from the optimizer
\item \strong{datetime} Date and time the fit was done
}

After the fit is done, use the \emph{LP_est()} function to get estimates of abundance.
}
\description{
This will take a data frame of capture histories, frequencies, and
additional covariates (e.g., strata and/or continuous covariates) and the model
for the capture probabilities and will use conditional likelihood (Huggins, 1989)
to fit the model. The population abundance is estimated using
a Horvitz-Thompson type estimator and the user can request abundance
estimates for sub-sets of the population.
}
\details{
The frequency variable (\code{freq} in the \code{data} argument) is the number of animals with the corresponding capture history.

Capture histories (\code{cap_hist} in the \code{data} argument) are character values of length 2.
\itemize{
\item \strong{10}  Animals tagged but never seen again.
\item \strong{11}  Animals tagged and recaptured and tag present at event 2.
\item \strong{01}  Animals captured at event 2 that appear to be untagged.
}
}
\examples{
# fit a simple Petersen model and get the estimated abundance
data(data_rodli)
fit <- Petersen::LP_fit(data=data_rodli, p_model=~..time)
fit$summary
# Now to get the estimated abundance
est <- Petersen::LP_est(fit, N_hat=~1)
est$summary

# repeat the fit with the Chapman correction
# we add an additional animal with history 11
rodli.chapman <- plyr::rbind.fill(data_rodli,
                                  data.frame(cap_hist="11",
                                             freq=1,
                                             comment="Added for Chapman"))

rodli.chapman
fit.chapman <- Petersen::LP_fit(data=rodli.chapman, p_model=~..time)
fit.chapman$summary
# Now to get the estimated abundance
est.chapman <- Petersen::LP_est(fit.chapman, N_hat=~1)
est.chapman$summary


# Example of simple stratification (by sex)
data(data_NorthernPike)
nop.red <- plyr::ddply(data_NorthernPike, c("cap_hist","Sex"), plyr::summarize,
                       freq=sum(freq))
nop.red # reduced capture history to speed execution time of example

# Fit the various models
nop.fit.sex.time   <- Petersen::LP_fit(nop.red, p_model=~-1+Sex:..time)
nop.fit.sex.time$summary

# estimate of overall abundance
nop.est.ALL   <- Petersen::LP_est(nop.fit.sex.time, N=~1)
nop.est.ALL$summary

# estimate of abundance for each sex
nop.est.by.sex <- Petersen::LP_est(nop.fit.sex.time, N=~-1+Sex)
nop.est.by.sex$summary


# Refer to vignettes for example using continuous variable (e.g. length) to model catchability
}
\references{
Huggins, R. M. 1989. On the Statistical Analysis of Capture Experiments.
Biometrika 76: 133--40.
}
\author{
Schwarz, C. J. \email{cschwarz.stat.sfu.ca@gmail.com}.
}
