% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fastmatch.R
\name{fastmatch}
\alias{fastmatch}
\title{Create couples using a weighted age group structure}
\usage{
fastmatch(
  people,
  pplage,
  probSS = NULL,
  uwProp = NULL,
  uwLA = NULL,
  uwUA = NULL,
  HHStartNum = NULL,
  HHNumVar = NULL,
  userseed = NULL
)
}
\arguments{
\item{people}{A data frame containing individual people.}

\item{pplage}{The variable containing the ages.}

\item{probSS}{The probability of a person being in a same-sex couple.}

\item{uwProp}{The proportion of individuals who are to be over-sampled. By default, no age group is up-sampled, and people are selected based on simple random sampling, without replacement.}

\item{uwLA}{The youngest age for the over-sampling. Required if uwProp value is provided.}

\item{uwUA}{The oldest age for the over-sampling. Required if uwProp value is provided.}

\item{HHStartNum}{The starting value for HHNumVar Must be numeric.}

\item{HHNumVar}{The name for the household variable.}

\item{userseed}{If specified, this will set the seed to the number provided. If not, the normal set.seed() function will be used.}
}
\value{
A data frame of an even number of observations for allocation into same-sex couples. If HHStartNum is specified, household allocation will be performed.
}
\description{
Creates couples when the only information is the proportions of people in couples, by age group. If there is an age range that should be up-sampled compared to other ages, this can be specified using the uwProp, uwLA, and uwUA variables. If uwProp is not provided, a simple random sampling without replacement is used. The number of couples that are output is determined by probSS. At least one same-sex couple will be output.
}
\examples{

library(dplyr)

PersonDataframe <- data.frame(cbind(PersonID = c(1:1000),
                                    PersonAge = c(round(runif(200, min=18, max=23),0),
                                    round(runif(300, min=24, max=50),0),
                                    round(runif(500, min=51, max=90),0))))

# unweighted example, probability of being in a same-sex couple is 0.03
Unweighted <- fastmatch(PersonDataframe, pplage = "PersonAge", probSS = 0.03, HHStartNum = 1,
                        HHNumVar = "Household", userseed = 4)
NumUnweighted <- Unweighted \%>\%
  filter(between(PersonAge, 25, 54))
# prop is
nrow(NumUnweighted)/nrow(Unweighted)

# weighted example, same probability, 66\% of people in a same-sex relationship are aged between 25
# and 54
Weighted <- fastmatch(PersonDataframe, pplage = "PersonAge", probSS = 0.03, uwProp = .66,
                      uwLA = 25, uwUA = 54, HHStartNum = 1, HHNumVar = "Household", userseed = 4)
NumWeighted <- Weighted \%>\%
  filter(between(PersonAge, 25, 54))
# prop is
nrow(NumWeighted)/nrow(Weighted)
}
