% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prediction_model.v16.R
\name{pred.curve}
\alias{pred.curve}
\title{Epidemic Curve Model}
\usage{
pred.curve(
  a1 = 0,
  a2 = 0,
  a3 = 0,
  a4 = 0,
  nu = 0,
  variant.transm = NULL,
  Psi = NULL,
  betas = NULL,
  cases = NULL,
  cfg = NULL,
  use.actual.not.predicted = FALSE,
  restrictions = NULL,
  restriction.starts = NULL,
  ranges = NULL,
  rt_func = 1,
  fit.t.pred = NULL,
  predict.beyond = 0,
  scenario = NULL,
  H.E = NULL,
  H.W = NULL,
  adj.period = NULL,
  population = NULL,
  rho = NULL,
  serial_mean = NULL,
  serial_var = NULL,
  lt = NULL,
  window_size = NULL,
  verbose = FALSE
)
}
\arguments{
\item{a1, a2, a3, a4}{Parameters of the contact rate curve specified by \code{rt_func}.}

\item{nu}{Loss of immunity rate beyond the first wave.}

\item{variant.transm}{Vector of transmissibility of variants in each wave,
as relative multiplication factors compared to transmissibility in wave 1. Should always be 1 for the first wave.}

\item{Psi}{Vector of restriction parameters for severity levels 1 - 4.}

\item{betas}{Vector containing (beta0,beta.R,beta.E,beta.W), when restrictions = NULL. Not currently implemented.}

\item{cases}{vector of case counts.}

\item{cfg}{The object that contains all variables from the configuration file.
\code{a1}, \code{a2}, \code{a3}, \code{a4}, \code{nu}, \code{variant.transm},
\code{Psi}, \code{betas}, and \code{cases} are also required for the method
to execute. All other parameters will not be used if \code{cfg} is passed
to the method.}

\item{use.actual.not.predicted}{Logical; if FALSE (default), the susceptible fraction is updated using predicted cases. Otherwise updated using actual cases.}

\item{restrictions}{A numeric integer vector giving the severity of restrictions.
Zero means no restriction, and higher numbers means greater severity/disruption.
The ordered unique values should be consecutive integers starting from zero.
Each number (other than 0) adds a new parameter to the fit.}

\item{restriction.starts}{A vector of same length as restrictions, of times when restrictions
came into effect. Note: the first index time should be 1.}

\item{ranges}{A vector of time ranges for the different waves.
The wave ranges should be contiguous, with at least one unit of time
between consecutive waves.}

\item{rt_func}{The parametric form of function c(). Options are listed under function \code{c_helper}.}

\item{fit.t.pred}{Time from which prediction is done. If use.actual.not.predicted is TRUE, values of \eqn{S_t} before this time will be computed using actual counts.}

\item{predict.beyond}{Number of days to predict beyond the end of \code{cases}. See Details for usage notes.}

\item{scenario}{A character string describing options to deal with restrictions. Currently unsupported.}

\item{H.E}{Mobility metrics for category Retail & Entertainment. Currently unsupported.}

\item{H.W}{Mobility metrics for category Workplaces. Currently unsupported.}

\item{adj.period}{Adjustment period following a change in severity level. Restriction level (psi) is linearly interpolated from the old to the new value over this period.}

\item{population}{Total population size.}

\item{rho}{A vector of under-reporting rates of the same length as cases. If a scalar is supplied, the vector will be constant with this value.}

\item{serial_mean}{Mean of the serial interval on the log scale.}

\item{serial_var}{Variance of the serial interval on the log scale.}

\item{lt}{The length of cases.}

\item{window_size}{The maximum value for the serial interval.}

\item{verbose}{Logical. If TRUE, provides additional details while running the function.}
}
\value{
Returns list:
   \itemize{
     \item Predicted Infections - Vector of estimated infections, computed as predicted cases divided by rho.
     \item Predicted Cases - Vector of predicted cases.
     \item Predicted \eqn{R_t} - Vector of predicted susceptible fractions
     \item Predicted \eqn{R_t} - Vector of (model) predicted \eqn{R_t}.
     \item Predicted Lambda t - Vector of predicted Lambda_t, which is the numerator used in computing the empirical \eqn{R_t}.
     \item Psi.vec - Vector of psi values, which pastes together parameters psi over the period they apply, or 1 when there are no restrictions.
     \item Contact rate params - Vector of the curve parameters (a1, a2, a3, a4).
     }
}
\description{
Computes the epidemic curve and associated quantities for a given parameter set.
}
\details{
At each time step, \eqn{R_{t}} is computed using the contact rate function \eqn{c(S_t)} implemented via \link{c_helper}. Then the number of cases is estimated using formula:
    \deqn{y_{t+1}=R_{t} \sum_{s=1}^M w_s y_{t+1-s}}
    Finally, the fraction \eqn{S_{t+1}} is updated. This creates a curve over the entire range of \code{ranges}. See Romanescu R, Hu S, Nanton D, Torabi M, Tremblay-Savard O, Haque MA. The effective reproductive number:
    modeling and prediction with application to the multi-wave Covid-19 pandemic. Epidemics. 2023 Jul 20:100708 \doi{10.1016/j.epidem.2023.100708} for more details.

    For predicting an ongoing wave beyond the end of cases, the end of \code{ranges} (or \code{waves_list}, if using \code{cfg})
    should be specified to match the \code{predict.beyond} argument. As well, argument \code{use.actual.not.predicted} should be set to FALSE when predicting beyond the end of \code{cases}.
}
\examples{
library(REffectivePred)
## Read in the data
path_to_data <- system.file("extdata/NY_OCT_4_2022.csv", package = "REffectivePred")
data <- read.csv(path_to_data)
head(data)
cases <- diff(c(0, data$cases)) # Convert cumulative cases into daily cases
lt <- length(cases)             # Length of cases
Time <- as.Date(data$date, tryFormats = c("\%d-\%m-\%Y", "\%d/\%m/\%Y"))

navigate_to_config() # Open the config file, make any necessary changes here.
path_to_config <- system.file("config.yml", package = "REffectivePred")  # Read config file
cfg <- load_config()    # Build the cfg object

# Example 1. Using fits from Romanescu et al. (2023)

r1 <- pred.curve(
a1 = 0.58,
a2 = 1.12,
nu = 0.56,
variant.transm = c(1,1.22,0.36,0.56),
Psi = c(0.58,0.52,0.49),
cases = cases,
cfg = cfg
)

plot(cases, xlab="Day", ylab="Predicted cases")
lines(r1$'Predicted Cases', col='red')

# Example 2. Best fit curve
est <- estimate.mle(
    cases = cases,
    cfg = cfg
    )
a1 <- est$a1
a2 <- est$a2
a3 <- est$a3
a4 <- est$a4
nu <- est$nu
vt <- c(1, est$vt_params_est)
psi <- est$Psi
betas <- est$betas

r1 <- pred.curve(
a1 = a1,
a2 = a2,
a3 = a3,
a4 = a4,
nu = nu,
variant.transm = vt,
Psi = psi,
betas = betas,
cases = cases,
cfg = cfg
)
plot(r1$'Predicted Infections', xlab="Day", ylab="Predicted infections")

}
