% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/semLearn.R
\name{resizeGraph}
\alias{resizeGraph}
\title{Interactome-assisted graph re-seizing}
\usage{
resizeGraph(g = list(), gnet, d = 2, v = TRUE, verbose = FALSE, ...)
}
\arguments{
\item{g}{A list of two graphs as igraph objects, g=list(graph1, graph2).}

\item{gnet}{External directed network as an igraph object. The reference
network should have weighted edges, corresponding to their interaction
p-values, as an edge attribute \code{E(gnet)$pv}. Then, connections in
\code{graph2} will be checked by known connections from the reference network,
intercepted by the minimum-weighted shortest path found among the equivalent
ones by the Dijkstra algorithm, as implemented in the \pkg{igraph} function
\code{all_shortest_paths()}.}

\item{d}{An integer value indicating the maximum geodesic distance between
two nodes in the interactome to consider the inferred interaction between
the same two nodes in \code{graph2} as validated, otherwise the edges are
removed. For instance, if \code{d = 2}, two interacting nodes must either
share a direct interaction or being connected through at most one mediator
in the reference interactome (in general, at most \code{d - 1} mediators are
allowed). Typical \code{d} values include \code{2} (at most one mediator), or
\code{mean_distance(gnet)} (i.e., the average shortest path length for
the reference network). Setting d = 0, is equivalent to \code{gnet = NULL}.}

\item{v}{A logical value. If TRUE (default) new nodes and edges on the
validated shortest path in the reference interactome will be added in the
re-sized graph.}

\item{verbose}{A logical value. If FALSE (default), the processed graphs
will not be plotted to screen, saving execution time (for large graphs)}

\item{...}{Currently ignored.}
}
\value{
"Ug", the re-sized graph, the graph union of the causal graph \code{graph1}
and the re-sized graph \code{graph2}
}
\description{
An input directed graph is re-sized, removing edges
or adding edges/nodes. This function takes three input graphs: the
first is the input causal model (i.e., a directed graph), and the
second can be either a directed or undirected graph, providing a set
of connections to be checked against a directed reference network
(i.e., the third input) and imported to the first graph.
}
\details{
Typically, the first graph is an estimated causal graph (DAG),
and the second graph is the output of either \code{\link[SEMgraph]{SEMdag}}
or \code{\link[SEMgraph]{SEMbap}}. Alternatively, the first graph is an
empthy graph, and the second graph is a external covariance graph.
In the former we use the new inferred causal structure stored in the
\code{dag.new} object. In the latter, we use the new inferred covariance
structure stored in the \code{guu} object. Both directed (causal) edges
inferred by \code{SEMdag()} and covariances (i.e., bidirected edges)
added by \code{SEMbap()}, highlight emergent hidden topological
proprieties, absent in the input graph. Estimated directed edges between
nodes X and Y are interpreted as either direct links or direct paths
mediated by hidden connector nodes. Covariances between any two bow-free
nodes X and Y may hide causal relationships, not explicitly represented
in the current model. Conversely, directed edges could be redundant or
artifact, specific to the observed data and could be deleted.
Function \code{resizeGraph()} leverage on these concepts to extend/reduce a
causal model, importing new connectors or deleting estimated edges, if they are
present or absent in a given reference network. The whole process may lead to
the discovery of new paths of information flow, and cut edges not corroborate
by a validated network. Since added nodes can already be present in the causal
graph, network resize may create cross-connections between old and new paths
and their possible closure into circuits.
}
\examples{

\donttest{

# Extract the "Protein processing in endoplasmic reticulum" pathway:

g <- kegg.pathways[["Protein processing in endoplasmic reticulum"]]
G <- properties(g)[[1]]; summary(G)

# Reference network (KEGG interactome)
gnet <- kegg

# Extend a graph using new inferred DAG edges (dag+dag.new):

# Nonparanormal(npn) transformation
als.npn <- transformData(alsData$exprs)$data

dag <- SEMdag(graph = G, data = als.npn, beta = 0.1)
gplot(dag$dag)
ext <- resizeGraph(g=list(dag$dag, dag$dag.new), gnet = gnet, d = 2)
gplot(ext) 

# Create a directed graph from correlation matrix, using
# i) an empty graph as causal graph,
# ii) a covariance graph,
# iii) KEGG as reference:

corr2graph<- function(R, n, alpha=5e-6, ...)
{
	Z <- qnorm(alpha/2, lower.tail=FALSE)
thr <- (exp(2*Z/sqrt(n-3))-1)/(exp(2*Z/sqrt(n-3))+1)
	A <- ifelse(abs(R) > thr, 1, 0)
	diag(A) <- 0
	return(graph_from_adjacency_matrix(A, mode="undirected"))
}

v <- which(colnames(als.npn) \%in\% V(G)$name)
selectedData <- als.npn[, v]
G0 <- make_empty_graph(n = ncol(selectedData))
V(G0)$name <- colnames(selectedData)
G1 <- corr2graph(R = cor(selectedData), n= nrow(selectedData))
ext <- resizeGraph(g=list(G0, G1), gnet = gnet, d = 2, v = TRUE)

#Graphs
old.par <- par(no.readonly = TRUE)
par(mfrow=c(1,2), mar=rep(1,4))
plot(G1, layout = layout.circle)
plot(ext, layout = layout.circle)
par(old.par)
}


}
\references{
Grassi M, Palluzzi F, Tarantino B (2022). SEMgraph: An R Package for Causal Network
Analysis of High-Throughput Data with Structural Equation Models.
Bioinformatics, 38 (20), 4829–4830 <https://doi.org/10.1093/bioinformatics/btac567>
}
\author{
Mario Grassi \email{mario.grassi@unipv.it}
}
