
#' Estimates LFDR values per genomic site
#'
#' Based on a given read count matrix, identifies single nucleotide variants (SNVs) by estimating local false discovery rates (LFDRs). Users can set an initial value for the proportion of non-mutant sites and specify thresholds for allele frequency, read depth and LFDR cut-off value.
#' @examples
#' bam_input <- system.file("extdata", "bam_input.csv", package="SNVLFDR")
#' bedfile <- system.file("extdata", "regions.bed", package="SNVLFDR")
#' BQ.T=20
#' MQ.T=20
#' pi0.initial=0.95
#' AF.T=0.01
#' DP.T=10
#' LFDR.T=0.01
#' error=NULL
#' method='empirical'
#' epsilon=0.01
#' output=get_LFDRs(bam_input,bedfile,BQ.T,MQ.T,pi0.initial,AF.T,DP.T,LFDR.T,error,method,epsilon)
#'
#' @param bam_input Path to an input BAM file. The file must be in the format of a csv file generated by bam-readcount (https://github.com/genome/bam-readcount). See Examples.
#' @param bedfile Path to a bed file containing genomic regions of interest. This file has to have three columns with chr# in the first column and start and end positions in the second and third columns respectively. No headers.
#' @param BQ.T Minimum base call quality threshold. It can be set to 0 to include all sites. Otherwise, this threshold excludes sites with  a base call quality below the specified threshold. It is recomended to set it to 20.
#' @param MQ.T Minimum mapping  quality threshold. It can be set to 0 to include all sites. Otherwise, this threshold excludes sites with a mapping quality below the specified threshold. It is recomended to set it to 20.
#' @param pi0.initial Initial value for the proportion of non-mutant sites. It can be any number between 0 and 1. However it is recommended to set it to a number between 0.95 and 0.99 for more accuracy. If no value is specified, it will be set to 0.95 by default.
#' @param AF.T Allele frequency threshold. It can be set to 0 to include all sites. Otherwise, this threshold excludes sites with an allele frequency below the specified threshold.  If no value is specified, it will be set to 0.01 by default.
#' @param DP.T Read depth threshold. It can be set to 0 to include all sites. Otherwise, this threshold excludes sites with a read depth below the specified threshold.
#' @param LFDR.T A number between 0 and 1. It can be set to 0 to include all sites. Otherwise, this threshold excludes sites with an estimated LFDR below the specified threshold.  If no value is specified, it will be set to 0.01 by default.
#' @param error Error rate between 0 and 1. If it is set to NULL, a weighted average of average base call quality and average mapping quality per site will be calculated. Otherwise, it may be set to 0.01 or a desired error vector can be introduced by the user.
#' @param method Method used to estimate pi0 and LFDRs. It can be "empirical", "uniform_empirical" or "uniform". If no method is specified, it will be set to "empirical" by default (recommended).
#' @param epsilon The difference between old and new estimates of pi0 used for convergence. If no value is specified, it will be set to 0.01 by default.
#' @return A list. Slot \strong{estimated.pi0} returns estimated proportion of non-mutant sites. Slot \strong{estimated.LFDRs} returns estimated LFDRs for genomic sites that were not filtered out.
#' Slot \strong{filtered.bam} adds estimated LFDRs, model errors and a mutant variable (indicating whether each site is detected to be a mutant (1) or non-mutant (0) site) to the filtered input file .
#'@import stats
#'@import utils
#'@references Karimnezhad, A. and Perkins, T.J. (2023). Empirical Bayes Single Nucleotide Variant Calling For Next-Generation Sequencing Data. Working Paper. <https://mysite.science.uottawa.ca/akarimne/wp-content/uploads/2023/12/AK-TJP-SR.pdf>
#'
#' @export
get_LFDRs<-function(bam_input,bedfile,BQ.T,MQ.T,pi0.initial,AF.T,DP.T,LFDR.T,error,method,epsilon){
  if (is.null(pi0.initial)){pi0.initial=0.95}
  if (is.null(AF.T)){AF.T=0.01}
  if (is.null(DP.T)){DP.T=10}
  if (is.null(LFDR.T)){LFDR.T=0.01}
  if (is.null(method)){method='empirical'}
  if (is.null(method)){epsilon=0.01}
  if (is.null(BQ.T)){epsilon=20}
  if (is.null(MQ.T)){epsilon=20}
  A<-utils::read.csv(bam_input,sep="\t",quote = "\"",header=F,stringsAsFactors = F)
  Prepare_LFDR_input<-function(Input){
    index<-c()
    Allele1<-Ct1<-Ave.Map.Q1<-Ave.Base.Q1<-Ct.Plus.Str.1<-Ct.Mns.Str.1<-c()
    Allele2<-Ct2<-Ave.Map.Q2<-Ave.Base.Q2<-Ct.Plus.Str.2<-Ct.Mns.Str.2<-c()
    Allele3<-Ct3<-Ave.Map.Q3<-Ave.Base.Q3<-Ct.Plus.Str.3<-Ct.Mns.Str.3<-c()
    Allele4<-Ct4<-Ave.Map.Q4<-Ave.Base.Q4<-Ct.Plus.Str.4<-Ct.Mns.Str.4<-c()
    Allele5<-Ct5<-Ave.Map.Q5<-Ave.Base.Q5<-Ct.Plus.Str.5<-Ct.Mns.Str.5<-c()
    for ( i in 1:dim(Input)[1]){
      # for ( i in 1:10000){
      if (nchar(as.character(Input[i,10]))>0){
        Allele1<-c(Allele1,unlist(strsplit(Input[i,6], ":"))[1])
        Ct1<-c(Ct1,unlist(strsplit(Input[i,6], ":"))[2])
        Ave.Map.Q1<-c(Ave.Map.Q1,unlist(strsplit(Input[i,6], ":"))[3])
        Ave.Base.Q1<-c(Ave.Base.Q1,unlist(strsplit(Input[i,6], ":"))[4])
        Ct.Plus.Str.1<-c(Ct.Plus.Str.1,unlist(strsplit(Input[i,6], ":"))[6])
        Ct.Mns.Str.1<-c(Ct.Mns.Str.1,unlist(strsplit(Input[i,6], ":"))[7])
        Allele2<-c(Allele2,unlist(strsplit(Input[i,7], ":"))[1])
        Ct2<-c(Ct2,unlist(strsplit(Input[i,7], ":"))[2])
        Ave.Map.Q2<-c(Ave.Map.Q2,unlist(strsplit(Input[i,7], ":"))[3])
        Ave.Base.Q2<-c(Ave.Base.Q2,unlist(strsplit(Input[i,7], ":"))[4])
        Ct.Plus.Str.2<-c(Ct.Plus.Str.2,unlist(strsplit(Input[i,7], ":"))[6])
        Ct.Mns.Str.2<-c(Ct.Mns.Str.2,unlist(strsplit(Input[i,7], ":"))[7])
        Allele3<-c(Allele3,unlist(strsplit(Input[i,8], ":"))[1])
        Ct3<-c(Ct3,unlist(strsplit(Input[i,8], ":"))[2])
        Ave.Map.Q3<-c(Ave.Map.Q3,unlist(strsplit(Input[i,8], ":"))[3])
        Ave.Base.Q3<-c(Ave.Base.Q3,unlist(strsplit(Input[i,8], ":"))[4])
        Ct.Plus.Str.3<-c(Ct.Plus.Str.3,unlist(strsplit(Input[i,8], ":"))[6])
        Ct.Mns.Str.3<-c(Ct.Mns.Str.3,unlist(strsplit(Input[i,8], ":"))[7])
        Allele4<-c(Allele4,unlist(strsplit(Input[i,9], ":"))[1])
        Ct4<-c(Ct4,unlist(strsplit(Input[i,9], ":"))[2])
        Ave.Map.Q4<-c(Ave.Map.Q4,unlist(strsplit(Input[i,9], ":"))[3])
        Ave.Base.Q4<-c(Ave.Base.Q4,unlist(strsplit(Input[i,9], ":"))[4])
        Ct.Plus.Str.4<-c(Ct.Plus.Str.4,unlist(strsplit(Input[i,9], ":"))[6])
        Ct.Mns.Str.4<-c(Ct.Mns.Str.4,unlist(strsplit(Input[i,9], ":"))[7])
        Allele5<-c(Allele5,unlist(strsplit(Input[i,10], ":"))[1])
        Ct5<-c(Ct5,unlist(strsplit(Input[i,10], ":"))[2])
        Ave.Map.Q5<-c(Ave.Map.Q5,unlist(strsplit(Input[i,10], ":"))[3])
        Ave.Base.Q5<-c(Ave.Base.Q5,unlist(strsplit(Input[i,10], ":"))[4])
        Ct.Plus.Str.5<-c(Ct.Plus.Str.5,unlist(strsplit(Input[i,10], ":"))[6])
        Ct.Mns.Str.5<-c(Ct.Mns.Str.5,unlist(strsplit(Input[i,10], ":"))[7])
        index<-c(index,i)
      }
    }
    Ideal.Mx<-cbind(Input[index,1:4],
                    as.character(Allele1),as.numeric(as.character(Ct1)),as.numeric(as.character(Ave.Map.Q1)),
                    as.numeric(as.character(Ave.Base.Q1)),as.numeric(as.character(Ct.Plus.Str.1)),
                    as.numeric(as.character(Ct.Mns.Str.1)),
                    Allele2,as.numeric(as.character(Ct2)),as.numeric(as.character(Ave.Map.Q2)),
                    as.numeric(as.character(Ave.Base.Q2)),as.numeric(as.character(Ct.Plus.Str.2)),
                    as.numeric(as.character(Ct.Mns.Str.2)),
                    Allele3,as.numeric(as.character(Ct3)),as.numeric(as.character(Ave.Map.Q3)),
                    as.numeric(as.character(Ave.Base.Q3)),as.numeric(as.character(Ct.Plus.Str.3)),
                    as.numeric(as.character(Ct.Mns.Str.3)),
                    Allele4,as.numeric(as.character(Ct4)),as.numeric(as.character(Ave.Map.Q4)),
                    as.numeric(as.character(Ave.Base.Q4)),as.numeric(as.character(Ct.Plus.Str.4)),
                    as.numeric(as.character(Ct.Mns.Str.4)),
                    Allele5,as.numeric(as.character(Ct5)),as.numeric(as.character(Ave.Map.Q5)),
                    as.numeric(as.character(Ave.Base.Q5)),as.numeric(as.character(Ct.Plus.Str.5)),
                    as.numeric(as.character(Ct.Mns.Str.5)))
    J=droplevels(Ideal.Mx)
    colnames(Ideal.Mx)<-c('CHR','POS','Ref','Total',
                          'Allele1','Ct1','Ave.Map.Q1','Ave.Base.Q1','Ct.Plus.Str.1','Ct.Mns.Str.1',
                          'Allele2','Ct2','Ave.Map.Q2','Ave.Base.Q2','Ct.Plus.Str.2','Ct.Mns.Str.2',
                          'Allele3','Ct3','Ave.Map.Q3','Ave.Base.Q3','Ct.Plus.Str.3','Ct.Mns.Str.3',
                          'Allele4','Ct4','Ave.Map.Q4','Ave.Base.Q4','Ct.Plus.Str.4','Ct.Mns.Str.4',
                          'Allele5','Ct5','Ave.Map.Q5','Ave.Base.Q5','Ct.Plus.Str.5','Ct.Mns.Str.5' )
    I<-Ideal.Mx
    I[which(!as.character(I[,3])%in%c('A','C','G','T')),'Ref']<-toupper(as.character(I[which(!as.character(I[,3])%in%c('A','C','G','T')),3]))
    I$Total<-as.numeric(as.character(I$Ct1))+as.numeric(as.character(I$Ct2))+as.numeric(as.character(I$Ct3))+as.numeric(as.character(I$Ct4))
    x<-c()
    ideal.col<-c(5,11,17,23)
    for ( i in 1:dim(I)[1]){
      #  for ( i in 1:10){
      #print(i)
      j.Ref<-j.nonRef<-c()
      #for ( i in 1:10){
      for ( j in ideal.col){
        if (I[i,j]==I[i,3]){
          j.Ref<-j}
        else{
          j.nonRef<-ideal.col[!ideal.col%in%j.Ref]
        }
      }
      L.Ref<-cbind(as.character(I[i,c(j.Ref)]),I[i,c(j.Ref+1)],I[i,c(j.Ref+2)],I[i,c(j.Ref+3)],I[i,c(j.Ref+4)],I[i,c(j.Ref+5)])
      Alt.counts=c(I[i,c(j.nonRef[1]+1)],I[i,c(j.nonRef[2]+1)],I[i,c(j.nonRef[3]+1)])
      w01<-which(Alt.counts==max(Alt.counts))
      w02<-which(Alt.counts!=max(Alt.counts))
      if (length(w01)==1){
        w_alt=w01
        w_alts=w02
      } else if (length(w01)==2){
        w_alt=w01[1]
        w_alts<-c(w01[2],w02)
      } else if (length(w01)==3){
        w_alt=w01[1]
        w_alts<-c(w01[2],w01[3])
      }
      L.Alt<-c(as.character(I[i,j.nonRef[w_alt]]),I[i,j.nonRef[w_alt]+1],I[i,j.nonRef[w_alt]+2],I[i,j.nonRef[w_alt]+3],I[i,j.nonRef[w_alt]+4],I[i,j.nonRef[w_alt]+5])
      L.Alt1<-c(as.character(I[i,j.nonRef[w_alts[1]]]),I[i,j.nonRef[w_alts[1]]+1],I[i,j.nonRef[w_alts[1]]+2],I[i,j.nonRef[w_alts[1]]+3],I[i,j.nonRef[w_alts[1]]+4],I[i,j.nonRef[w_alts[1]]+5])
      L.Alt2<-c(as.character(I[i,j.nonRef[w_alts[2]]]),I[i,j.nonRef[w_alts[2]]+1],I[i,j.nonRef[w_alts[2]]+2],I[i,j.nonRef[w_alts[2]]+3],I[i,j.nonRef[w_alts[2]]+4],I[i,j.nonRef[w_alts[2]]+5])

      x<-rbind(x,c(I[i,1],I[i,2],L.Ref,L.Alt,L.Alt1,L.Alt2))
    }
    #x<-cbind(x,I$Total)
    colnames(x)<-c('CHR','POS','Ref','Ct.Ref','Ave.Map.Q.Ref','Ave.Base.Q.Ref','Ct.Plus.Ref.Str.1','Ct.Plus.Ref.Str.2',
                   'Alt','Ct.Alt','Ave.Map.Q.Alt','Ave.Base.Q.Alt','Ct.Plus.Alt.Str.1','Ct.Plus.Alt.Str.2',
                   'Alt1','Ct.Alt1','Ave.Map.Q.Alt1','Ave.Base.Q.Alt1','Ct.Plus.Alt1.Str.1','Ct.Plus.Alt1.Str.2',
                   'Alt2','Ct.Alt2','Ave.Map.Q.Alt2','Ave.Base.Q.Alt2','Ct.Plus.Alt2.Str.1','Ct.Plus.Alt2.Str.2')
    Depth<-as.numeric(as.character(x[,'Ct.Ref']))+as.numeric(as.character(x[,'Ct.Alt']))+as.numeric(as.character(x[,'Ct.Alt1']))+as.numeric(as.character(x[,'Ct.Alt2']))
    #1st definition
    #AF<-as.numeric(x[,'Ct.Alt'])/Depth
    #2nd definition
    AF<-as.numeric(x[,'Ct.Alt'])/(as.numeric(x[,'Ct.Alt'])+as.numeric(x[,'Ct.Ref']))
    x<-cbind(x,Depth,AF)
    #x<-noquote(x)
    #x[1:8,]
    C<-data.frame(x)
    D<-noquote(C)
    x.e<-D
    x.e[3,]
    dim(x.e)
    return(x.e)
  }
  filter_input<-function(A,bedfile,BQ.T,MQ.T){
    A<-A[A$V2!="",]
    ave_b_q_alt=BQ.T
    ave_b_q_ref=BQ.T
    ave_m_q_alt=MQ.T
    ave_m_q_ref=MQ.T
    if (is.null(bedfile)){
      D<-Prepare_LFDR_input(Input=A)
    } else{
      Manifest.DNA<-utils::read.table(bedfile,sep="\t",header=F,stringsAsFactors=F)
      colnames(Manifest.DNA)<-c('chr','start','end')
      w<-grep('chr',A$V1)
      u=unique(A$V1[w])
      B<-A
      C<-c()
      D<-c()
      for ( l in u){
        M0<-Manifest.DNA[Manifest.DNA$chr==l,]
        A0=B[B$V1==l,]
        if (dim(M0)[1]>0){
          for ( j in 1 :dim(M0)[1]){
            Cj = A0[A0$V2 >= M0$start[j] & A0$V2 <= M0$end[j],]
            C<-rbind(C,Cj)
            if (dim(Cj)[1]>1){
              Dj<-Prepare_LFDR_input(Input=Cj)
              D<-rbind(D,Dj)
            }
          }
        }
        #print(l)
      }
    }
    D_noduplicated<-D[!duplicated(D),]
    filter_BAM_bq<-function(Input_BAM,ave_b_q_alt,ave_b_q_ref,ave_m_q_alt,ave_m_q_ref){
      x.e<-Input_BAM
      x.e<-data.frame(x.e)
      x.e_2<-x.e[as.numeric(as.character(x.e$Ave.Base.Q.Alt))>=ave_b_q_alt &as.numeric(as.character(x.e$Ave.Base.Q.Ref))>=ave_b_q_ref&as.numeric(as.character(x.e$Ave.Map.Q.Alt))>=ave_m_q_alt &as.numeric(as.character(x.e$Ave.Map.Q.Ref))>=ave_m_q_ref ,]
      x.e_0<-x.e[as.numeric(as.character(x.e$Ave.Map.Q.Ref))==0 & as.numeric(as.character(x.e$Ct.Ref))>0 & as.numeric(as.character(x.e$Ct.Alt))>0 &as.numeric(as.character(x.e$Ave.Base.Q.Alt))>=ave_b_q_alt &as.numeric(as.character(x.e$Ave.Map.Q.Alt))>=ave_m_q_alt,]
      x.e<-rbind(x.e_2,x.e_0)
      return(x.e)
    }
    filtered_BAM_bq<-filter_BAM_bq(Input_BAM=D_noduplicated,ave_b_q_alt,ave_b_q_ref,ave_m_q_alt,ave_m_q_ref) #filter calls based on average base call quality/mapping quality
    return(filtered_BAM_bq)
  }
  get.f1<-function(x,theta,what.error){
    R<-as.numeric(as.character(x$Ct.Ref))
    M<-as.numeric(as.character(x$Ct.Alt))
    X1<-as.numeric(as.character(x$Ct.Alt1))
    X2<-as.numeric(as.character(x$Ct.Alt2))
    x<-cbind(R,M,X1,X2)
    error=what.error
    f1.all<-sapply(1:dim(x)[1], function (j) {mean(sapply(1:length(theta), function(k) {stats::dmultinom(x[j,], size = NULL,prob=c(theta[k]*(error[j]/3)+(1-theta[k])*(1-error[j]),theta[k]*(1-error[j])+(1-theta[k])*error[j]/3,error[j]/3,error[j]/3),log = FALSE)}))})
    return(f1.all)
  }
  get.f0<-function(x,what.error){
    R<-as.numeric(as.character(x$Ct.Ref))
    M<-as.numeric(as.character(x$Ct.Alt))
    X1<-as.numeric(as.character(x$Ct.Alt1))
    X2<-as.numeric(as.character(x$Ct.Alt2))
    x<-cbind(R,M,X1,X2)
    probs_0=cbind(1-what.error,what.error/3,what.error/3,what.error/3)
    f0.all<-sapply(1:dim(x)[1], function(j) {stats::dmultinom(x[j,], size = NULL, prob=probs_0[j,], log = FALSE)})
    return(f0.all)
  }
  x<-filter_input(A,bedfile,BQ.T,MQ.T)
  if (is.null(error)){
    M_BQ1=as.numeric(as.character(x$Ave.Base.Q.Alt))
    M_BQ2=as.numeric(as.character(x$Ave.Map.Q.Alt))
    M_C=as.numeric(as.character(x$Ct.Alt))
    Alt1_BQ1=as.numeric(as.character(x$Ave.Base.Q.Alt1))
    Alt1_BQ2=as.numeric(as.character(x$Ave.Map.Q.Alt1))
    Alt1_C=as.numeric(as.character(x$Ct.Alt1))
    Alt2_BQ1=as.numeric(as.character(x$Ave.Base.Q.Alt2))
    Alt2_BQ2=as.numeric(as.character(x$Ave.Map.Q.Alt2))
    Alt2_C=as.numeric(as.character(x$Ct.Alt2))
    R_BQ1=as.numeric(as.character(x$Ave.Base.Q.Ref))
    R_BQ2=as.numeric(as.character(x$Ave.Map.Q.Ref))
    R_C=as.numeric(as.character(x$Ct.Ref))
    M_BQ=(M_BQ1+M_BQ2)/2
    Alt1_BQ=(Alt1_BQ1+Alt1_BQ2)/2
    Alt2_BQ=(Alt2_BQ1+Alt2_BQ2)/2
    R_BQ=(R_BQ1+R_BQ2)/2
    error=10^(-((M_BQ*M_C + Alt1_BQ*Alt1_C  + Alt2_BQ*Alt2_C  + R_BQ*R_C)/(M_C+Alt1_C+Alt2_C+R_C))/10)
  }
  w<-which(as.numeric(as.character(x$AF))>=AF.T & as.numeric(as.character(x$Depth))>=DP.T)
  if (length(w)>0){
    x.filtered<-x[w,]
    w0<-dim(x)[1]-length(w)
  } else {
    #warning('With the current AF.T and/or DP.T thresholds no site was filtered out. If this is not satisfactory, consider changing the thresholds.')
    x.filtered<-x
    w0=0
  }
  message(paste("The input bam file has ",dim(A)[1]," sites. However, ",dim(A)[1]-dim(x.filtered)[1]," sites were excluded due to the pre-specfied thresholds and/or the intervals specifed in the bam file",sep=''))
  R<-as.numeric(as.character(x.filtered$Ct.Ref))
  M<-as.numeric(as.character(x.filtered$Ct.Alt))
  X1<-as.numeric(as.character(x.filtered$Ct.Alt1))
  X2<-as.numeric(as.character(x.filtered$Ct.Alt2))
  #theta.initial=x.filtered[x.filtered$Ct.Alt!=0,]$AF
  if (dim(x.filtered)[1]>10^4){
    theta.initial<-sample(size=10^3,as.numeric(as.character(x.filtered$AF)))
  } else{
    theta.initial<-as.numeric(as.character(x.filtered$AF))
  }
  f0<-get.f0(x =x.filtered ,what.error = error )
  Q<-x.filtered
  if (method=='empirical'){
    f1.empirical<-get.f1(x =x.filtered ,theta =theta.initial,what.error = error )
    O.LFDRs.empirical<-(pi0.initial*f0)/(pi0.initial*f0+(1-pi0.initial)*f1.empirical)
    O.LFDRs.empirical[is.na(O.LFDRs.empirical)]<-0
    Q.empirical=cbind(Q,O.LFDRs.empirical)
    w_l.empirical=which(colnames(Q.empirical)=='O.LFDRs.empirical')
    N.Mutatons.empirical.all<-pi0.empirical.all<-LFDR.empirical.all<-c()
    Q0.empirical<-Q.empirical
    pi0.empirical.1=2
    pi0.empirical<-c(pi0.empirical.1,pi0.initial)
    d_pi0.empirical<-length(pi0.empirical)
    dQ0.empirical<-dim(Q0.empirical)[1]
    N.Mutatons.empirical<-sum(Q0.empirical[, w_l.empirical]<LFDR.T)
    message("empirical estimation of g(.)")
    i=0
    while(abs(pi0.empirical[d_pi0.empirical]-pi0.empirical[d_pi0.empirical-1])>epsilon){
     #pi0.empirical.2=(sum(Q0.empirical[,w_l.empirical+i]>LFDR.T,na.rm = T)+w0)/(dQ0.empirical+w0)
      pi0.empirical.2=(sum(Q0.empirical[,w_l.empirical+i]>LFDR.T,na.rm = T))/(dQ0.empirical)
      pi0.empirical<-c(pi0.empirical,pi0.empirical.2)
      d_pi0.empirical<-length(pi0.empirical)
      Qs.empirical<-Q0.empirical[Q0.empirical[,w_l.empirical+i]<=LFDR.T,]
      theta.empirical.1=as.numeric(as.character(Qs.empirical$AF))
      f1.empirical.updated<-get.f1(x.filtered,theta=theta.empirical.1,what.error=error)
      L2.empirical<-(pi0.empirical.2*f0)/(pi0.empirical.2*f0+(1-pi0.empirical.2)*f1.empirical.updated)
      L2.empirical[is.na(L2.empirical)]<-0
      Q0.empirical=cbind(Q0.empirical,L2.empirical)
      i=i+1
      #print(paste('iteration: ',i,sep=''))
      N.Mutatons.empirical<-c(N.Mutatons.empirical,sum(Q0.empirical[,w_l.empirical+i]<LFDR.T))
    }
    message(paste0("converged at iteration: ",i))
    N.Mutatons=utils::tail(N.Mutatons.empirical,n=1)
    pi0.empirical=utils::tail(pi0.empirical,n=1)
    LFDRs=L2.empirical
    Mutant=ifelse(LFDRs>LFDR.T,0,1)
    output=cbind(x.filtered,LFDRs,Mutant)
    return(list('estimated.pi0'=pi0.empirical,'estimated.LFDRs'=LFDRs,'filtered.bam'=output))
  } else if (method=='uniform_empirical'){
    theta.initial.uniform =stats::runif(1000,0,1)
    f1.uniform<-get.f1(x =x.filtered ,theta =theta.initial.uniform,what.error = error )
    O.LFDRs.uniform<-(pi0.initial*f0)/(pi0.initial*f0+(1-pi0.initial)*f1.uniform)
    O.LFDRs.uniform[is.na(O.LFDRs.uniform)]<-0
    Q.uniform=cbind(Q,O.LFDRs.uniform)
    w_l.uniform=which(colnames(Q.uniform)=='O.LFDRs.uniform')
    N.Mutatons.uniform.all<-pi0.uniform.all<-LFDR.uniform.all<-c()
    Q0.uniform<-Q.uniform
    pi0.uniform.1=2
    pi0.uniform<-c(pi0.uniform.1,pi0.initial)
    d_pi0.uniform<-length(pi0.uniform)
    dQ0.uniform<-dim(Q0.uniform)[1]
    N.Mutatons.uniform<-sum(Q0.uniform[, w_l.uniform]<LFDR.T)

    message("uniform and then empirical estimation of g(.)")
    j=0
    while(abs(pi0.uniform[d_pi0.uniform]-pi0.uniform[d_pi0.uniform-1])>epsilon){
    #pi0.uniform.2=(sum(Q0.uniform[,w_l.uniform+j]>LFDR.T,na.rm = T)+w0)/(dQ0.uniform+w0)
      pi0.uniform.2=(sum(Q0.uniform[,w_l.uniform+j]>LFDR.T,na.rm = T))/(dQ0.uniform)
      pi0.uniform<-c(pi0.uniform,pi0.uniform.2)

      d_pi0.uniform<-length(pi0.uniform)
      Qs.uniform<-Q0.uniform[Q0.uniform[,w_l.uniform+j]<=LFDR.T,]
      theta.uniform.1=as.numeric(as.character(Qs.uniform$AF))
      f1.uniform.updated<-get.f1(x.filtered,theta=theta.uniform.1,what.error=error)
      L2.uniform<-(pi0.uniform.2*f0)/(pi0.uniform.2*f0+(1-pi0.uniform.2)*f1.uniform.updated)
      L2.uniform[is.na(L2.uniform)]<-0
      Q0.uniform=cbind(Q0.uniform,L2.uniform)
      j=j+1
      #print(paste('iteration: ',j,sep=''))
      N.Mutatons.uniform<-c(N.Mutatons.uniform,sum(Q0.uniform[,w_l.uniform+j]<LFDR.T))
    }
    message(paste0("converged at iteration: ",j))
    N.Mutatons=utils::tail(N.Mutatons.uniform,n=1)
    pi0.uniform=utils::tail(pi0.uniform,n=1)
    LFDRs=L2.uniform
    Mutant=ifelse(LFDRs>LFDR.T,0,1)
    output=cbind(x.filtered,LFDRs,Mutant)
    return(list('estimated.pi0'=pi0.uniform,'estimated.LFDRs'=LFDRs,'filtered.bam'=output))
  } else if (method=='uniform'){
    theta.initial.uniform =stats::runif(1000,0,1)
    f1.uniform<-get.f1(x =x.filtered ,theta =theta.initial.uniform,what.error = error )
    O.LFDRs.uniform.nonconvergent<-(pi0.initial*f0)/(pi0.initial*f0+(1-pi0.initial)*f1.uniform)
    O.LFDRs.uniform.nonconvergent[is.na(O.LFDRs.uniform.nonconvergent)]<-0
    Q.uniform.nonconvergent=cbind(Q,O.LFDRs.uniform.nonconvergent)
    w_l.uniform.nonconvergent=which(colnames(Q.uniform.nonconvergent)=='O.LFDRs.uniform.nonconvergent')
    N.Mutatons.uniform.nonconvergent.all<-pi0.uniform.nonconvergent.all<-LFDR.uniform.nonconvergent.all<-c()
    Q0.uniform.nonconvergent<-cbind(Q,O.LFDRs.uniform.nonconvergent)

    pi0.uniform.nonconvergent.1=2
    pi0.uniform.nonconvergent<-c(pi0.uniform.nonconvergent.1,pi0.initial)
    d_pi0.uniform.nonconvergent<-length(pi0.uniform.nonconvergent)
    dQ0.uniform.nonconvergent<-dim(Q0.uniform.nonconvergent)[1]
    N.Mutatons.uniform.nonconvergent<-sum(Q0.uniform.nonconvergent[, w_l.uniform.nonconvergent]<LFDR.T)
    message("uniform estimation of g(.)")
    l=0
    while(abs(pi0.uniform.nonconvergent[d_pi0.uniform.nonconvergent]-pi0.uniform.nonconvergent[d_pi0.uniform.nonconvergent-1])>epsilon){
     #pi0.uniform.nonconvergent.2=(sum(Q0.uniform.nonconvergent[,w_l.uniform.nonconvergent+l]>LFDR.T,na.rm = T)+w0)/(dQ0.uniform.nonconvergent+w0)
      pi0.uniform.nonconvergent.2=(sum(Q0.uniform.nonconvergent[,w_l.uniform.nonconvergent+l]>LFDR.T,na.rm = T))/(dQ0.uniform.nonconvergent)
      pi0.uniform.nonconvergent<-c(pi0.uniform.nonconvergent,pi0.uniform.nonconvergent.2)

      d_pi0.uniform.nonconvergent<-length(pi0.uniform.nonconvergent)
      Qs.uniform.nonconvergent<-Q0.uniform.nonconvergent[Q0.uniform.nonconvergent[,w_l.uniform.nonconvergent+l]<=LFDR.T,]
      L2.uniform.nonconvergent<-(pi0.uniform.nonconvergent.2*f0)/(pi0.uniform.nonconvergent.2*f0+(1-pi0.uniform.nonconvergent.2)*f1.uniform)
      L2.uniform.nonconvergent[is.na(L2.uniform.nonconvergent)]<-0
      Q0.uniform.nonconvergent=cbind(Q0.uniform.nonconvergent,L2.uniform.nonconvergent)
      l=l+1
      #print(paste('iteration: ',l,sep=''))
      N.Mutatons.uniform.nonconvergent<-c(N.Mutatons.uniform.nonconvergent,sum(Q0.uniform.nonconvergent[,w_l.uniform.nonconvergent+l]<LFDR.T))
    }
    message(paste0("converged at iteration: ",l))
    N.Mutatons=utils::tail(N.Mutatons.uniform.nonconvergent,n=1)
    pi0.uniform.nonconvergent=utils::tail(pi0.uniform.nonconvergent,n=1)
    LFDRs=L2.uniform.nonconvergent
    Mutant=ifelse(LFDRs>LFDR.T,0,1)
    output=cbind(x.filtered,LFDRs,Mutant)
    return(list('estimated.pi0'=pi0.uniform.nonconvergent,'estimated.LFDRs'=LFDRs,'filtered.bam'=output))
  }
}


#' Estimates LFDR values per mutant site detected by a desired variant caller
#'
#' Based on a given read count matrix and a list of calls made by a desired variant caller, estimates LFDRs that corresponds to each genomic site. It further
#' classifies sites to either mutant or non-mutant sites by comparing their estimated LFDRs with an LFDR cut-off value. The cut-off value as well as error rates can be defined by users.
#' @examples
#' bam_path <- system.file("extdata", "bam_input.csv", package="SNVLFDR")
#' calls_path <- system.file("extdata", "calls.vcf", package="SNVLFDR")
#' output=get_LFDRs_given_caller(bam_input=bam_path,calls=calls_path,LFDR.T=0.01,error=NULL)
#'
#' @param bam_input Path to an original BAM file used to call variants. The file must be in the format of a csv file generated by bam-readcount (https://github.com/genome/bam-readcount). See Examples.
#' @param calls Path to a vcf file generated by a variant caller. The first and second columns of this file have to be CHR names and positions, respectively.
#' @param LFDR.T A number between 0 and 1. It can be set to 0 to include all sites. Otherwise, this threshold excludes sites with an LFDR below the specified threshold.  If no value is specified, it will be set to 0.01 by default.
#' @param error Error rate between 0 and 1. If it is set to NULL, a weighted average of average base call quality and average mapping quality per site will be calculated. Otherwise, it may be set to 0.01 or a desired error vector can be introduced by the user.
#' @return A list. Slot \strong{estimated.LFDRs} returns estimated LFDRs for all sites in the input file.
#' Slot \strong{updated.vcf} outputs estimated LFDRs, model errors and a mutant variable (indicating whether each site is detected to be a mutant (1) or non-mutant (0) site) for those variants already detected by a desired variant caller.
#'@references Karimnezhad, A. and Perkins, T.J. (2023). Empirical Bayes Single Nucleotide Variant Calling For Next-Generation Sequencing Data. Working Paper. <https://mysite.science.uottawa.ca/akarimne/wp-content/uploads/2023/12/AK-TJP-SR.pdf>
#'
#' @export
get_LFDRs_given_caller<-function(bam_input,calls,LFDR.T,error){
  if (is.null(LFDR.T)){LFDR.T=0.01}

  A=utils::read.csv(bam_input,sep="\t",quote = "\"",header=F,stringsAsFactors = F)
  colnames(A)[1:2]=c('CHR','POS')
  B=utils::read.table(calls,fill = T)
  colnames(B)[1:2]=c('CHR','POS')
  A0=NULL
  u=unique(B$CHR)
  for (i in u){
    A0=rbind(A0,A[A$CHR==i & A$POS%in%B[B$CHR==i,]$POS,])
  }
  if(dim(A0)[1]==0){warning('no CHR and postion no. was found in bam_input')}
  Prepare_LFDR_input<-function(Input){
    index<-c()
    Allele1<-Ct1<-Ave.Map.Q1<-Ave.Base.Q1<-Ct.Plus.Str.1<-Ct.Mns.Str.1<-c()
    Allele2<-Ct2<-Ave.Map.Q2<-Ave.Base.Q2<-Ct.Plus.Str.2<-Ct.Mns.Str.2<-c()
    Allele3<-Ct3<-Ave.Map.Q3<-Ave.Base.Q3<-Ct.Plus.Str.3<-Ct.Mns.Str.3<-c()
    Allele4<-Ct4<-Ave.Map.Q4<-Ave.Base.Q4<-Ct.Plus.Str.4<-Ct.Mns.Str.4<-c()
    Allele5<-Ct5<-Ave.Map.Q5<-Ave.Base.Q5<-Ct.Plus.Str.5<-Ct.Mns.Str.5<-c()
    for ( i in 1:dim(Input)[1]){
      # for ( i in 1:10000){
      if (nchar(as.character(Input[i,10]))>0){
        Allele1<-c(Allele1,unlist(strsplit(Input[i,6], ":"))[1])
        Ct1<-c(Ct1,unlist(strsplit(Input[i,6], ":"))[2])
        Ave.Map.Q1<-c(Ave.Map.Q1,unlist(strsplit(Input[i,6], ":"))[3])
        Ave.Base.Q1<-c(Ave.Base.Q1,unlist(strsplit(Input[i,6], ":"))[4])
        Ct.Plus.Str.1<-c(Ct.Plus.Str.1,unlist(strsplit(Input[i,6], ":"))[6])
        Ct.Mns.Str.1<-c(Ct.Mns.Str.1,unlist(strsplit(Input[i,6], ":"))[7])
        Allele2<-c(Allele2,unlist(strsplit(Input[i,7], ":"))[1])
        Ct2<-c(Ct2,unlist(strsplit(Input[i,7], ":"))[2])
        Ave.Map.Q2<-c(Ave.Map.Q2,unlist(strsplit(Input[i,7], ":"))[3])
        Ave.Base.Q2<-c(Ave.Base.Q2,unlist(strsplit(Input[i,7], ":"))[4])
        Ct.Plus.Str.2<-c(Ct.Plus.Str.2,unlist(strsplit(Input[i,7], ":"))[6])
        Ct.Mns.Str.2<-c(Ct.Mns.Str.2,unlist(strsplit(Input[i,7], ":"))[7])
        Allele3<-c(Allele3,unlist(strsplit(Input[i,8], ":"))[1])
        Ct3<-c(Ct3,unlist(strsplit(Input[i,8], ":"))[2])
        Ave.Map.Q3<-c(Ave.Map.Q3,unlist(strsplit(Input[i,8], ":"))[3])
        Ave.Base.Q3<-c(Ave.Base.Q3,unlist(strsplit(Input[i,8], ":"))[4])
        Ct.Plus.Str.3<-c(Ct.Plus.Str.3,unlist(strsplit(Input[i,8], ":"))[6])
        Ct.Mns.Str.3<-c(Ct.Mns.Str.3,unlist(strsplit(Input[i,8], ":"))[7])
        Allele4<-c(Allele4,unlist(strsplit(Input[i,9], ":"))[1])
        Ct4<-c(Ct4,unlist(strsplit(Input[i,9], ":"))[2])
        Ave.Map.Q4<-c(Ave.Map.Q4,unlist(strsplit(Input[i,9], ":"))[3])
        Ave.Base.Q4<-c(Ave.Base.Q4,unlist(strsplit(Input[i,9], ":"))[4])
        Ct.Plus.Str.4<-c(Ct.Plus.Str.4,unlist(strsplit(Input[i,9], ":"))[6])
        Ct.Mns.Str.4<-c(Ct.Mns.Str.4,unlist(strsplit(Input[i,9], ":"))[7])
        Allele5<-c(Allele5,unlist(strsplit(Input[i,10], ":"))[1])
        Ct5<-c(Ct5,unlist(strsplit(Input[i,10], ":"))[2])
        Ave.Map.Q5<-c(Ave.Map.Q5,unlist(strsplit(Input[i,10], ":"))[3])
        Ave.Base.Q5<-c(Ave.Base.Q5,unlist(strsplit(Input[i,10], ":"))[4])
        Ct.Plus.Str.5<-c(Ct.Plus.Str.5,unlist(strsplit(Input[i,10], ":"))[6])
        Ct.Mns.Str.5<-c(Ct.Mns.Str.5,unlist(strsplit(Input[i,10], ":"))[7])
        index<-c(index,i)
      }
    }
    Ideal.Mx<-cbind(Input[index,1:4],
                    as.character(Allele1),as.numeric(as.character(Ct1)),as.numeric(as.character(Ave.Map.Q1)),
                    as.numeric(as.character(Ave.Base.Q1)),as.numeric(as.character(Ct.Plus.Str.1)),
                    as.numeric(as.character(Ct.Mns.Str.1)),
                    Allele2,as.numeric(as.character(Ct2)),as.numeric(as.character(Ave.Map.Q2)),
                    as.numeric(as.character(Ave.Base.Q2)),as.numeric(as.character(Ct.Plus.Str.2)),
                    as.numeric(as.character(Ct.Mns.Str.2)),
                    Allele3,as.numeric(as.character(Ct3)),as.numeric(as.character(Ave.Map.Q3)),
                    as.numeric(as.character(Ave.Base.Q3)),as.numeric(as.character(Ct.Plus.Str.3)),
                    as.numeric(as.character(Ct.Mns.Str.3)),
                    Allele4,as.numeric(as.character(Ct4)),as.numeric(as.character(Ave.Map.Q4)),
                    as.numeric(as.character(Ave.Base.Q4)),as.numeric(as.character(Ct.Plus.Str.4)),
                    as.numeric(as.character(Ct.Mns.Str.4)),
                    Allele5,as.numeric(as.character(Ct5)),as.numeric(as.character(Ave.Map.Q5)),
                    as.numeric(as.character(Ave.Base.Q5)),as.numeric(as.character(Ct.Plus.Str.5)),
                    as.numeric(as.character(Ct.Mns.Str.5)))
    J=droplevels(Ideal.Mx)
    colnames(Ideal.Mx)<-c('CHR','POS','Ref','Total',
                          'Allele1','Ct1','Ave.Map.Q1','Ave.Base.Q1','Ct.Plus.Str.1','Ct.Mns.Str.1',
                          'Allele2','Ct2','Ave.Map.Q2','Ave.Base.Q2','Ct.Plus.Str.2','Ct.Mns.Str.2',
                          'Allele3','Ct3','Ave.Map.Q3','Ave.Base.Q3','Ct.Plus.Str.3','Ct.Mns.Str.3',
                          'Allele4','Ct4','Ave.Map.Q4','Ave.Base.Q4','Ct.Plus.Str.4','Ct.Mns.Str.4',
                          'Allele5','Ct5','Ave.Map.Q5','Ave.Base.Q5','Ct.Plus.Str.5','Ct.Mns.Str.5' )
    I<-Ideal.Mx
    I[which(!as.character(I[,3])%in%c('A','C','G','T')),'Ref']<-toupper(as.character(I[which(!as.character(I[,3])%in%c('A','C','G','T')),3]))
    I$Total<-as.numeric(as.character(I$Ct1))+as.numeric(as.character(I$Ct2))+as.numeric(as.character(I$Ct3))+as.numeric(as.character(I$Ct4))
    x<-c()
    ideal.col<-c(5,11,17,23)
    for ( i in 1:dim(I)[1]){
      #  for ( i in 1:10){
      #print(i)
      j.Ref<-j.nonRef<-c()
      #for ( i in 1:10){
      for ( j in ideal.col){
        if (I[i,j]==I[i,3]){
          j.Ref<-j}
        else{
          j.nonRef<-ideal.col[!ideal.col%in%j.Ref]
        }
      }
      L.Ref<-cbind(as.character(I[i,c(j.Ref)]),I[i,c(j.Ref+1)],I[i,c(j.Ref+2)],I[i,c(j.Ref+3)],I[i,c(j.Ref+4)],I[i,c(j.Ref+5)])
      Alt.counts=c(I[i,c(j.nonRef[1]+1)],I[i,c(j.nonRef[2]+1)],I[i,c(j.nonRef[3]+1)])
      w01<-which(Alt.counts==max(Alt.counts))
      w02<-which(Alt.counts!=max(Alt.counts))
      if (length(w01)==1){
        w_alt=w01
        w_alts=w02
      } else if (length(w01)==2){
        w_alt=w01[1]
        w_alts<-c(w01[2],w02)
      } else if (length(w01)==3){
        w_alt=w01[1]
        w_alts<-c(w01[2],w01[3])
      }
      L.Alt<-c(as.character(I[i,j.nonRef[w_alt]]),I[i,j.nonRef[w_alt]+1],I[i,j.nonRef[w_alt]+2],I[i,j.nonRef[w_alt]+3],I[i,j.nonRef[w_alt]+4],I[i,j.nonRef[w_alt]+5])
      L.Alt1<-c(as.character(I[i,j.nonRef[w_alts[1]]]),I[i,j.nonRef[w_alts[1]]+1],I[i,j.nonRef[w_alts[1]]+2],I[i,j.nonRef[w_alts[1]]+3],I[i,j.nonRef[w_alts[1]]+4],I[i,j.nonRef[w_alts[1]]+5])
      L.Alt2<-c(as.character(I[i,j.nonRef[w_alts[2]]]),I[i,j.nonRef[w_alts[2]]+1],I[i,j.nonRef[w_alts[2]]+2],I[i,j.nonRef[w_alts[2]]+3],I[i,j.nonRef[w_alts[2]]+4],I[i,j.nonRef[w_alts[2]]+5])

      x<-rbind(x,c(I[i,1],I[i,2],L.Ref,L.Alt,L.Alt1,L.Alt2))
    }
    #x<-cbind(x,I$Total)
    colnames(x)<-c('CHR','POS','Ref','Ct.Ref','Ave.Map.Q.Ref','Ave.Base.Q.Ref','Ct.Plus.Ref.Str.1','Ct.Plus.Ref.Str.2',
                   'Alt','Ct.Alt','Ave.Map.Q.Alt','Ave.Base.Q.Alt','Ct.Plus.Alt.Str.1','Ct.Plus.Alt.Str.2',
                   'Alt1','Ct.Alt1','Ave.Map.Q.Alt1','Ave.Base.Q.Alt1','Ct.Plus.Alt1.Str.1','Ct.Plus.Alt1.Str.2',
                   'Alt2','Ct.Alt2','Ave.Map.Q.Alt2','Ave.Base.Q.Alt2','Ct.Plus.Alt2.Str.1','Ct.Plus.Alt2.Str.2')
    Depth<-as.numeric(as.character(x[,'Ct.Ref']))+as.numeric(as.character(x[,'Ct.Alt']))+as.numeric(as.character(x[,'Ct.Alt1']))+as.numeric(as.character(x[,'Ct.Alt2']))
    #1st definition
    #AF<-as.numeric(x[,'Ct.Alt'])/Depth
    #2nd definition
    AF<-as.numeric(x[,'Ct.Alt'])/(as.numeric(x[,'Ct.Alt'])+as.numeric(x[,'Ct.Ref']))
    x<-cbind(x,Depth,AF)
    #x<-noquote(x)
    #x[1:8,]
    C<-data.frame(x)
    D<-noquote(C)
    x.e<-D
    x.e[3,]
    dim(x.e)
    return(x.e)
  }
  D=Prepare_LFDR_input(Input=A0)
  D_noduplicated<-D[!duplicated(D),]
  get.f1<-function(x,theta,what.error){
    R<-as.numeric(as.character(x$Ct.Ref))
    M<-as.numeric(as.character(x$Ct.Alt))
    X1<-as.numeric(as.character(x$Ct.Alt1))
    X2<-as.numeric(as.character(x$Ct.Alt2))
    x<-cbind(R,M,X1,X2)
    error=what.error
    f1.all<-sapply(1:dim(x)[1], function (j) {mean(sapply(1:length(theta), function(k) {stats::dmultinom(x[j,], size = NULL,prob=c(theta[k]*(error[j]/3)+(1-theta[k])*(1-error[j]),theta[k]*(1-error[j])+(1-theta[k])*error[j]/3,error[j]/3,error[j]/3),log = FALSE)}))})
    return(f1.all)
  }
  get.f0<-function(x,what.error){
    R<-as.numeric(as.character(x$Ct.Ref))
    M<-as.numeric(as.character(x$Ct.Alt))
    X1<-as.numeric(as.character(x$Ct.Alt1))
    X2<-as.numeric(as.character(x$Ct.Alt2))
    x<-cbind(R,M,X1,X2)
    probs_0=cbind(1-what.error,what.error/3,what.error/3,what.error/3)
    f0.all<-sapply(1:dim(x)[1], function(j) {stats::dmultinom(x[j,], size = NULL, prob=probs_0[j,], log = FALSE)})
    return(f0.all)
  }
  x.filtered=D_noduplicated
  if (is.null(error)){
    M_BQ1=as.numeric(as.character(x.filtered$Ave.Base.Q.Alt))
    M_BQ2=as.numeric(as.character(x.filtered$Ave.Map.Q.Alt))
    M_C=as.numeric(as.character(x.filtered$Ct.Alt))
    Alt1_BQ1=as.numeric(as.character(x.filtered$Ave.Base.Q.Alt1))
    Alt1_BQ2=as.numeric(as.character(x.filtered$Ave.Map.Q.Alt1))
    Alt1_C=as.numeric(as.character(x.filtered$Ct.Alt1))
    Alt2_BQ1=as.numeric(as.character(x.filtered$Ave.Base.Q.Alt2))
    Alt2_BQ2=as.numeric(as.character(x.filtered$Ave.Map.Q.Alt2))
    Alt2_C=as.numeric(as.character(x.filtered$Ct.Alt2))
    R_BQ1=as.numeric(as.character(x.filtered$Ave.Base.Q.Ref))
    R_BQ2=as.numeric(as.character(x.filtered$Ave.Map.Q.Ref))
    R_C=as.numeric(as.character(x.filtered$Ct.Ref))
    M_BQ=(M_BQ1+M_BQ2)/2
    Alt1_BQ=(Alt1_BQ1+Alt1_BQ2)/2
    Alt2_BQ=(Alt2_BQ1+Alt2_BQ2)/2
    R_BQ=(R_BQ1+R_BQ2)/2
    error=10^(-((M_BQ*M_C + Alt1_BQ*Alt1_C  + Alt2_BQ*Alt2_C  + R_BQ*R_C)/(M_C+Alt1_C+Alt2_C+R_C))/10)
  }
  x.filtered<-cbind(x.filtered,error)


  R<-as.numeric(as.character(x.filtered$Ct.Ref))
  M<-as.numeric(as.character(x.filtered$Ct.Alt))
  X1<-as.numeric(as.character(x.filtered$Ct.Alt1))
  X2<-as.numeric(as.character(x.filtered$Ct.Alt2))
  pi0=1-dim(x.filtered)[1]/dim(A)[1]
  theta=as.numeric(as.character(x.filtered$AF))
  f0<-get.f0(x =x.filtered ,what.error = error )
  f1<-get.f1(x =x.filtered,theta =theta,what.error =error )
  LFDRs<-(pi0*f0)/(pi0*f0+(1-pi0)*f1)
  Mutant=ifelse(LFDRs>LFDR.T,0,1)
  output=cbind(x.filtered,LFDRs,Mutant)
  return(list('estimated.LFDRs'=LFDRs,'updated.vcf'=output))
}

