#' @title Harsanyi dividend
#' @description This function computes the Harsanyi dividend of the given coalition in the given game.
#' @param v A characteristic function, as a vector.
#' @param S The position of a coalition, as an integer.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} and \code{S} are introduced according to binary order instead of lexicographic order.
#' @return The Harsanyi dividend of the coalition that occupies the given position in the given order.
#' @details The Harsanyi dividends of \eqn{v\in G^N} are the coordinates of the game in the base of unanimity games.
#' They are defined, for all \eqn{S\in 2^N}, by
#' \deqn{c_S=\sum_{S'\subset S}(-1)^{|S|-|S'|}v(S')}.
#' @seealso \link{unanimitygame}
#' @examples
#' n <- 3
#' v <- c(1, 5, 10, 7, 11, 15, 16) # introduced in lexicographic order
#'
#' coalitionsvector<-character()
#' dividendsvector<-numeric()
#'
#' for (i in 1:(2^n-1)){
#'   coalitionsvector <- c(coalitionsvector,
#'                         paste(getcoalition(i)[getcoalition(i) != 0],collapse = " "))
#'   dividendsvector <- c(dividendsvector,
#'                        harsanyidividend(v, codelex2bin(n,i), binary = FALSE))
#' }
#'
#' data.frame(Coalition = coalitionsvector, Dividend = dividendsvector)
#' data.frame(Coalition = bin2lex(coalitionsvector), Dividend = bin2lex(dividendsvector))
#' @references Hammer, P.J., Peled, U.N., & Sorensen, S. (1977). Pseudo-boolean function and game theory I. Core elements and Shapley value. \emph{Cahiers du Centre d'Etudes de Recherche Opérationnelle}, 19, 156-176.
#' @export

harsanyidividend <- function (v, S, binary = FALSE) {
  nC <- length(v)
  n <- log(nC + 1)/log(2)
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }
  if (S > nC) {
    stop("'S' must not be greater than the length of 'v'.")
  }
  if (binary == FALSE) {
    ###v <- lex2bin(v) ELIMINEI ESTA LIÑA
    S <- codelex2bin(n, S)
    HD <- 0
    nS <- sum(as.numeric(intToBits(S)[1:n]))
    for (R in 1:S) {
      if (bitwAnd(S, R) == R) {
        nR <- sum(as.numeric(intToBits(R)[1:n]))
        HD <- HD + v[codebin2lex(n,R)] * ((-1)^(nS - nR)) ##Fago o cambio de bin a lex en cada R
      }
    }
  }
  else {
    HD <- 0
    nS <- sum(as.numeric(intToBits(S)[1:n]))
    for (R in 1:S) {
      if (bitwAnd(S, R) == R) {
        nR <- sum(as.numeric(intToBits(R)[1:n]))
        HD <- HD + v[R] * ((-1)^(nS - nR))
      }
    }
  }
  return(HD)
}
