#' @title Positively weighted Shapley value
#' @description Given a game, positive player weights and an ordered partition of the set of players, this function returns the corresponding weighted Shapley value.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param weights The player weights, as a vector of positive numbers.
#' @param partition An ordered partition of the set of players, as a list of vectors. When not specified, it is taken to be the partition whose only element is the set of all players.
#' @return The positively weighted Shapley value of the game, as a vector.
#' @details A weight system \eqn{\omega} is a pair \eqn{\omega=(\lambda,\mathcal{S})} where \eqn{\lambda=(\lambda_{i})_{i \in N}} is a positive weight vector (\eqn{\lambda_{i}>0} for each \eqn{i \in N}) and \eqn{\mathcal{S}=(S_{1},\dots,S_{m})} is an ordered partition of \eqn{N}.
#' The weighted Shapley value with weight system \eqn{\omega=(\lambda,\mathcal{S})} is the linear map \eqn{Sh^{\omega}} that assigns to each unanimity game \eqn{u_{T}}, with \eqn{T \in 2^{N} \setminus \emptyset},
#' the allocations \eqn{Sh^{\omega}_{i}(u_{T})=\frac{\lambda_{i}}{\lambda(T \cap S_{k})}} if \eqn{i \in T \cap S_{k}} and \eqn{Sh^{\omega}_{i}=0} if \eqn{i \notin T \cap S_{k}},
#' where \eqn{k=\max\{i \in N : S_{i} \cap T \neq \emptyset\}}. Then, for each \eqn{v \in G^{N}} and being \eqn{c_{T}} the Harsanyi dividend of coalition \eqn{T \in 2^{N}},
#' \deqn{Sh^{\omega}(v)=\sum_{T \in 2^{N} \setminus \emptyset}c_{T}Sh^{\omega}(u_{T}).}
#' @examples
#' v <- c(0,0,0,0,0,0,1,0,0,1,3,4,6,8,10)
#' weightedshapleyvalue(v,binary=TRUE,weights=c(0.5,0.2,0.2,0.1))
#' w <- c(0,0,0,0,30,30,40,40,50,50,60,70,80,90,100)
#' weightedshapleyvalue(w,weights=c(1,2,3,4),partition=list(c(1,2),c(3,4)))
#' @seealso \link{coalitionweightedshapleyvalue}, \link{harsanyidividend}, \link{shapleyvalue}
#' @references Shapley, L. S. (1953). \emph{Additive and non-additive set functions}. PhD thesis, Department of Mathematics, Princeton University.
#' @export

weightedshapleyvalue <- function(v, binary=FALSE, weights, partition=NULL) {

  # Inputs:
  # v: juego (en orden binario)
  # weights: vector de pesos positivos
  # partition: lista con una partición ordenada de N
  # WeightedSHg(v,c(1,2,3),list(c(1,2),c(3))), el 3 tiene prioridad sobre la coalición (1,2)
  # WeightedSHg(v,c(1,2,3),list(c(3),c(1,2)))
  # WeightedSHg(v,c(1,2,3),list(c(1,2,3))), se obtiene el valor sin partición
  # WeightedSHg(v,c(1,2,3),list(c(1),c(2),c(3))), se obtiene un vector de contrib. marginales

  # Comprobaciones de siempre.
  nC=length(v) # número de coaliciones
  n=log(nC+1)/log(2) # número de jugadores
  if (n>floor(n))
  {
    stop("'v' must have length 2^n-1 for some n.")
  }
  # Fin comprobaciones de siempre.

  # Comprobaciones partition.
  if (is.null(partition)) { # si no se proporciona partition, consideramos N
    partition <- list(getcoalition(2^n-1))
  }
  m <- length(partition) # número de elementos en la partición
  # Convertimos los elementos de la partición en números binarios
  # P <- sapply(partition, function(part) sum(2^(part - 1)))
  P <- rep(0,m) # inicializamos P
  for (i in 1:m){
    P[i] <- getcoalitionnumber(partition[[i]])
  }
  # Control: nos han dado una partición?
  if (sum(P) + 1 != 2^n) {
    stop("'partition' is not a partition of the set of players.")
  }
  # Fin comprobaciones partition.

  ###################################
  ### cálculo del valor ponderado ###
  ###################################

  Sh <- rep(0, n)

  if(binary == FALSE) { # Cuentas en lexicográfico.
    for (S in 1:(2^n - 1)) { # recorremos todas las coaliciones
      # Calculamos el HD de S.
      HD <- harsanyidividend(v=v,S=codebin2lex(n=n,Nbin=S),binary=binary)
      # Determinamos el grupo T de jugadores de S que pertenecen a la clase más alta de la partición
      T <- 0
      for (p in 1:m) {
        if (bitwAnd(S, P[p]) > 0) { # si hay jugadores de S en la clase P[p]
          T <- bitwAnd(S, P[p])
        }
      }
      T <- as.integer(intToBits(T)[1:n]) # vector binario de T
      # Actualizamos el valor de Shapley ponderado.
      Sh <- Sh + HD * T * weights / sum(T * weights)
    }
  } else { # Cuentas originales en binario.
    for (S in 1:(2^n - 1)) { # recorremos todas las coaliciones
      # Calculamos el HD de S.
      HD <- harsanyidividend(v=v,S=S,binary=binary)
      # Determinamos el grupo T de jugadores de S que pertenecen a la clase más alta de la partición
      T <- 0
      for (p in 1:m) {
        if (bitwAnd(S, P[p]) > 0) { # si hay jugadores de S en la clase P[p]
          T <- bitwAnd(S, P[p])
        }
      }
      T <- as.integer(intToBits(T)[1:n]) # vector binario de T
      # Actualizamos el valor de Shapley ponderado.
      Sh <- Sh + HD * T * weights / sum(T * weights)
    }
  }

  return(Sh)
}
