<!-- README.md is generated from README.Rmd. Please edit that file -->

# TideCurves

This packages provides functions for synthesizing tide curves. Ideally
you collected data for 19 years for a fixed location. Shorter periods
are also possible and will produce good results.

The functions are based on the Harmonic Representation of Inequalities
(HRoI) and not on the harmonic method. Please consult the following
links for a detailed description of HRoI:

-   <https://www.bsh.de/DE/PUBLIKATIONEN/_Anlagen/Downloads/Meer_und_Umwelt/Berichte-des-BSH/Berichte-des-BSH_50_de.pdf?__blob=publicationFile&v=13/>

-   <https://doi.org/10.5194/os-15-1363-2019>

## Why should I use this package?

You should use this package for producing tide curves from past data.

## How do I use it?

Import your data set first and transform it to a readable form. See
attached data `tideObservation` for an example data frame.

``` r
library(TideCurves)
tideObservation[1:10, ]
#>    observation_date observation_time height
#> 1        2015/06/01         00:00:00  6.570
#> 2        2015/06/01         00:30:00  6.528
#> 3        2015/06/01         01:00:00  6.410
#> 4        2015/06/01         01:30:00  6.200
#> 5        2015/06/01         02:00:00  5.893
#> 6        2015/06/01         02:30:00  5.556
#> 7        2015/06/01         03:00:00  5.248
#> 8        2015/06/01         03:30:00  4.970
#> 9        2015/06/01         04:00:00  4.745
#> 10       2015/06/01         04:30:00  4.550

sapply(tideObservation, typeof)
#> observation_date observation_time           height 
#>      "character"      "character"         "double"
```

As of version 0.0.5 you have the option to build a model of class
`tidecurve` and then synthesize different periods using this model. The
`TideCurve` functions on the other hand builds and synthesizes in one
go. The different options are explained below.

### TideCurve

You can use your data as input for the function `TideCurve`. Setting the
periods for analyzing and synthesizing and wait for the table to be
produced. Afterwards you can compute the differences between the
observed data and lunar and solar tide curves.

``` r
mytidecurve <- TideCurve(dataInput = tideObservation, asdate = "2015/12/06",
             astime = "00:00:00",      aedate = "2015/12/31",
             aetime = "23:30:00",      ssdate = "2015/12/17",
             sstime = "00:00:00",      sedate = "2015/12/31",
             setime = "23:30:00")

str(mytidecurve)
#> List of 5
#>  $ data_matrix    :Classes 'data.table' and 'data.frame':    1233 obs. of  4 variables:
#>   ..$ numm     : num [1:1233] 23264 23264 23264 23264 23264 ...
#>   ..$ imm      : num [1:1233] 1 2 3 4 5 6 7 8 9 10 ...
#>   ..$ tmmttmond: num [1:1233] 42341 42341 42341 42341 42341 ...
#>   ..$ height   : num [1:1233] 6.65 6.45 6.21 5.98 5.76 ...
#>   ..- attr(*, ".internal.selfref")=<externalptr> 
#>   ..- attr(*, "sorted")= chr "numm"
#>  $ synthesis.lunar:Classes 'data.table' and 'data.frame':    720 obs. of  7 variables:
#>   ..$ date_time      : chr [1:720] "2015/12/16 16:46:55" "2015/12/16 17:17:58" "2015/12/16 17:49:02" "2015/12/16 18:20:05" ...
#>   ..$ time1          : num [1:720] 42353 42353 42353 42353 42353 ...
#>   ..$ height         : num [1:720] 5.99 5.73 5.38 5.05 4.75 ...
#>   ..$ i              : num [1:720] 23275 23275 23275 23275 23275 ...
#>   ..$ k              : num [1:720] 1 2 3 4 5 6 7 8 9 10 ...
#>   ..$ prediction_date: chr [1:720] "2015/12/16" "2015/12/16" "2015/12/16" "2015/12/16" ...
#>   ..$ prediction_time: chr [1:720] "16:46:55" "17:17:58" "17:49:02" "18:20:05" ...
#>   ..- attr(*, ".internal.selfref")=<externalptr> 
#>  $ tide.curve     :Classes 'data.table' and 'data.frame':    720 obs. of  5 variables:
#>   ..$ date_time      : chr [1:720] "2015/12/17 00:00:00" "2015/12/17 00:30:00" "2015/12/17 01:00:00" "2015/12/17 01:30:00" ...
#>   ..$ time1          : num [1:720] 42353 42353 42353 42353 42353 ...
#>   ..$ height         : num [1:720] 4.09 4.59 5.12 5.61 5.95 ...
#>   ..$ prediction_date: chr [1:720] "2015/12/17" "2015/12/17" "2015/12/17" "2015/12/17" ...
#>   ..$ prediction_time: chr [1:720] "00:00:00" "00:30:00" "01:00:00" "01:30:00" ...
#>   ..- attr(*, ".internal.selfref")=<externalptr> 
#>  $ lm.coeff       :List of 48
#>   ..$ 1 : num [1, 1:9] 6.4778 0.3482 0.34 -0.1687 -0.0651 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 2 : num [1, 1:9] 6.3002 0.448 0.4033 -0.1357 -0.0718 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 3 : num [1, 1:9] 6.0469 0.5405 0.416 -0.1245 -0.0808 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 4 : num [1, 1:9] 5.7536 0.6177 0.3878 -0.118 -0.0763 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 5 : num [1, 1:9] 5.4518 0.6585 0.3285 -0.1311 -0.0702 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 6 : num [1, 1:9] 5.1621 0.6804 0.2668 -0.1561 -0.0652 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 7 : num [1, 1:9] 4.9018 0.6918 0.222 -0.1619 -0.0555 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 8 : num [1, 1:9] 4.6628 0.683 0.1827 -0.1836 -0.0366 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 9 : num [1, 1:9] 4.4385 0.6737 0.144 -0.2082 -0.0174 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 10: num [1, 1:9] 4.2397 0.6633 0.0885 -0.2362 0.0181 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 11: num [1, 1:9] 4.083011 0.625853 -0.000901 -0.260132 0.051686 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 12: num [1, 1:9] 3.9981 0.5514 -0.1325 -0.2926 0.0989 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 13: num [1, 1:9] 4.046 0.393 -0.328 -0.326 0.171 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 14: num [1, 1:9] 4.247 0.185 -0.535 -0.351 0.241 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 15: num [1, 1:9] 4.5886 -0.00687 -0.64615 -0.37609 0.26602 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 16: num [1, 1:9] 5.04 -0.122 -0.596 -0.359 0.25 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 17: num [1, 1:9] 5.489 -0.111 -0.465 -0.334 0.229 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 18: num [1, 1:9] 5.862 -0.055 -0.313 -0.308 0.209 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 19: num [1, 1:9] 6.15 0.013 -0.2 -0.288 0.187 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 20: num [1, 1:9] 6.3652 0.0583 -0.125 -0.2783 0.1677 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 21: num [1, 1:9] 6.5186 0.0951 -0.0768 -0.2906 0.1565 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 22: num [1, 1:9] 6.6281 0.125 -0.0295 -0.3013 0.1527 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 23: num [1, 1:9] 6.6835 0.1785 0.0376 -0.2999 0.1424 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 24: num [1, 1:9] 6.662 0.249 0.109 -0.285 0.123 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 25: num [1, 1:9] 6.556 0.336 0.191 -0.269 0.103 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 26: num [1, 1:9] 6.3619 0.4289 0.2538 -0.2542 0.0885 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 27: num [1, 1:9] 6.097 0.504 0.264 -0.2433 0.0779 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 28: num [1, 1:9] 5.783 0.551 0.23 -0.239 0.07 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 29: num [1, 1:9] 5.4698 0.5767 0.1609 -0.2368 0.0726 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 30: num [1, 1:9] 5.1757 0.5924 0.0881 -0.2361 0.0706 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 31: num [1, 1:9] 4.9032 0.6035 0.0206 -0.2379 0.0748 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 32: num [1, 1:9] 4.6566 0.6166 -0.0287 -0.2385 0.0703 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 33: num [1, 1:9] 4.4302 0.6371 -0.0672 -0.2414 0.0609 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 34: num [1, 1:9] 4.18206 0.72949 -0.07193 -0.28938 -0.00489 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 35: num [1, 1:9] 4.0209 0.6987 -0.1566 -0.289 0.0152 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 36: num [1, 1:9] 3.9318 0.6161 -0.2572 -0.2903 0.0315 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 37: num [1, 1:9] 3.9621 0.446 -0.4015 -0.2812 0.0889 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 38: num [1, 1:9] 4.158 0.208 -0.562 -0.278 0.135 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 39: num [1, 1:9] 4.5135 -0.0147 -0.6281 -0.2667 0.1366 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 40: num [1, 1:9] 4.973 -0.12 -0.544 -0.276 0.115 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 41: num [1, 1:9] 5.434 -0.105 -0.377 -0.271 0.106 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 42: num [1, 1:9] 5.8102 -0.0232 -0.2257 -0.2476 0.102 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 43: num [1, 1:9] 6.0927 0.0628 -0.1244 -0.2116 0.1037 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 44: num [1, 1:9] 6.285 0.129 -0.051 -0.183 0.102 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 45: num [1, 1:9] 6.423 0.176 -0.013 -0.173 0.112 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 46: num [1, 1:9] 6.5183 0.2091 0.0232 -0.1687 0.1165 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 47: num [1, 1:9] 6.573 0.274 0.075 -0.161 0.111 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 48: num [1, 1:9] 6.559 0.356 0.134 -0.144 0.103 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>  $ diff.analyse   : num 26

mydifferences <- ResCurve(tcData = mytidecurve, obsData = tideObservation)
```

``` r
head(mydifferences)
#> $lunar.res
#>       numm imm tmmttmond dmheight numm_imm           date_time    time1
#>   1: 23275   1  42352.70 5.833343  23275_1 2015/12/16 16:46:55 42352.70
#>   2: 23275   2  42352.72 5.554384  23275_2 2015/12/16 17:17:58 42352.72
#>   3: 23275   3  42352.74 5.137982  23275_3 2015/12/16 17:49:02 42352.74
#>   4: 23275   4  42352.76 4.819359  23275_4 2015/12/16 18:20:05 42352.76
#>   5: 23275   5  42352.79 4.529649  23275_5 2015/12/16 18:51:08 42352.79
#>  ---                                                                   
#> 716:    NA  NA        NA       NA 23289_44 2016/01/01 02:48:45 42368.12
#> 717:    NA  NA        NA       NA 23289_45 2016/01/01 03:19:48 42368.14
#> 718:    NA  NA        NA       NA 23289_46 2016/01/01 03:50:51 42368.16
#> 719:    NA  NA        NA       NA 23289_47 2016/01/01 04:21:54 42368.18
#> 720:    NA  NA        NA       NA 23289_48 2016/01/01 04:52:57 42368.20
#>      lsheight     i  k prediction_date prediction_time        res
#>   1: 5.986578 23275  1      2015/12/16        16:46:55 -0.1532351
#>   2: 5.729373 23275  2      2015/12/16        17:17:58 -0.1749886
#>   3: 5.384323 23275  3      2015/12/16        17:49:02 -0.2463415
#>   4: 5.050630 23275  4      2015/12/16        18:20:05 -0.2312703
#>   5: 4.753140 23275  5      2015/12/16        18:51:08 -0.2234906
#>  ---                                                             
#> 716: 6.179895 23289 44      2016/01/01        02:48:45         NA
#> 717: 6.278912 23289 45      2016/01/01        03:19:48         NA
#> 718: 6.335339 23289 46      2016/01/01        03:50:51         NA
#> 719: 6.308614 23289 47      2016/01/01        04:21:54         NA
#> 720: 6.211828 23289 48      2016/01/01        04:52:57         NA
#> 
#> $solar.res
#>      observation_date observation_time obheight           date_time    time1
#>   1:       2015/12/17         00:00:00    4.180 2015/12/17 00:00:00 42353.00
#>   2:       2015/12/17         00:30:00    4.780 2015/12/17 00:30:00 42353.02
#>   3:       2015/12/17         01:00:00    5.340 2015/12/17 01:00:00 42353.04
#>   4:       2015/12/17         01:30:00    5.820 2015/12/17 01:30:00 42353.06
#>   5:       2015/12/17         02:00:00    6.130 2015/12/17 02:00:00 42353.08
#>  ---                                                                        
#> 716:       2015/12/31         21:30:00    5.007 2015/12/31 21:30:00 42367.90
#> 717:       2015/12/31         22:00:00    4.775 2015/12/31 22:00:00 42367.92
#> 718:       2015/12/31         22:30:00    4.550 2015/12/31 22:30:00 42367.94
#> 719:       2015/12/31         23:00:00    4.330 2015/12/31 23:00:00 42367.96
#> 720:       2015/12/31         23:30:00    4.150 2015/12/31 23:30:00 42367.98
#>      tsheight prediction_date prediction_time        res
#>   1: 4.085008      2015/12/17        00:00:00 0.09499181
#>   2: 4.591633      2015/12/17        00:30:00 0.18836680
#>   3: 5.120420      2015/12/17        01:00:00 0.21957968
#>   4: 5.608902      2015/12/17        01:30:00 0.21109807
#>   5: 5.954887      2015/12/17        02:00:00 0.17511274
#>  ---                                                    
#> 716: 4.225695      2015/12/31        21:30:00 0.78130549
#> 717: 4.038341      2015/12/31        22:00:00 0.73665870
#> 718: 3.487573      2015/12/31        22:30:00 1.06242661
#> 719: 3.243748      2015/12/31        23:00:00 1.08625218
#> 720: 3.267625      2015/12/31        23:30:00 0.88237527
```

### BuildTC & SynTC

Model building and synthesis are now decoupled. You need to build a
model with `BuildTC`, setting the analysis period and afterwards you
call `SynTC` for a defined synthesis period.

``` r
tc_model <- BuildTC(dataInput = tideObservation, asdate = "2015/12/06",
             astime = "00:00:00", aedate = "2015/12/31",
             aetime = "23:30:00")
```

``` r
str(tc_model)
#> List of 8
#>  $ lm.coeff     :List of 48
#>   ..$ 1 : num [1, 1:9] 6.4778 0.3482 0.34 -0.1687 -0.0651 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 2 : num [1, 1:9] 6.3002 0.448 0.4033 -0.1357 -0.0718 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 3 : num [1, 1:9] 6.0469 0.5405 0.416 -0.1245 -0.0808 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 4 : num [1, 1:9] 5.7536 0.6177 0.3878 -0.118 -0.0763 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 5 : num [1, 1:9] 5.4518 0.6585 0.3285 -0.1311 -0.0702 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 6 : num [1, 1:9] 5.1621 0.6804 0.2668 -0.1561 -0.0652 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 7 : num [1, 1:9] 4.9018 0.6918 0.222 -0.1619 -0.0555 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 8 : num [1, 1:9] 4.6628 0.683 0.1827 -0.1836 -0.0366 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 9 : num [1, 1:9] 4.4385 0.6737 0.144 -0.2082 -0.0174 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 10: num [1, 1:9] 4.2397 0.6633 0.0885 -0.2362 0.0181 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 11: num [1, 1:9] 4.083011 0.625853 -0.000901 -0.260132 0.051686 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 12: num [1, 1:9] 3.9981 0.5514 -0.1325 -0.2926 0.0989 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 13: num [1, 1:9] 4.046 0.393 -0.328 -0.326 0.171 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 14: num [1, 1:9] 4.247 0.185 -0.535 -0.351 0.241 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 15: num [1, 1:9] 4.5886 -0.00687 -0.64615 -0.37609 0.26602 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 16: num [1, 1:9] 5.04 -0.122 -0.596 -0.359 0.25 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 17: num [1, 1:9] 5.489 -0.111 -0.465 -0.334 0.229 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 18: num [1, 1:9] 5.862 -0.055 -0.313 -0.308 0.209 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 19: num [1, 1:9] 6.15 0.013 -0.2 -0.288 0.187 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 20: num [1, 1:9] 6.3652 0.0583 -0.125 -0.2783 0.1677 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 21: num [1, 1:9] 6.5186 0.0951 -0.0768 -0.2906 0.1565 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 22: num [1, 1:9] 6.6281 0.125 -0.0295 -0.3013 0.1527 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 23: num [1, 1:9] 6.6835 0.1785 0.0376 -0.2999 0.1424 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 24: num [1, 1:9] 6.662 0.249 0.109 -0.285 0.123 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 25: num [1, 1:9] 6.556 0.336 0.191 -0.269 0.103 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 26: num [1, 1:9] 6.3619 0.4289 0.2538 -0.2542 0.0885 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 27: num [1, 1:9] 6.097 0.504 0.264 -0.2433 0.0779 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 28: num [1, 1:9] 5.783 0.551 0.23 -0.239 0.07 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 29: num [1, 1:9] 5.4698 0.5767 0.1609 -0.2368 0.0726 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 30: num [1, 1:9] 5.1757 0.5924 0.0881 -0.2361 0.0706 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 31: num [1, 1:9] 4.9032 0.6035 0.0206 -0.2379 0.0748 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 32: num [1, 1:9] 4.6566 0.6166 -0.0287 -0.2385 0.0703 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 33: num [1, 1:9] 4.4302 0.6371 -0.0672 -0.2414 0.0609 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 34: num [1, 1:9] 4.18206 0.72949 -0.07193 -0.28938 -0.00489 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 35: num [1, 1:9] 4.0209 0.6987 -0.1566 -0.289 0.0152 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 36: num [1, 1:9] 3.9318 0.6161 -0.2572 -0.2903 0.0315 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 37: num [1, 1:9] 3.9621 0.446 -0.4015 -0.2812 0.0889 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 38: num [1, 1:9] 4.158 0.208 -0.562 -0.278 0.135 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 39: num [1, 1:9] 4.5135 -0.0147 -0.6281 -0.2667 0.1366 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 40: num [1, 1:9] 4.973 -0.12 -0.544 -0.276 0.115 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 41: num [1, 1:9] 5.434 -0.105 -0.377 -0.271 0.106 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 42: num [1, 1:9] 5.8102 -0.0232 -0.2257 -0.2476 0.102 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 43: num [1, 1:9] 6.0927 0.0628 -0.1244 -0.2116 0.1037 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 44: num [1, 1:9] 6.285 0.129 -0.051 -0.183 0.102 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 45: num [1, 1:9] 6.423 0.176 -0.013 -0.173 0.112 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 46: num [1, 1:9] 6.5183 0.2091 0.0232 -0.1687 0.1165 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 47: num [1, 1:9] 6.573 0.274 0.075 -0.161 0.111 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>   ..$ 48: num [1, 1:9] 6.559 0.356 0.134 -0.144 0.103 ...
#>   .. ..- attr(*, "dimnames")=List of 2
#>   .. .. ..$ : NULL
#>   .. .. ..$ : chr [1:9] "V1" "V2" "V3" "V4" ...
#>  $ tdiff.analyse: num 26
#>  $ km           : num 48
#>  $ mindt        : num 30
#>  $ otz.24       : num 0.0417
#>  $ tplus        : num 18262
#>  $ tm24         : num 1.04
#>  $ data_matrix  : NULL
#>  - attr(*, "class")= chr "tidecurve"
```

``` r
tc <- SynTC(tmodel = tc_model, ssdate = "2015/12/17", sstime = "00:00:00",
sedate = "2015/12/31", setime = "23:30:00")
```

``` r
str(tc)
#> List of 2
#>  $ synthesis.lunar:Classes 'data.table' and 'data.frame':    720 obs. of  7 variables:
#>   ..$ date_time      : chr [1:720] "2015/12/16 16:46:55" "2015/12/16 17:17:58" "2015/12/16 17:49:02" "2015/12/16 18:20:05" ...
#>   ..$ time1          : num [1:720] 42353 42353 42353 42353 42353 ...
#>   ..$ height         : num [1:720] 5.99 5.73 5.38 5.05 4.75 ...
#>   ..$ i              : num [1:720] 23275 23275 23275 23275 23275 ...
#>   ..$ k              : num [1:720] 1 2 3 4 5 6 7 8 9 10 ...
#>   ..$ prediction_date: chr [1:720] "2015/12/16" "2015/12/16" "2015/12/16" "2015/12/16" ...
#>   ..$ prediction_time: chr [1:720] "16:46:55" "17:17:58" "17:49:02" "18:20:05" ...
#>   ..- attr(*, ".internal.selfref")=<externalptr> 
#>  $ tide.curve     :Classes 'data.table' and 'data.frame':    720 obs. of  5 variables:
#>   ..$ date_time      : chr [1:720] "2015/12/17 00:00:00" "2015/12/17 00:30:00" "2015/12/17 01:00:00" "2015/12/17 01:30:00" ...
#>   ..$ time1          : num [1:720] 42353 42353 42353 42353 42353 ...
#>   ..$ height         : num [1:720] 4.09 4.59 5.12 5.61 5.95 ...
#>   ..$ prediction_date: chr [1:720] "2015/12/17" "2015/12/17" "2015/12/17" "2015/12/17" ...
#>   ..$ prediction_time: chr [1:720] "00:00:00" "00:30:00" "01:00:00" "01:30:00" ...
#>   ..- attr(*, ".internal.selfref")=<externalptr>
```

Calling ResCurve using the `BuildTC` followed by `SynTC` is also
possible:

``` r
tc_model <- BuildTC(dataInput = tideObservation, asdate = "2015/12/06",
             astime = "00:00:00", aedate = "2015/12/31",
             aetime = "23:30:00", keep_data = TRUE)

syn_tc   <- SynTC(tmodel = tc_model,  ssdate = "2015/12/17", sstime = "00:00:00" ,
              sedate = "2015/12/31", setime = "23:30:00")

syn_tc$data_matrix <- tc_model$data_matrix

res_tc_n <- ResCurve(syn_tc, tideObservation)
```

``` r
res_tc_n
#> $lunar.res
#>       numm imm tmmttmond dmheight numm_imm           date_time    time1
#>   1: 23275   1  42352.70 5.833343  23275_1 2015/12/16 16:46:55 42352.70
#>   2: 23275   2  42352.72 5.554384  23275_2 2015/12/16 17:17:58 42352.72
#>   3: 23275   3  42352.74 5.137982  23275_3 2015/12/16 17:49:02 42352.74
#>   4: 23275   4  42352.76 4.819359  23275_4 2015/12/16 18:20:05 42352.76
#>   5: 23275   5  42352.79 4.529649  23275_5 2015/12/16 18:51:08 42352.79
#>  ---                                                                   
#> 716:    NA  NA        NA       NA 23289_44 2016/01/01 02:48:45 42368.12
#> 717:    NA  NA        NA       NA 23289_45 2016/01/01 03:19:48 42368.14
#> 718:    NA  NA        NA       NA 23289_46 2016/01/01 03:50:51 42368.16
#> 719:    NA  NA        NA       NA 23289_47 2016/01/01 04:21:54 42368.18
#> 720:    NA  NA        NA       NA 23289_48 2016/01/01 04:52:57 42368.20
#>      lsheight     i  k prediction_date prediction_time        res
#>   1: 5.986578 23275  1      2015/12/16        16:46:55 -0.1532351
#>   2: 5.729373 23275  2      2015/12/16        17:17:58 -0.1749886
#>   3: 5.384323 23275  3      2015/12/16        17:49:02 -0.2463415
#>   4: 5.050630 23275  4      2015/12/16        18:20:05 -0.2312703
#>   5: 4.753140 23275  5      2015/12/16        18:51:08 -0.2234906
#>  ---                                                             
#> 716: 6.179895 23289 44      2016/01/01        02:48:45         NA
#> 717: 6.278912 23289 45      2016/01/01        03:19:48         NA
#> 718: 6.335339 23289 46      2016/01/01        03:50:51         NA
#> 719: 6.308614 23289 47      2016/01/01        04:21:54         NA
#> 720: 6.211828 23289 48      2016/01/01        04:52:57         NA
#> 
#> $solar.res
#>      observation_date observation_time obheight           date_time    time1
#>   1:       2015/12/17         00:00:00    4.180 2015/12/17 00:00:00 42353.00
#>   2:       2015/12/17         00:30:00    4.780 2015/12/17 00:30:00 42353.02
#>   3:       2015/12/17         01:00:00    5.340 2015/12/17 01:00:00 42353.04
#>   4:       2015/12/17         01:30:00    5.820 2015/12/17 01:30:00 42353.06
#>   5:       2015/12/17         02:00:00    6.130 2015/12/17 02:00:00 42353.08
#>  ---                                                                        
#> 716:       2015/12/31         21:30:00    5.007 2015/12/31 21:30:00 42367.90
#> 717:       2015/12/31         22:00:00    4.775 2015/12/31 22:00:00 42367.92
#> 718:       2015/12/31         22:30:00    4.550 2015/12/31 22:30:00 42367.94
#> 719:       2015/12/31         23:00:00    4.330 2015/12/31 23:00:00 42367.96
#> 720:       2015/12/31         23:30:00    4.150 2015/12/31 23:30:00 42367.98
#>      tsheight prediction_date prediction_time        res
#>   1: 4.085008      2015/12/17        00:00:00 0.09499181
#>   2: 4.591633      2015/12/17        00:30:00 0.18836680
#>   3: 5.120420      2015/12/17        01:00:00 0.21957968
#>   4: 5.608902      2015/12/17        01:30:00 0.21109807
#>   5: 5.954887      2015/12/17        02:00:00 0.17511274
#>  ---                                                    
#> 716: 4.225695      2015/12/31        21:30:00 0.78130549
#> 717: 4.038341      2015/12/31        22:00:00 0.73665870
#> 718: 3.487573      2015/12/31        22:30:00 1.06242661
#> 719: 3.243748      2015/12/31        23:00:00 1.08625218
#> 720: 3.267625      2015/12/31        23:30:00 0.88237527
```
