% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/boilerplate_batch_edit_functions.R
\name{boilerplate_batch_edit}
\alias{boilerplate_batch_edit}
\title{Batch Edit Fields in boilerplate Database}
\usage{
boilerplate_batch_edit(
  db,
  field,
  new_value,
  target_entries = NULL,
  match_pattern = NULL,
  match_values = NULL,
  category = NULL,
  recursive = TRUE,
  case_sensitive = FALSE,
  preview = FALSE,
  confirm = TRUE,
  quiet = FALSE
)
}
\arguments{
\item{db}{List or character. The database to edit (can be a single category or unified database),
or a file path to a JSON/RDS database file which will be loaded automatically.}

\item{field}{Character. The field to edit (e.g., "reference", "description").}

\item{new_value}{Character. The new value to set for the field.}

\item{target_entries}{Character vector. Entries to edit. Can be:
\itemize{
\item Specific entry names (e.g., c("ban_hate_speech", "born_nz"))
\item Patterns with wildcards (e.g., "anxiety*" matches all entries starting with "anxiety")
\item "all" to edit all entries
\item NULL to use pattern/value matching
}}

\item{match_pattern}{Character. Pattern to match in the current field values.
Only entries with matching values will be updated. Ignored if target_entries is specified.}

\item{match_values}{Character vector. Specific values to match.
Only entries with these exact values will be updated. Ignored if target_entries is specified.}

\item{category}{Character. If db is unified, specifies which category to edit
(e.g., "measures", "methods"). If NULL, attempts to detect.}

\item{recursive}{Logical. Whether to search recursively through nested structures.}

\item{case_sensitive}{Logical. Whether pattern matching is case sensitive.}

\item{preview}{Logical. If TRUE, shows what would be changed without making changes.}

\item{confirm}{Logical. If TRUE, asks for confirmation before making changes.}

\item{quiet}{Logical. If TRUE, suppresses non-essential messages.}
}
\value{
The modified database (invisibly if preview=TRUE).
}
\description{
This function allows batch editing of specific fields across multiple entries
in a boilerplate database. It supports pattern matching, explicit lists, and
various editing operations.
}
\examples{
# Create a temporary directory and initialise database
temp_dir <- tempdir()
data_path <- file.path(temp_dir, "boilerplate_batch_edit_example", "data")

# Initialise database with default content
boilerplate_init(
  categories = "measures",
  data_path = data_path,
  create_dirs = TRUE,
  create_empty = FALSE,
  confirm = FALSE,
  quiet = TRUE
)

# Load database
unified_db <- boilerplate_import(data_path = data_path, quiet = TRUE)

# Example 1: Change specific references
unified_db <- boilerplate_batch_edit(
  db = unified_db,
  field = "reference",
  new_value = "example2024",
  target_entries = c("anxiety", "depression"),
  category = "measures",
  confirm = FALSE,
  quiet = TRUE
)

# Check the changes
unified_db$measures$anxiety$reference

# Example 2: Preview changes before applying
boilerplate_batch_edit(
  db = unified_db,
  field = "waves",
  new_value = "1-5",
  target_entries = "anxiety*",  # All entries starting with "anxiety"
  category = "measures",
  preview = TRUE,
  quiet = TRUE
)

# Example 3: Load database directly from file
\donttest{
# First save the database to a JSON file
json_file <- file.path(temp_dir, "boilerplate_unified.json")
boilerplate_save(unified_db, format = "json", data_path = temp_dir, quiet = TRUE)

# Now edit directly from the file
db <- boilerplate_batch_edit(
  db = json_file,  # File path instead of database object
  field = "description",
  new_value = "Updated description",
  target_entries = "anxiety",
  category = "measures",
  confirm = FALSE,
  quiet = TRUE
)
}

# Clean up
unlink(file.path(temp_dir, "boilerplate_batch_edit_example"), recursive = TRUE)

}
