##
# ------------------------------------------------------------------------
#
#  "pkc(nul_dist,alt_dist, alpha,ad=FALSE)" --
#
#
# ------------------------------------------------------------------------
##
#' @aliases pkc
#' @title Plot Kernel Density Estimates for Null and Alternative Distributions.
#' @description  Plots kernel density estimates for null and alternative distributions, 
#' showing the acceptance region for a hypothesis test and highlighting 
#' the type II error against an alternative hypothesis.
#' @param nul_dist Numeric vector representing the distribution under the null 
#' hypothesis.
#' @param alt_dist Numeric vector representing the distribution under an 
#' alternative hypothesis.
#' @param alpha Numeric value between \eqn{[0,0.5]}; the significance level of the test 
#' (type I error rate).
#' @details This function visualizes the kernel density estimates of two 
#' distributions:
#' one under the null hypothesis and the other one under an alternative hypothesis.
#' It highlights the acceptance region of the test (using the significance
#'  level \code{alpha}) 
#' and the region corresponding to type II error. This visual representation can
#'  be useful for understanding the behaviour of bootstrapped test statistics and the 
#'  trade-off between type I and type II errors.
#' @return Creates a plot showing the kernel density estimates of the null and 
#' alternative distributions with the relevant regions highlighted. The function
#'  does not return any values.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted..
#' 
#' Beran R. (1986).  Simulated Power Functions. \emph{Ann. Statist.}, \bold{14}, 151 - 173. 
#' 
#' @seealso \code{\link{compute_power}}.
#' @keywords "Bootstrap testing" Power
#' @examples
#' # Example: Comparing null and alternative distributions
#' # Generate two normally distributed samples
#' set.seed(123)
#' null_dist <- rnorm(1000, mean = 0, sd = 1) # Null distribution
#' alt_dist <- rnorm(1000, mean = 0.5, sd = 1) # Alternative distribution
#' alpha <- 0.05 # Significance level
#'
#' # Plot kernel density estimates
#' pkc(null_dist, alt_dist, alpha)
#' @export
##

pkc<- function(nul_dist,alt_dist, alpha) {
  
  # compute Kernel estimators of distribution under the null and the alternative
  density_est <- density(nul_dist)
  density_alt <- density(alt_dist)
  
  # Compute quantiles of the null distribution of order alpha/2 and 1-alpha/2
  q_low <- quantile(nul_dist, alpha/2)
  q_high <- quantile(nul_dist, 1-alpha/2)
  
  # Plot kernel density estimate of null distribution
  plot(density_est, main = "Kernel Density Estimates and error of type I and II", xlab = "Confidence intervals under H0 (blue) and error of type II (orange)", ylab = "Density",col="blue")
  # Add distribution under the alternative
  lines(density_alt,col="red")
  
  # Highlight area between quantiles under the H0 density curve in red
  polygon(c(density_est$x, rev(density_est$x)),
          c(ifelse(density_est$x < q_low, density_est$y, 0),
            rev(ifelse(density_est$x < q_high, density_est$y, 0))),
          col = rgb(red=0,green = 0, blue = 1,alpha=0.50), border = NA)
  
  # Highlight area between quantiles under H1 = type II error in red
  
  polygon(c(density_alt$x, rev(density_alt$x)),
          c(ifelse(density_alt$x < q_low, density_alt$y, 0),
            rev(ifelse(density_alt$x < q_high, density_alt$y, 0))),
          col = rgb(red=1,green = 0, blue = 0,alpha=0.50), border = NA)
  
  
  # Add legend
  legend("topleft", legend = c("Smoothed Bootstrap distribution under the null",
                               "Smoothed Bootstrap distribution under the alternative"),
         fill =c("blue","red"), border = NA)
}