% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RCode.r
\docType{package}
\name{cmpp}
\alias{cmpp-package}
\alias{cmpp}
\title{Direct Parametric Inference for the Cumulative Incidence Function in Competing Risks}
\description{
The \code{cmpp} package provides parametric (Direct) modeling methods for analyzing cumulative incidence functions (CIFs)
in the context of competing risks. It includes Gompertz-based models, regression techniques, and parametric (Direct) approaches
such as the Generalized odds rate (GOR), Proportional Odds Model (POM), and Proportional Hazards Model (PHM).
The package enables users to estimate and compare CIFs using maximum likelihood estimation, perform regression analysis,
and visualize CIFs with confidence intervals. It also supports covariate adjustment and bootstrap variance estimation.
}
\details{
The \code{cmpp} package offers functions for modeling cumulative incidence functions (CIFs) Directly
using the Gompertz distribution and generalized regression models.

Key features include:
\itemize{
\item Direct parametric modeling for cumulative incidence functions.
\item Maximum likelihood estimation of parameters.
\item Regression analysis with covariates, including treatment effects.
\item Visualization of CIFs with confidence intervals.
\item Covariate-adjusted CIF estimation.
\item Bootstrap variance estimation for model parameters.
}

Commonly used functions include:
\itemize{
\item \code{\link{Initialize}}: Initializes the data for the Cmpp model.
\item \code{\link{LogLike1}}: Computes the negative log-likelihood for the model without covariate effect.
\item \code{\link{compute_grad}}: Computes the gradient of the log-likelihood.
\item \code{\link{compute_hessian}}: Computes the Hessian matrix of the log-likelihood.
\item \code{\link{estimate_parameters_GOR}}: Estimates parameters using the Generalized odds rate (GOR).
\item \code{\link{estimate_parameters_POM}}: Estimates parameters using the Proportional Odds Model (POM).
\item \code{\link{estimate_parameters_PHM}}: Estimates parameters using the Proportional Hazards Model (PHM).
\item \code{\link{CIF_res1}}: Computes CIF results for competing risks without covariates.
\item \code{\link{CIF_Figs}}: Plots CIFs with confidence intervals (without covariate effect).
\item \code{\link{Cmpp_CIF}}: Computes and plots CIFs for competing risks using GOR, POM, and PHM.
\item \code{\link{FineGray_Model}}: Fits a Fine-Gray regression model for competing risks data and
visualize CIF by Fine-Gray model result using \code{\link[cmprsk:cuminc]{cmprsk::cuminc}} and \code{\link[cmprsk:crr]{cmprsk::crr}}.
\item \code{\link{bootstrap_variance}}: Estimates variance of parameters using the bootstrap method.
\item \code{\link{GetData}}: Retrieves initialized data from the Cmpp model.
\item \code{\link{Cleanup}}: Cleans up memory by deleting the Cmpp instance.
}
}
\examples{
## Example: Initialize the Cmpp model and compute CIFs
library(cmpp)
features <- matrix(rnorm(300, 1, 2), nrow = 100, ncol = 3)
delta1 <- sample(c(0, 1), 100, replace = TRUE)
delta2 <- 1 - delta1
timee <- rexp(100, rate = 1/10)
Initialize(features, timee, delta1, delta2, h = 1e-5)
# Initialize the Cmpp model
# Estimate parameters using the Generalized odds rate (GOR)
initial_params <- rep(0.001, 2 * (ncol(features) + 3))
result <- estimate_parameters_GOR(initial_params)
print(result)
# Compute CIFs for competing risks (without covariate effect | Not Regression model)
cif_results <- CIF_res1()
print(cif_results)
# Plot CIFs with confidence intervals
plot <- CIF_Figs(rep(0.01, 4), timee)
print(plot)
# Compute and plot adjusted CIFs
result_cif <- Cmpp_CIF(
featureID = c(1, 2),
featureValue = c(0.5, 1.2),
RiskNames = c("Event1", "Event2"),
TypeMethod = "GOR",
predTime = seq(0, 10, by = 0.5)
)
print(result_cif$Plot$Plot_InputModel) # Plot for the specified model
print(result_cif$CIF$CIFAdjusted) # Adjusted CIF values
# Fit a Fine-Gray model for competing risks
result_fg <- FineGray_Model(
CovarNames = c("Covar1", "Covar2", 'Covar3'),
Failcode = 1,
RiskNames = c("Event1", "Event2")
)
print(result_fg$Results)  # Summary of the Fine-Gray model
print(result_fg$Plot) # Plot of the CIFs

# Clean up memory
Cleanup()

}
\references{
\itemize{
\item Jeong, J.-H., & Fine, J. (2006). Direct parametric inference for the cumulative incidence function. \emph{Applied Statistics}, 55(2), 187-200. \cr
\item Jeong, J.-H., & Fine, J. (2007). Parametric regression on cumulative incidence function. \emph{Biostatistics}, 8(2), 184-196.
}
}
\seealso{
\link{Initialize}, \link{LogLike1}, \link{compute_grad}, \link{compute_hessian}, \link{estimate_parameters_GOR},
\link{estimate_parameters_POM}, \link{estimate_parameters_PHM}, \link{CIF_res1}, \link{CIF_Figs},
\link{Cmpp_CIF}, \link{FineGray_Model}, \link{bootstrap_variance}, \link{GetData}, \link{Cleanup}
}
\author{
Habib Ezzatabadipour \email{habibezati@outlook.com}
}
\keyword{cumulative}
\keyword{incidence}
\keyword{parametric}
\keyword{regression}
\keyword{risks}
\keyword{survival}
