% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gl.grm.network.r
\name{gl.grm.network}
\alias{gl.grm.network}
\title{Represents a genomic relationship matrix (GRM) as a network}
\usage{
gl.grm.network(
  G,
  x,
  method = "fr",
  node.size = 8,
  node.label = TRUE,
  node.label.size = 2,
  node.label.color = "black",
  node.shape = NULL,
  link.color = NULL,
  link.size = 2,
  relatedness_factor = 0.125,
  title = "Network based on a genomic relationship matrix",
  palette_discrete = gl.select.colors(x, library = "brewer", palette = "PuOr", ncolors =
    nPop(x), verbose = 0),
  plot.dir = NULL,
  plot.file = NULL,
  verbose = NULL
)
}
\arguments{
\item{G}{A genomic relationship matrix (GRM) generated by
\code{\link{gl.grm}} [required].}

\item{x}{A genlight object from which the G matrix was generated [required].}

\item{method}{One of 'fr', 'kk', 'gh' or 'mds' [default 'fr'].}

\item{node.size}{Size of the symbols for the network nodes [default 8].}

\item{node.label}{TRUE to display node labels [default TRUE].}

\item{node.label.size}{Size of the node labels [default 3].}

\item{node.label.color}{Color of the text of the node labels
[default 'black'].}

\item{node.shape}{Optionally provide a vector of nPop shapes
(run gl.select.shapes() for shape options) [default NULL].}

\item{link.color}{Colors for links [default gl.select.colors].}

\item{link.size}{Size of the links [default 2].}

\item{relatedness_factor}{Factor of relatedness [default 0.125].}

\item{title}{Title for the plot
[default 'Network based on genomic relationship matrix'].}

\item{palette_discrete}{A discrete set of colors
with as many colors as there are populations in the dataset
[default NULL].}

\item{plot.dir}{Directory to save the plot RDS files [default as specified
by the global working directory or tempdir()]}

\item{plot.file}{Name for the RDS binary file to save (base name only, exclude extension) [default NULL]}

\item{verbose}{Verbosity: 0, silent or fatal errors; 1, begin and end; 2,
progress log ; 3, progress and results summary; 5, full report
[default 2 or as specified using gl.set.verbosity].}
}
\value{
A network plot showing relatedness between individuals
}
\description{
This script takes a G matrix generated by \code{\link{gl.grm}} and represents
the relationship among the specimens as a network diagram. In order to use
this script, a decision is required on a threshold for relatedness to be
represented as link in the network, and on the layout used to create the
diagram.
}
\details{
The gl.grm.network function creates a network diagram that represents 
genetic relationships among individuals in a dataset using a Genomic 
Relationship Matrix (GRM). The GRM is generated by the gl.grm function, 
which utilizes the A.mat function from the rrBLUP package. This method 
follows the approach developed by Endelman and Jannink (2012).

The GRM quantifies the additive genetic relationships between individuals 
based on genome-wide SNP data. It provides an estimate of the actual genetic 
similarity — known as realized relatedness— between individuals by measuring 
how much of their genome they share identical by descent (IBD).

Two alleles are Identical by State (IBS) if they are the same in state, 
regardless of whether they come from a common ancestor. Two alleles are 
Identical by Descent (IBD) if they are inherited from a common ancestor. 
While IBS does not necessarily imply IBD, using high-density SNP data 
improves the estimation of IBD probabilities from IBS measures.

The off-Diagonal elements of the GRM represent twice the kinship coefficient 
between pairs of individuals. The kinship coefficient is the probability 
that a randomly selected allele from each individual at the same locus is 
IBD. Diagonal elements represent one plus twice the inbreeding coefficient 
of each individual. The inbreeding coefficient is the probability that both 
alleles at a random locus within an individual are IBD.

Choosing meaningful thresholds to represent relationships between 
individuals can be challenging because kinship and inbreeding coefficients 
re relative measures. To standardize the GRM and facilitate interpretation, 
the function adjusts the matrix through the following steps:

1. Centering Inbreeding Coefficients: Subtract 1 from the mean of the 
diagonal elements to calculate the average inbreeding coefficient. This 
centers the inbreeding coefficients around zero, providing a reference point 
relative to the population's average inbreeding level.

2. Calculating Kinship Coefficients: Divide the off-diagonal elements by 2 
to obtain the kinship coefficients. This conversion reflects the probability 
of sharing alleles IBD between pairs of individuals.

3. Centering Kinship Coefficients: Subtract the adjusted mean inbreeding 
coefficient (from step 1) from each kinship coefficient (from step 2). This 
centers the kinship coefficients relative to the population average, 
allowing for meaningful comparisons.

This adjustment method aligns with the approach used by Goudet et al. (2018), 
enabling the relationships to be interpreted in the context of the overall 
genetic relatedness within the population.

Below is a table modified from Speed & Balding (2015) showing kinship values,
 and their confidence intervals (CI), for different relationships that could
 be used to guide the choosing of the relatedness threshold in the function.

|Relationship|Kinship|95% CI|
|Identical twins/clones/same individual     | 0.5     |        -         |

|Sibling/Parent-Offspring                   | 0.25    |    (0.204, 0.296)|

|Half-sibling                               | 0.125   |    (0.092, 0.158)|

|First cousin                               | 0.062   |    (0.038, 0.089)|

|Half-cousin                                | 0.031   |    (0.012, 0.055)|

|Second cousin                              | 0.016   |    (0.004, 0.031)|

|Half-second cousin                         | 0.008   |    (0.001, 0.020)|

|Third cousin                               | 0.004   |    (0.000, 0.012)|

|Unrelated                                  | 0       |        -         |

Four layout options are implemented in this function:
\itemize{
\item 'fr' Fruchterman-Reingold layout  \link[igraph]{layout_with_fr}
(package igraph)
\item 'kk' Kamada-Kawai layout \link[igraph]{layout_with_kk} (package igraph)
\item 'gh' Graphopt layout \link[igraph]{layout_with_graphopt}
(package igraph)
\item 'mds' Multidimensional scaling layout \link[igraph]{layout_with_mds}
(package igraph)
}
}
\examples{
if (requireNamespace("igraph", quietly = TRUE) & requireNamespace("rrBLUP",
  quietly = TRUE
) & requireNamespace("fields", quietly = TRUE)) {
  t1 <- possums.gl
  # filtering on call rate
  t1 <- gl.filter.callrate(t1)
  t1 <- gl.subsample.loc(t1, n = 100)
  # relatedness matrix
  res <- gl.grm(t1, plotheatmap = FALSE)
  # relatedness network
  res2 <- gl.grm.network(res, t1, relatedness_factor = 0.125)
}
}
\references{
\itemize{
\item Endelman, J. B. , Jannink, J.-L. (2012). Shrinkage estimation of the
realized relationship matrix. G3: Genes, Genomics, Genetics 2, 1405.
\item Goudet, J., Kay, T., & Weir, B. S. (2018). How to estimate kinship.
Molecular Ecology, 27(20), 4121-4135.
\item Speed, D., & Balding, D. J. (2015). Relatedness in the post-genomic era:
is it still useful?. Nature Reviews Genetics, 16(1), 33-44.
 }
}
\seealso{
\code{\link{gl.grm}}

Other inbreeding functions: 
\code{\link{gl.grm}()}
}
\author{
Custodian: Arthur Georges -- Post to
\url{https://groups.google.com/d/forum/dartr}
}
\concept{inbreeding functions}
