#' Print a 'degross' object.
#'
#' @description Print a summary of the information contained in a \link{degross.object} generated by \code{\link{degross}} for density estimation from tabulated frequency and central moment data.
#'
#' @usage
#' \method{print}{degross}(x, ...)
#'
#' @param x A \link{degross.object} generated by \link{degross}.
#' @param ... Possible additional printing options.
#'
#' @return Print information on the fitted density corresponding to the \link{degross.object} \code{x}: the estimated central moments within each class (or big bin) are printed with global fit statistics. A summary of the observed data is also provided: it includes the total sample size, the numbers of small and big bins with their limits in addition to the number of B-splines used for density estimation with \code{\link{degross}}.
#'
#' @author Philippe Lambert \email{p.lambert@uliege.be}
#' @references
#' Lambert, P. (2021) Moment-based density and risk estimation from grouped summary statistics. arXiv:2107.03883.
#'
#' @seealso \code{\link{degross}}, \code{\link{plot.degross}}.
#'
#' @examples
#' sim = simDegrossData(n=3500, plotting=TRUE)
#' obj.data = degrossData(Big.bins=sim$Big.bins, freq.j=sim$freq.j, m.j=sim$m.j)
#'
#' ## Estimate the density underlying the grouped data
#' obj.fit = degross(obj.data)
#' print(obj.fit)
#'
#' @export
print.degross = function(x,...){
    print(x$degross.data)
    cat("\nFitted moments of order 1 to 4:\n")
    print(x$M.j[,1:4])
    temp = with(x,round(c(edf=edf,aic=aic,bic=bic,log.evidence=log.evidence),1))
    cat("\nGlobal fit statistics:\n")
    print(temp)
}
