#_______________________________________________________________________________

# Create the base readsData class
#_______________________________________________________________________________
#' @importFrom methods setClass setGeneric setMethod new validObject
NULL

#' An S4 class to represent summarised counts and the output of Bayesian
#' inference.
#'
#' @slot counts A positive integer matrix containing summarised counts for each
#'  genomic event (genes, exons, transcripts, etc)  in the two conditions, A and
#'  B.
#' @slot replicates A list containing the indices of the columns that belong to
#' each  of the two experimental conditions, A and B. It is defaulted to A =
#' 1:(n/2), B = (n/2+1):n for an \eqn{m} by \eqn{n} matrix.
#' @slot geneNames A character vector containing the names of the genomic event.
#'  It is appropriately defaulted to names of  the matrix.
#' @slot initValues A list containing initial values for each parameter.
#' Defaulted to \code{N_A} = rep(1, nrow(counts)), \code{N_B} = rep(1,
#' nrow(counts)), \code{p}= 0.0001, \code{f} = 0.01.
#' @slot stepSizes A list containing step sizes for sampling each parameter.
#' Defaulted to \code{stepsizeN_A} = rep(1, nrow(counts)), \code{stepsizeN_B} =
#' rep(1, nrow(counts)), \code{stepsize_p}= 1e3, \code{stepsize_f} = 5e7
#' @slot output A list containing the  samples for each parameter which are
#' generated by Bayesian inference. It can only be  filled
#' inside the results function.
#' @name readsData
readsData <- setClass("readsData",
    slots = c(# Define the slots
    counts = "matrix",
    geneNames = "character",
    replicates   = "list",
    initValues   = "list",
    stepSizes   = "list",
    output = "list"
  )
)


#Initializing
setMethod ("initialize", signature  = "readsData",
           definition = function (.Object,

                                  counts,
                                  geneNames = rownames(counts),
                                  replicates = list(A = 1 : (ncol(counts) / 2),B= (ncol(counts) / 2 + 1) : ncol(counts)),
                                  stepSizes = list(stepsizeN_A = rep(1, nrow(counts)),
                                                    stepsizeN_B = rep(1, nrow(counts)),
                                                    stepsize_p = 5e+07, stepsize_f = 1e3),
                                  initValues = list(N_A = rep(1, nrow(counts)),
            N_B = rep(1, nrow(counts)), p = 0.0001, f = 0.01)){
             .Object@counts <- counts
             .Object@replicates <- replicates
             .Object@stepSizes <- stepSizes
             .Object@geneNames <- geneNames
             .Object@initValues <- initValues
             validObject(.Object)
             return (.Object)
           })

validity <- function(object){
  if( nrow(object@counts) != length(object@geneNames) ){
    return("reads and geneNamess don't match")
  }
  else if(sum(object@counts < 0) > 0){
    return("counts cannot be negative")
  }
  else return(TRUE)
}

setValidity( "readsData", validity)
#_______________________________________________________________________________

#initValues
#_______________________________________________________________________________
#generic for setting the initValues slot

#' Generic  for altering the  initValues slot
#'
#' Updates the value of the initValues slot for the readsData object supplied.
#'
#' @param object a readsData object
#' @param initval A list of initial values for each of the  parameters.
#' @return The same readsData object with the initValues slot updated.
#' @examples
#' RD <- new("readsData", counts = ERCC)
#' initvals <- list(N_A = rep(2, 92), N_B = rep(1.5, 92), p = 0.0005, f = 0.03)
#' RD <- setInitValues(RD, initvals)
#' RD@initValues
#' @export
setGeneric(name = "setInitValues",
           def = function(object, initval){
             standardGeneric("setInitValues")
           }
)

#' @describeIn setInitValues  Alters the value of the initValues slot of
#'  a readsData object.

setMethod(f = "setInitValues",
          signature = "readsData",
          definition = function(object, initval){
            object@initValues <- initval
            return(object)
          }
)
#_______________________________________________________________________________

#stepSizes
#_______________________________________________________________________________
#generic for setting the stepSizes slot

#' Generic  for altering the stepSizes slot.
#'
#' Updates the value of the stepSizes slot for the readsData object supplied.
#'
#' @param object a readsData object
#' @param stepSizesval A list of step sizes for each of  the  parameters.
#' @return The same readsData object with the stepSizes slot updated.
#' @examples
#' RD <- new("readsData", counts = ERCC)
#' ss <- list(N_A = rep(2,  92), N_B = rep(1.5, 92), p = 3e5, f = 3.5e7)
#' RD <- setStepSizes(RD, ss)
#' RD@stepSizes
#' @export

setGeneric(name =  "setStepSizes",
           def =  function(object, stepSizesval)
           {
             standardGeneric("setStepSizes")
           }
)

#' @describeIn setStepSizes  Alters the value of the stepSizes slot of
#'  a readsData object.

setMethod(f =  "setStepSizes",
          signature =  "readsData",
          definition =  function(object, stepSizesval)
          {
            object@stepSizes <- stepSizesval
            return(object)
          }
)
#_______________________________________________________________________________

#replicates
#_______________________________________________________________________________
#method for setting the replicates slot

#' Generic  for the altering setReplicates slot.
#'
#' Updates the value of the replicates slot for the readsData object supplied.
#'
#' @param object a readsData object
#' @param repsval A list  of column indices for the samples in each condition.
#' @return The same readsData object with the replicates slot updated.
#' @examples
#' RD <- new("readsData", counts = ERCC)
#' reps <- list(A = c(2,4,5,3,10),B = c(9,7,1,8,6))
#' RD <- setReplicates(RD, reps)
#' RD@replicates
#'
#' @export
setGeneric(name =  "setReplicates",
           def =  function(object, repsval)
           {
             standardGeneric("setReplicates")
           }
)
#' @describeIn setReplicates  Alters the value of the replicates slot of
#'  a readsData object.
setMethod(f =  "setReplicates",
          signature =  "readsData",
          definition =  function(object, repsval)
          {
            object@replicates <- repsval
            return(object)
          }
)

#_______________________________________________________________________________

#setOutput
#_______________________________________________________________________________
#method for setting the ouput slot

#' Generic  for the altering output slot.
#'
#' Updates the value of the output slot for the readsData object supplied.
#'
#' @param object a readsData object.
#' @param outval A list of the output from Bayesian inference.
#' @return The same readsData object with the output slot updated.
#'
setGeneric(name =  "setOutput",
           def =  function(object, outval)
           {
             standardGeneric("setOutput")
           }
)

#' @describeIn setOutput  Alters the value of the output slot of
#'  the readsData object.
#' @keywords internal

setMethod(f =  "setOutput",
          signature =  "readsData",
          definition =  function(object, outval)
          {
            object@output <- outval
            return(object)
          }
)
