% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/add_power_pi.r
\name{add_power_pi}
\alias{add_power_pi}
\title{Add Bayesian posterior predictive intervals for power estimates}
\usage{
add_power_pi(x, future_nsims = NULL, pi_level = 0.95, prior = c(1, 1))
}
\arguments{
\item{x}{(data.frame)\cr
A data frame returned by \code{\link[=power]{power()}}, containing columns \code{power} and \code{nsims}.}

\item{future_nsims}{(Scalar integer or \code{NULL}: \code{NULL}; \verb{[2, Inf)})\cr
Number of simulated data sets in the future power estimate study.
If \code{NULL} (default), uses the same number as the original study (\code{nsims}).}

\item{pi_level}{(Scalar numeric: \code{0.95}; \verb{(0,1)})\cr
The posterior predictive interval level.}

\item{prior}{(Numeric vector of length 2: \code{c(1, 1)}; each \verb{(0, Inf)})\cr
Parameters \eqn{(\alpha, \beta)} for the Beta prior on true power.
Default \code{c(1, 1)} is the uniform prior.
Use \code{c(0.5, 0.5)} for the Jeffreys prior.}
}
\value{
The input data frame with additional columns:
\tabular{ll}{
Name \tab Description \cr
\code{power_pi_mean}  \tab Predictive mean of future power estimate. \cr
\code{power_pi_lower} \tab Lower bound of posterior predictive interval. \cr
\code{power_pi_upper} \tab Upper bound of posterior predictive interval.
}
and added attribute \code{"pi_info"} containing the method description, method name, level, prior values, and future simulation count.
}
\description{
Calculates and adds Bayesian posterior predictive intervals for power estimates in objects returned by \code{\link[=power]{power()}}.
The posterior predictive interval quantifies the expected range of power estimates from a future simulation study.
}
\details{
Power estimation via simulation is a binomial proportion problem.
The posterior predictive interval answers: "If I run a new simulation study with \eqn{m} simulations, what range of power estimates might I observe?"

Let \eqn{\pi} denote the true power value, \eqn{\hat{\pi} = x/n} denote the observed power value, \eqn{n} denote the number of simulations, and \eqn{x = \text{round}(\hat{\pi} \cdot n)} denote the number of rejections.
With a \eqn{\text{Beta}(\alpha, \beta)} prior on the true power \eqn{\pi}, the posterior after observing \eqn{x} successes in \eqn{n} trials is:

\deqn{
  \pi \mid X = x \sim \text{Beta}(\alpha + x, \beta + n - x).
}

The posterior predictive distribution for \eqn{Y}, the number of successes in a future study with \eqn{m} trials, is Beta-Binomial:

\deqn{
  Y \mid X = x \sim \text{BetaBinomial}(m, \alpha + x, \beta + n - x).
}

The posterior predictive interval is constructed from quantiles of this distribution, expressed as proportions \eqn{Y/m}.

The posterior predictive mean and variance of \eqn{\hat{\pi}_{\text{new}} = Y/m} are:
\deqn{
\begin{aligned}
  E[\hat{\pi}_{\text{new}} \mid X = x] &= \frac{\alpha + x}{\alpha + \beta + n} \\
  \text{Var}[\hat{\pi}_{\text{new}} \mid X = x]
  &= \frac
      {(\alpha + x)(\beta + n - x)(\alpha + \beta + n + m)}
      {m (\alpha + \beta + n)^{2} (\alpha + \beta + n + 1)}.
\end{aligned}
}
\subsection{Argument \code{future_nsims}}{

The argument \code{future_nsims} allows you to estimate prediction interval bounds for a hypothetical future study with different number of simulations.
Note that a small initial number for \code{nsims} results in substantial uncertainty about the true power.
A correspondingly large number of future simulations \code{future_nsims} will more precisely estimate the true power, but the past large uncertainty is still carried forward.
Therefore you still need an adequate number of simulations \code{nsims} in the original study, not just more in the replication \code{future_nsims}, to ensure narrow prediction intervals.
}

\subsection{Approximate parametric tests}{

When power is computed using approximate parametric tests (see \code{\link[=simulated]{simulated()}}), the power estimate and confidence/prediction intervals apply to the Monte Carlo test power \eqn{\mu_K = P(\hat{p} \leq \alpha)} rather than the exact test power \eqn{\pi = P(p \leq \alpha)}.
These quantities converge as the number of datasets simulated under the null hypothesis \eqn{K} increases.
The minimum observable p-value is \eqn{1/(K+1)}, so \eqn{K > 1/\alpha - 1} is required to observe any rejections.
For practical accuracy, we recommend choosing \eqn{\text{max}(5000, K \gg 1/\alpha - 1)} for most scenarios.
For example, if \eqn{\alpha = 0.05}, use \code{simulated(nsims = 5000)}.
}
}
\examples{
#----------------------------------------------------------------------------
# add_power_pi() examples
#----------------------------------------------------------------------------
library(depower)

set.seed(1234)
x <- sim_nb(
  n1 = 10,
  mean1 = 10,
  ratio = c(1.4, 1.6),
  dispersion1 = 2,
  nsims = 200
) |>
  power(wald_test_nb())

# Add posterior predictive intervals
# default: predict for same number of simulations
add_power_pi(x)

# Compare posterior predictive interval width across different future
# study sizes
add_power_pi(x, future_nsims = 100)  # wider
add_power_pi(x, future_nsims = 1000) # narrower

# Use Jeffreys prior instead of uniform
add_power_pi(x, prior = c(0.5, 0.5))

# Plot with shaded region for prediction interval of the power estimate.
add_power_pi(x) |>
  plot()

}
\references{
\insertRef{gelman_2013}{depower}
}
\seealso{
\code{\link[=power]{power()}},
\code{\link[=eval_power_pi]{eval_power_pi()}},
\code{\link[=add_power_ci]{add_power_ci()}}
}
