##' Pretty print p-values
##'
##' Pretty print p-values with org-mode syntax for bold face
##' @title prettyPvalues
##' @param p_vals numeric
##' @param digits how many digits to print
##' @param signiflev print in bold face if smaller than this
##' @param lhs character.  left hand side of the printed 'formula'.
##'   Defaults to NULL, in which case neither lhs nor lhs_sep will be printed.
##' @param lhs_sep character.  Separator between lhs and pretty
##'   pvalue.  Defaults to "<".
##' @param orgbold boolean.  Surround significant p values by '*'?.
##'   Defaults to TRUE.
##' @param roundonly boolean.  if TRUE, do neiter prepend any number with '<'
##'                  nor do enclose any number with '*'.
##'                  Defaults to FALSE
##' @return character vector of pretty printed p-values
##' @author Andreas Leha
##' @export
prettyPvalues <- function(p_vals, digits=5, signiflev=0.05, lhs = NULL, lhs_sep = "=", orgbold = TRUE, roundonly=FALSE) {
  idx_bold <- p_vals <= signiflev
  idx_too_small <- p_vals < (1 * 10^(-digits))

  ## cope with NAs
  idx_bold[is.na(p_vals)] <- FALSE
  idx_too_small[is.na(p_vals)] <- FALSE

  pp_vals <- round(p_vals, digits=digits)
  pp_vals <- format(pp_vals, scientific=FALSE, digits=digits)
  pp_vals <- sapply(pp_vals, function(pval) {
    while (nchar(pval) < (digits+2)) {
      if (length(grep(".", pval, fixed=TRUE))>0) {
        pval <- paste0(pval, "0")
      } else {
        pval <- paste0(pval, ".0")
      }
    }
    pval
  })
  names(pp_vals) <- NULL

  pp_vals[grep("NA", pp_vals)] <- NA

  if (!roundonly) {
    pp_vals[idx_too_small] <- paste0("< 0.",
                                     paste(rep("0", digits-1), collapse=""),
                                     "1")
    if (!is.null(lhs)) {
      pp_vals[idx_too_small]  <- paste(lhs,          pp_vals[idx_too_small])
      pp_vals[!idx_too_small] <- paste(lhs, lhs_sep, pp_vals[!idx_too_small])
    }

    if (orgbold) {
      pp_vals[idx_bold] <- paste0("*", pp_vals[idx_bold], "*")
    }
  } else {
    if (!is.null(lhs)) {
      pp_vals[!idx_too_small] <- paste(lhs, lhs_sep, pp_vals[!idx_too_small])
    }
  }

  names(pp_vals) <- names(p_vals)

  return(pp_vals)
}
