% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/customBetaDiv.R
\name{customBetaDiv}
\alias{customBetaDiv}
\title{Custom beta diversity metrics}
\usage{
customBetaDiv(
  x,
  fun,
  radius,
  minTaxCount = 1,
  focalCoord = NULL,
  metricName = "custom_metric"
)
}
\arguments{
\item{x}{object of class \code{epmGrid}}

\item{fun}{a function to apply to grid cell neighborhoods (see details)}

\item{radius}{Radius of the moving window in map units.}

\item{minTaxCount}{the minimum number of taxa needed to apply the function.
For instance, should the function be applied to gridcells with just 1
taxon?}

\item{focalCoord}{vector of x and y coordinate, see details}

\item{metricName}{the name you would like to attach to the output}
}
\value{
object of class \code{epmGrid}, or list of \code{epmGrid} objects
}
\description{
Define your own function for summarizing information across a
 moving window of grid cells.
}
\details{
This function will identify the neighbors of every cell and will
apply the specified function to those sets of cell neighborhoods. 

 The custom function should have just one input: a list of taxon names, where
 the list will represent a set of grid cells (focal cell + neighboring cells).

 However, if a set of focal coordinates is provided, then rather than apply the
 function to each neighborhood of cells, the function should have two inputs: the
 focal cell and another cell, and that function will be applied to every pair 
 defined by the focal cell and another cell. See examples.

 Within the function call, the trait data already attached to the epmGrid object 
 must be referred to as dat, and the phylogenetic tree already attached to the 
 epmGrid must be referred to as phylo.\cr 

 If the input epmGrid object contains a set of trees, then this function will
	 be applied, using each tree in turn, and will return a list of results. This
 list can then be passed to \code{\link{summarizeEpmGridList}} to be summarized.
 
 See examples below.
}
\examples{
\donttest{
tamiasEPM <- addPhylo(tamiasEPM, tamiasTree)
tamiasEPM <- addTraits(tamiasEPM, tamiasTraits)

# An example using a multivariate dataset
## For each focal cell + neighbors, calculate the morphological standard deviation
## per grid cell and return the standard deviation.
f <- function(cellList) {
	vec <- numeric(length(cellList))
	for (i in 1:length(cellList)) {
		vec[[i]] <- max(dist(dat[cellList[[i]], ]))
	}
	return(sd(vec, na.rm = TRUE))	
}

xx <- customBetaDiv(tamiasEPM, fun = f, radius = 70000, minTaxCount = 2, metricName = 'maxdist')


# An example using only the phylogeny.
## Calculate standard deviation of phylogenetic diversity across cell neighborhood.
f <- function(cellList) {
	vec <- numeric(length(cellList))
	for (i in 1:length(cellList)) {
		vec[[i]] <- faithPD(phylo, cellList[[i]])
	}
	return(sd(vec, na.rm = TRUE))	
}

xx <- customBetaDiv(tamiasEPM, fun = f, radius = 70000, minTaxCount = 1, metricName = 'faithPD')



# an example that involves both morphological and phylogenetic data
## nonsensical, but for illustrative purposes:
## ratio of Faith's phylogenetic diversity to morphological range
## the standard deviation of this measure across grid cells 
## in a neighborhood.
f <- function(cellList) {
	vec <- numeric(length(cellList))
	for (i in 1:length(cellList)) {
		vec[[i]] <- faithPD(phylo, cellList[[i]]) / 
		max(dist(dat[cellList[[i]], ]))
	}
	return(sd(vec, na.rm = TRUE))	
}

xx <- customBetaDiv(tamiasEPM, fun = f, radius = 70000, minTaxCount = 2, 
  metricName = 'ratio_PD_maxdist')




# from a focal coordinate to all other sites
## Here, the function has 2 inputs.
## Example: calculate the per grid cell mean and take the distance.
f <- function(focalCell, otherCell) {
	x1 <- colMeans(dat[focalCell, ])
	x2 <- colMeans(dat[otherCell, ])
	return(as.matrix(dist(rbind(x1, x2)))[1,2])
}

xx <- customBetaDiv(tamiasEPM, fun = f, radius = 70000, minTaxCount = 1,
 focalCoord = c(-1413764, 573610.8), metricName = 'meandist')



# Example involving a set of trees
tamiasEPM <- addPhylo(tamiasEPM, tamiasTreeSet, replace = TRUE)

## Calculate standard deviation of phylogenetic diversity across cell
## neighborhood.
f <- function(cellList) {
	vec <- numeric(length(cellList))
	for (i in 1:length(cellList)) {
		vec[[i]] <- faithPD(phylo, cellList[[i]])
	}
	return(sd(vec, na.rm = TRUE))	
}

# This time, a list of sf objects will be returned, one for each input tree.
xx <- customBetaDiv(tamiasEPM, fun = f, radius = 70000, minTaxCount = 1,
 metricName = 'faithPD')



# also works with square grid cells
tamiasEPM2 <- createEPMgrid(tamiasPolyList, resolution = 50000,
	cellType = 'square', method = 'centroid')
tamiasEPM2 <- addPhylo(tamiasEPM2, tamiasTree)
tamiasEPM2 <- addTraits(tamiasEPM2, tamiasTraits)


f <- function(cellList) {
	vec <- numeric(length(cellList))
	for (i in 1:length(cellList)) {
		vec[[i]] <- faithPD(phylo, cellList[[i]]) / 
		max(dist(dat[cellList[[i]], ]))
	}
	return(sd(vec, na.rm = TRUE))	
}

xx <- customBetaDiv(tamiasEPM2, fun = f, radius = 70000, minTaxCount = 2,
 metricName = 'ratio_PD_maxdist')
}


}
\author{
Pascal Title
}
