% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gips_class.R
\name{plot.gips}
\alias{plot.gips}
\title{Plot optimized matrix or optimization \code{gips} object}
\usage{
\method{plot}{gips}(
  x,
  type = NA,
  logarithmic_y = TRUE,
  logarithmic_x = FALSE,
  color = NULL,
  title_text = "Convergence plot",
  xlabel = NULL,
  ylabel = NULL,
  show_legend = TRUE,
  ylim = NULL,
  xlim = NULL,
  ...
)
}
\arguments{
\item{x}{Object of a \code{gips} class.}

\item{type}{A character vector of length 1. One of
\code{c("heatmap", "MLE", "best", "all", "both", "n0", "block_heatmap")}:
\itemize{
\item \code{"heatmap"}, \code{"MLE"} - Plots a heatmap of the Maximum Likelihood
Estimator of the covariance matrix given the permutation.
That is, the \code{S} matrix inside the \code{gips} object
projected on the permutation in the \code{gips} object.
\item \code{"best"} - Plots the line of the biggest a posteriori found over time.
\item \code{"all"} - Plots the line of a posteriori for all visited states.
\item \code{"both"} - Plots both lines from "all" and "best".
\item \code{"n0"} - Plots the line of \code{n0}s that were spotted during optimization
(only for "MH" optimization).
\item \code{"block_heatmap"} - Plots a heatmap of diagonally block representation of \code{S}.
Non-block entries (equal to 0) are white for better clarity.
For more information, see \strong{Block Decomposition - [1], Theorem 1}
section in \code{vignette("Theory", package = "gips")} or in its
\href{https://przechoj.github.io/gips/articles/Theory.html}{pkgdown page}.
}

The default value is \code{NA}, which will be changed to "heatmap" for
non-optimized \code{gips} objects and to "both" for optimized ones.
Using the default produces a warning.
All other arguments are ignored for
the \code{type = "heatmap"}, \code{type = "MLE"}, or \code{type = "block_heatmap"}.}

\item{logarithmic_y, logarithmic_x}{A boolean.
Sets the axis of the plot in logarithmic scale.}

\item{color}{Vector of colors to be used to plot lines.}

\item{title_text}{Text to be in the title of the plot.}

\item{xlabel}{Text to be on the bottom of the plot.}

\item{ylabel}{Text to be on the left of the plot.}

\item{show_legend}{A boolean. Whether or not to show a legend.}

\item{ylim}{Limits of the y axis. When \code{NULL},
the minimum, and maximum of the \code{\link[=log_posteriori_of_gips]{log_posteriori_of_gips()}} are taken.}

\item{xlim}{Limits of the x axis. When \code{NULL},
the whole optimization process is shown.}

\item{...}{Additional arguments passed to
other various elements of the plot.}
}
\value{
When \code{type} is one of \code{"best"}, \code{"all"}, \code{"both"} or \code{"n0"},
returns an invisible \code{NULL}.
When \code{type} is one of \code{"heatmap"}, \code{"MLE"} or \code{"block_heatmap"},
returns an object of class \code{ggplot}.
}
\description{
Plot the heatmap of the MAP covariance matrix estimator
or the convergence of the optimization method.
The plot depends on the \code{type} argument.
}
\examples{
require("MASS") # for mvrnorm()

perm_size <- 6
mu <- runif(6, -10, 10) # Assume we don't know the mean
sigma_matrix <- matrix(
  data = c(
    1.0, 0.8, 0.6, 0.4, 0.6, 0.8,
    0.8, 1.0, 0.8, 0.6, 0.4, 0.6,
    0.6, 0.8, 1.0, 0.8, 0.6, 0.4,
    0.4, 0.6, 0.8, 1.0, 0.8, 0.6,
    0.6, 0.4, 0.6, 0.8, 1.0, 0.8,
    0.8, 0.6, 0.4, 0.6, 0.8, 1.0
  ),
  nrow = perm_size, byrow = TRUE
) # sigma_matrix is a matrix invariant under permutation (1,2,3,4,5,6)
number_of_observations <- 13
Z <- MASS::mvrnorm(number_of_observations, mu = mu, Sigma = sigma_matrix)
S <- cov(Z) # Assume we have to estimate the mean

g <- gips(S, number_of_observations)
if (require("graphics")) {
  plot(g, type = "MLE")
}

g_map <- find_MAP(g, max_iter = 30, show_progress_bar = FALSE, optimizer = "hill_climbing")
if (require("graphics")) {
  plot(g_map, type = "both", logarithmic_x = TRUE)
}

if (require("graphics")) {
  plot(g_map, type = "MLE")
}
# Now, the output is (most likely) different because the permutation
  # `g_map[[1]]` is (most likely) not an identity permutation.

g_map_MH <- find_MAP(g, max_iter = 30, show_progress_bar = FALSE, optimizer = "MH")
if (require("graphics")) {
  plot(g_map_MH, type = "n0")
}
}
\seealso{
\itemize{
\item \code{\link[=find_MAP]{find_MAP()}} - Usually, the \code{plot.gips()}
is called on the output of \code{find_MAP()}.
\item \code{\link[=project_matrix]{project_matrix()}} - The function used with \code{type = "MLE"}.
\item \code{\link[=gips]{gips()}} - The constructor of a \code{gips} class.
The \code{gips} object is used as the \code{x} parameter.
}
}
