% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hasserls.R
\encoding{UTF-8}
\name{hasserls}
\alias{hasserls}
\title{Hasse diagram of the restricted layout structure}
\usage{
hasserls(
  datadesign,
  rand.objects,
  rand.arrows = NULL,
  randomfacsid = NULL,
  showRLS = TRUE,
  showpartialRLS = TRUE,
  showdfRLS = TRUE,
  showrandRLS = TRUE,
  check.confound.df = TRUE,
  maxlevels.df = TRUE,
  table.out = FALSE,
  equation.out = FALSE,
  pdf = FALSE,
  example = "example",
  outdir = NULL,
  hasse.font = "sans",
  produceBWPlot = FALSE,
  structural.colour = "grey",
  structural.width = 2,
  partial.colour = "orange",
  partial.width = 1.5,
  objects.colour = "mediumblue",
  df.colour = "red",
  arrow.colour = "mediumblue",
  arrow.width = 1.5,
  arrow.pos = 7.5,
  larger.fontlabelmultiplier = 1,
  middle.fontlabelmultiplier = 1,
  smaller.fontlabelmultiplier = 1
)
}
\arguments{
\item{datadesign}{A data frame, list or environment (or object coercible by \code{\link[base]{as.data.frame}} to a data frame) containing the factors in the experimental design. The data frame should \strong{only} include the factors/columns that the user wants to include in the Hasse diagram. All factors are treated as categorical. Moreover, the first two letters of factor names are used for interactions between factors so it is advised that these be unique.}

\item{rand.objects}{A character vector specifying the randomisation objects 
that define the Restricted Layout Structure (RLS).  

The vector must have the same length and order as the structural objects 
of the layout structure as produced by \code{\link[hassediagrams]{hasselayout}}. 
Each element is either \code{"NULL"} (if there is no randomisation object in 
the restricted layout structure that corresponds to the structural object) 
or the name of the randomisation object corresponding to the structural object. 
The user supplies labels for the randomisation objects

The recommended workflow is:
\enumerate{
  \item Run \code{\link[hassediagrams]{hasselayout}} to obtain the list of structural 
        objects (in the order required for \code{\link[hassediagrams]{hasserls}}) 
        and a template for the randomisation objects. 
  \item Observe the structural objects using 
  \code{hasselayout(datadesign)$str_objects}
  and create a vector of the randomisation objects potential using the
  suggested vector using  \code{hasselayout(datadesign)$rand_template}.
  \item Modify only those entries that correspond to randomisation objects 
        present in the restricted layout structure.  
}
The labels specified in \code{rand.objects} represent the labels of the 
randomisation objects on the Hasse diagram of the restricted layout structure. 
If the labels include Unicode symbols (e.g., the Kronecker product symbol), 
a Unicode-friendly font is required.}

\item{rand.arrows}{A matrix of two columns that takes integer entries. 
Each row of the matrix corresponds to one randomisation arrow on the Hasse diagram of the restricted layout structure. 
The entries in the first column contain the object(s) at the start of the randomisation arrow 
and the second column contains the object(s) at the end. 
The values correspond to the entry number for the randomisation object in \code{rand.objects}.
Therefore, any randomisation object(s) involved in the randomisation arrow(s) must first be specified in the \code{rand.objects} argument.
The randomisation arrows must point downwards, hence, in each row of the matrix the second column entry must be larger than the first column entry.}

\item{randomfacsid}{An optional vector specifying whether the factors are defined as fixed (entry = 0) or random (entry = 1). The default choice is NULL and the function automatically sets all entries to 0. The length of the vector should be equal to the number of factors in the design, i.e., the length of the vector should be equal to the number of columns of the argument \code{datadesign}.}

\item{showRLS}{logical. If \code{FALSE} then generation of the Hasse diagram of the restricted layout structure is suppressed. The default is TRUE.}

\item{showpartialRLS}{logical. If \code{FALSE} then the partial crossing between randomisation objects (using dotted connecting lines) is not illustrated
on the Hasse diagram of the restricted layout structure. The default is TRUE.}

\item{showdfRLS}{logical. If \code{FALSE} then the randomisation object label is not displayed on the Hasse diagram of the restricted layout structure. The default is TRUE.}

\item{showrandRLS}{logical. If \code{FALSE} then the randomisations are not illustrated (using arrows) on the Hasse diagram of the restricted layout structure. 
The default is TRUE. If \code{rand.arrows=NULL}, then \code{showrandRLS} defaults to FALSE.}

\item{check.confound.df}{logical. If \code{FALSE} then the check for confounded degrees of freedom is not performed. The default is TRUE.}

\item{maxlevels.df}{logical. If \code{FALSE} then the potential maximum number of levels of a generalised factor is removed from the randomisation object 
label on the Hasse diagram of the restricted layout structure. The default is TRUE.}

\item{table.out}{logical. If \code{TRUE} then a table that shows the relationships between the randomisation objects in the restricted layout structure is printed. 
The default is FALSE.}

\item{equation.out}{logical. If \code{TRUE} then a recommended mixed model to use in the statistical analysis is printed. The default is FALSE.}

\item{pdf}{logical. If \code{TRUE} then a pdf file containing the Hasse diagram of the restricted layout structure is generated. 
The default is FALSE, i.e., a pdf file is not generated.}

\item{example}{character. Filename for the pdf output file containing the Hasse diagram. The default is set to "example".}

\item{outdir}{character. Location of the pdf output file if \code{pdf=TRUE}. The default is set to \code{NULL} and in this case the pdf output file 
containing the Hasse diagram output is stored to a temporary file. To specify a permanent location this argument needs be specified.}

\item{hasse.font}{character. The name of the font family used for all text included on the Hasse diagram. 
Standard and safe font families to choose are "sans", "serif", and "mono". 
If any of the labels of the randomisation objects (as defined in the second column of \code{rand.objects} matrix) 
contain Unicode characters, a Unicode friendly font family should be selected. 
For more details on Unicode friendly family options see the Details section. 
If the font family selected fails to render, the font is automatically changed to "sans" instead.
The default is "sans".}

\item{produceBWPlot}{logical. If \code{TRUE} then the Hasse diagram will be generated in black and white format. 
The default is set to FALSE, i.e., a coloured version of the plot is produced.}

\item{structural.colour}{character. The colour of the structural lines that connect randomisation objects on the Hasse diagram. The default colour is "grey".}

\item{structural.width}{numeric. The width of the structural lines on the Hasse diagram. The default width is 2.}

\item{partial.colour}{character. The colour of the partial crossing dotted lines of the connecting randomisation objects on the Hasse diagram. 
The default colour is "orange".}

\item{partial.width}{numeric. The width of the partial crossing dotted lines on the Hasse diagram. The default width is 1.5.}

\item{objects.colour}{character. The colour of the labels of the randomisation objects on the Hasse diagram. The default colour is "mediumblue".}

\item{df.colour}{character. The colour of the degrees of the freedom labels on the Hasse diagram. The default colour is "red".}

\item{arrow.colour}{character. The colour of the randomisation arrows on the Hasse diagram. The default colour is "mediumblue".}

\item{arrow.width}{numeric. The randomisation arrows width on the Hasse diagram. The default width is 1.5.}

\item{arrow.pos}{numeric. Specifies the position of the randomisation arrows, i.e., how far the randomisation arrows will be from 
the objects they point at. The default is 7.5. A smaller number specifies longer arrows and a higher number specifies shorter arrows.}

\item{larger.fontlabelmultiplier}{numeric. The large font multiplier is the multiplier for the font used for the labels of objects on the 
Hasse diagram where there are four or less objects at that level in the diagram. The default is 1.}

\item{middle.fontlabelmultiplier}{numeric. The medium font multiplier is the multiplier for the font used for the labels of objects on the 
Hasse diagram involving a factor that is equivalent to a generalised factor. The default is 1.}

\item{smaller.fontlabelmultiplier}{numeric. The small font multiplier is the multiplier for the font used for the labels of objects on the 
Hasse diagram where there are five or more objects at that level of the diagram. The default is 1.}
}
\value{
The function \code{\link[hassediagrams]{hasserls}} returns:
1. The Hasse diagram of the restricted layout structure (if \code{showRLS = TRUE}).

2. The restricted layout structure table shows the relationships between the randomisation objects in the restricted layout structure 
(if \code{table.out=TRUE}). The individual entries in the table consist of blanks on the main diagonal and 0’s, (0)’s or 1’s elsewhere. 
If the factor (or generalised factor) corresponding to the ith row and the factor (or generalised factor) corresponding to the jth column are fully crossed, 
then a 0 is entered in the (i,j)th entry in the table. If these factors (or generalised factors) are partially crossed, or the ith row factor 
(or generalised factor) only has one level and nests the jth column factor (or generalised factor), then the (i,j)th entry is (0). 
If the ith row factor (or generalised factor) is nested within the jth column factor (or generalised factor), then a 1 is entered in the 
(i,j)th entry. If two factors (or generalised factor) are equivalent, then they share a single row and column in the table, 
where the row and column headers include both factor (or generalised factor) names, separated by an "=" sign.

3. An equation that suggests the mixed model to be fitted (if \code{equation.out=TRUE}).

4. If there are confounded degrees of freedom, a table of the structural objects and a description of the associated degrees of freedom is printed.
}
\description{
Returns a Hasse diagram of the restricted layout structure of an experimental design
}
\details{
The \code{\link[hassediagrams]{hasserls}} function generates the Hasse diagram of the restricted layout structure. 
The Hasse diagram consists of a set of randomisation objects, corresponding to the factors and generalised factors, 
and the relationships between the objects (either crossed, nested, partially crossed or equivalent), 
as defined by the structure of the experimental design and the randomisation performed, see Bate and Chatfield (2016b).

Where present, two partially crossed factors are illustrated on the diagram with a dotted line connecting them. 
This feature can be excluded using the \code{showpartialRLS} option.

The maximum number of possible levels of each generalised factor, along with the actual number present in the design 
and the "skeleton ANOVA" degrees of freedom, can be included in the randomisation object label on the Hasse diagram.

The randomisation arrows that illustrate the randomisation performed can be included on the Hasse diagram.

The \code{\link[hassediagrams]{hasserls}} function evaluates the design in order to identify if there are any 
confounded degrees of freedom across the design. It is not recommended to perform this evaluation for large designs, 
due to the potential high computational cost. This can be controlled using the \code{check.confound.df = FALSE} option. 

The rendering of Unicode symbols (e.g., u2297, u2192 for Kronecker symbol and arrow, respectively) in the Hasse diagram depends on the operating system and the font selected in hasse.font.

macOS / Linux:
Most system fonts that support Unicode work directly in plotting and PDF output without explicit registration. In many cases, the default "sans" family is sufficient for PDF rendering of these symbols.
However, for on-screen rendering usually unicode-friendly fonts like "AppleMyungjo", "Arial Unicode MS", .SF Compact, and "Noto Sans Math" are needed. 

Windows:
Base R plotting often requires explicit font registration before the font can be used. Even if the font is installed, it may not be accessible to the graphics device until registered.
Unicode-friendly fonts are "Lucida Sans Unicode", "Arial Unicode MS", "Segoe UI Symbol", "Cambria", "Noto Sans Math" and "Ebrima". 
The aforementioned fonts may not be available in your R session. The available system fonts can be printed by systemfonts::system_fonts()$family.
System available fonts can be imported by running showtext::font_import() or extrafont::font_import().
To check which fonts have been successfully imported, run showtext::fonts() or extrafont::fonts().
The Arial Unicode MS font can be downloaded from online sources.
The Noto Sans Math font can be installed using sysfonts::font_add_google("Noto Sans Math").
For Windows, fonts might not be accessible to the graphics device until registered using:
windowsFonts(LucidaSansUnicode = windowsFont("Lucida Sans Unicode"))
adapted to the font need to use.

For ease of execution in the examples below, we are using --> for the arrow symbol and (x) for the kronecker symbol.
}
\examples{
\donttest{
## NOTE TO USERS:
## In the examples below you may use Unicode symbols (e.g., "u2297 and "u2192"
## with a backslash, for the Kronecker and arrow symbols respectively),
## but we use ASCII fallbacks such as "(x)" and "-->" to ensure
## compatibility across systems.
## To render proper Unicode symbols in diagrams, update the labels manually
## and set a Unicode-friendly font via the hasse.font argument.

### Example: Asphalt concrete production (fractional factorial design)
## Obtain the structural objects from the layout structure
ls_concrete <- hasselayout(datadesign = concrete, 
                           showLS = FALSE,
                           showpartialLS = FALSE,
                           showdfLS = FALSE)
## Observe the structural objects and then use the suggested 
## template for randomisation objects
ls_concrete$str_objects
rand_spec <- ls_concrete$rand_template
## Fill in the randomisation objects that occur in the RLS
rand_spec[] <- ls_concrete$str_objects       
rand_spec[length(rand_spec)] <- "AC^AG^CC^CoT^CuT --> Run"

## Generate the Hasse diagram of the restricted layout structure
hasserls(datadesign = concrete,
         rand.objects = rand_spec,
         larger.fontlabelmultiplier = 1.6,
         smaller.fontlabelmultiplier = 1.3)


### Example: Crossover dental study
## Obtain the structural objects from the layout structure
ls_dental <- hasselayout(datadesign = dental, 
                         randomfacsid = c(0, 1, 0, 0, 0),
                         showLS = FALSE,
                         showpartialLS = FALSE,
                         showdfLS = FALSE)
## Observe the structural objects and then use the suggested 
## template for randomisation objects
ls_dental$str_objects
rand_spec <- ls_dental$rand_template
## Fill in the randomisation objects that occur in the RLS
rand_spec[c(2:5, 7, 8)] <- c("Period", "Sequence",
                             "Treatment", "Subject[Sequence]",
                             "Period (x) Sequence",
                             "Observation")
## Create a matrix for the randomisation arrows 
dental_rand_arrows <- matrix(c(3, 5, 4, 7), ncol = 2, byrow = TRUE)
## Generate the Hasse diagram of the restricted layout structure
hasserls(datadesign = dental,
         rand.objects = rand_spec,
         rand.arrows = dental_rand_arrows,
         randomfacsid = c(0, 1, 0, 0, 0),
         larger.fontlabelmultiplier = 1.6,
         arrow.pos = 15)


## Conditionally run examples requiring 'dae'
if (requireNamespace("dae", quietly = TRUE)) {
  data(BIBDWheat.dat, package = "dae")
  BIBDWheat <- BIBDWheat.dat[, -4]
  BIBDWheat$Plots <- 1:30
ls_BIBDWheat <- hasselayout(datadesign = BIBDWheat,
                            showLS = FALSE,
                            showpartialLS = FALSE,
                            showdfLS = FALSE)
## Observe the structural objects and then use the suggested 
## template for randomisation objects
ls_BIBDWheat$str_objects
rand_spec <- ls_BIBDWheat$rand_template
## Fill in the randomisation objects that occur in the RLS
rand_spec[c(2:4)] <- c("Blocks", "Varieties", "Plot[Block]")
## Create a matrix for the randomisation arrows 
IBDWheat_rand_arrows <- matrix(c(3, 4), ncol = 2, byrow = TRUE)
## Generate the Hasse diagram of the restricted layout structure
hasserls(datadesign = BIBDWheat,
         rand.objects = rand_spec,
         rand.arrows = IBDWheat_rand_arrows,
         equation.out = TRUE)


data(Fac4Proc.dat, package = "dae")
Fac4Proc <- Fac4Proc.dat[, -6]
## Obtain the structural objects from the layout structure
ls_Fac4Proc <- hasselayout(datadesign = Fac4Proc,
                           showLS = FALSE,
                           showpartialLS = FALSE,
                           showdfLS = FALSE)
## Observe the structural objects and then use the suggested 
## template for randomisation objects
ls_Fac4Proc$str_objects
rand_spec <- ls_Fac4Proc$rand_template
## Fill in the randomisation objects that occur in the RLS
rand_spec[] <- ls_Fac4Proc$str_objects       
rand_spec[length(rand_spec)] <- "Catal^Conc^Press^Temp --> Run"
## Generate the Hasse diagram of the restricted layout structure
hasserls(datadesign = Fac4Proc,
         rand.objects = rand_spec,
         showpartialRLS = FALSE,
         smaller.fontlabelmultiplier = 2)

} else {
  message("Install package 'dae' to run the final examples.")
}
}
}
\references{
Bate, S.T. and Chatfield, M.J. (2016a), Identifying the structure of the experimental design. Journal of Quality Technology, 48, 343-364.

Bate, S.T. and Chatfield, M.J. (2016b), Using the structure of the experimental design and the randomization to construct a mixed model. Journal of Quality Technology, 48, 365-387.

Box, G.E.P., Hunter, J.S., and Hunter, W.G., (1978), Statistics for Experimenters. Wiley.

Joshi, D.D. (1987), Linear Estimation and Design of Experiments. Wiley Eastern, New Delhi.

Williams, E.R., Matheson, A.C. and Harwood, C.E. (2002), Experimental design and analysis for tree improvement. 2nd edition. CSIRO, Melbourne, Australia.
}
\author{
Damianos Michaelides, Simon Bate, and Marion Chatfield
}
