\name{coreVennTree}
\alias{coreVennTree}
\title{
Phylogeny with Branches Colored Based on Venn Diagram
}
\description{
Plots a phylogeny with branches colored based on the compartment that they are associated with in the Venn diagram generated by the \code{corePhyloVenn} function.
}
\usage{
coreVennTree(x, grouping, core_fraction = 0.5, ab_threshold1 = 0,
ab_threshold2 = 0, ab_threshold3 = 0, mode='branch',
selection='basic', max_tax = NULL,increase_cutoff = 2, initial_branches = NULL,
rooted=TRUE, ordered_groups=NULL,branch_color=NULL,remove_zeros=TRUE,
plot.chronogram=FALSE,branch.width = 4, label.tips = FALSE, scaled = FALSE)
}
\arguments{
\item{x}{
  (Required) Microbial community data. This must be in the form of a phyloseq object and must contain, at a minimum, an OTU abundance table and a phylogeny.
}
\item{grouping}{
(Required) A vector specifying which samples belong to which habitat type.
}
\item{core_fraction}{
The fraction of samples that a microbial taxon must be found in to be considered part of the 'core' microbiome. This variable is only used when selection = 'basic' and is ignored when selection = 'shade'. The default value is 0.5.
}
\item{ab_threshold1}{
The threshold for mean relative abundance across all samples. This variable is only used when selection = 'basic' and is ignored when selection = 'shade'. The default value is 0.
}
\item{ab_threshold2}{
The threshold for maximum relative abundance in any sample. This variable is only used when selection = 'basic' and is ignored when selection = 'shade'. The default value is 0.
}
\item{ab_threshold3}{
The threshold for minimum relative abundance across all samples. This variable is only used when selection = 'basic' and is ignored when selection = 'shade'. The default value is 0.
}
\item{mode}{
Whether to build a tip-based ('tip') or a branch-based ('branch') phylogeny. The default is 'branch'.
}
\item{selection}{
Whether to use thresholds ('basic') or the Shade and Stopnisek method ('shade') to define the core community. The default is 'basic'.
}
\item{max_tax}{
The maximum number of branches to add sequentially, as a percentage of the total branches when using the Shade and Stopnisek method. This variable is only used when selection = 'shade' and is ignored when selection = 'basic'.
}
\item{increase_cutoff}{
The threshold for the percent increase in beta diversity used to identify the taxon at which point adding more taxa yields diminishing returns in explanatory power. This variable is only used when selection = 'shade' and is ignored when selection = 'basic'.
}
\item{initial_branches}{
The number of branches to include prior to testing for increases in beta diversity. The default is to use all branches that are present in every sample and to begin testing branches that are missing from at least one sample. This variable is only used when selection = 'shade' and is ignored when selection = 'basic'.
}
\item{rooted}{
Whether to include the root of the phylogeny. The default is TRUE, meaning that the root is necessarily included in all phylogenies. This requires that the input tree be rooted.
}
\item{ordered_groups}{
When provided, specifies the order in which different habitats should be plotted. This order is matched to the color order specified in \code{fill_color} (see below).
}
\item{branch_color}{
A vector specifying what colors to use for branches associated with each of the different habitat combinations in the Venn diagram. This vector must be as long as the number of possible habitat combinations (number of compartments in the Venn diagram plus one for branches not included in the core of any habitats). When no colors are specified or a vector of the wrong length is specified, the default is to use a range of colors from blue to red.
}
\item{branch.width}{
The width to use when plotting the branches of the phylogeny. The default is 4.
}
\item{label.tips}{
Whether or not to label the tips of the phylogeny with the microbial taxon names. The default is FALSE.
}
\item{scaled}{
Whether or not to scale the branch lengths. The default is FALSE.
}
\item{remove_zeros}{
Whether or not to remove taxa that are missing from all samples prior to drawing the phylogeny. The default is TRUE.
}
\item{plot.chronogram}{
Whether to plot a phylogeny or a chronogram. The default is FALSE.
}

}
\details{
\code{coreVennTree} generates a phylogeny with branches colored according to the compartments of an associated Venn diagram as generated using the \code{coreVenn} function. For more details, see Bewick and Camper (2025).
}
\value{
This function returns a color coded plot of the phylogeny. When a vector of colors is specified, the color key is printed out in the console.
}

\references{
Bewick, S.A. and Benjamin T. Camper. "Phylogenetic Measures of the Core Microbiome" <doi:TBD>

McMurdie, Paul J., and Susan Holmes. "phyloseq: an R package for reproducible interactive analysis and graphics of microbiome census data." PloS one 8.4 (2013): e61217.

McMurdie, Paul J., and Susan Holmes. "Phyloseq: a bioconductor package for handling and analysis of high-throughput phylogenetic sequence data." Biocomputing 2012. 2012. 235-246.
}

\examples{
#Test with enterotype dataset
library(phyloseq)
library(ape)
library(phytools)
library(ggplot2)
data(enterotype)

set.seed(1)

#Generate an example tree and label it with the names of the microbial taxa
enterotype_tree<-rtree(length(taxa_names(enterotype)))
enterotype_tree$tip.label<-taxa_names(enterotype)

#keep only those samples with gender identified
gendered<-which(!(is.na(sample_data(enterotype)$Gender)))
enterotypeMF<-prune_samples(sample_names(enterotype)[gendered],enterotype)

#Create a phyloseq object with a tree
example_phyloseq<-phyloseq(otu_table(enterotypeMF),phy_tree(as.phylo(enterotype_tree)))

#Define the groups
bygender<-sample_data(enterotypeMF)$Gender

#Define the colors for group combinations
clist<-c('black','red','orange','yellow')

#Plot the tree
coreVennTree(example_phyloseq,grouping=bygender,0.5,branch_color=clist)

}
