% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/probtrans_coxph.R
\name{probtrans_coxph}
\alias{probtrans_coxph}
\title{Calculate subject specific transition probabilities from a 
multi-state \code{\link[survival:coxph]{coxph}} model.}
\usage{
probtrans_coxph(
  object,
  predt,
  direction = c("forward", "fixedhorizon"),
  newdata,
  trans
)
}
\arguments{
\item{object}{A \code{\link[survival:coxph]{coxph}} object fit on multi-state
data. Must contain a \code{strata(X)} term. Data used for the coxph() fit preferably prepared 
using \code{\link[mstate:msprep]{msprep}}.}

\item{predt}{A positive number indicating the prediction time. This is either 
the time at which the prediction is made (if \code{direction = "forward"}) or 
the time for which the prediction is to be made (if \code{direction = "backward"}).}

\item{direction}{One of \code{"forward"} (default) or \code{"fixedhorizon"},
indicating whether prediction is forward or for a fixed horizon}

\item{newdata}{A \code{data.frame} containing a single row for each transition per
subject in the data. For a model with m possible transitions, and n subjects 
\code{newdata} must have m*n rows. It must contain the following named columns:
\describe{
  \item{\code{id}:}{Unique identifier of the subject, can be numeric or character;}
  \item{\code{from}:}{State from which the transition takes place;}
  \item{\code{to}:}{State to which the transition takes place;}
  \item{\code{trans}:}{Transition number in the \code{'transMat'} \code{trans} this transition relates to;}
  \item{\code{"variables"}:}{The variables and their values for the subject
  identified by "id" for the transition this entry relates to. Names must 
  match the names of the variables in coxph \code{object}.}
} Note that newdata must contain a column containing the variable which was 
used to determine the stratum of a transition in \code{object}. 
Usually the stratum is determined from one of the required columns. The 
"variables" columns can usually be obtained using \code{\link[mstate:expand.covs]{expand.covs}}.}

\item{trans}{A transition matrix as created by \code{\link[mstate:transMat]{transMat}}.}
}
\value{
An object of class \code{"probtrans.subjects"}.
This is a list of length n (number of subjects in newdata), with each list element
an object of class \code{\link[mstate:probtrans]{probtrans}} for the associated 
subject. List elements can be accessed using \code{[[x]]}, with \code{x} 
ranging from 1 to n. Additionally, each list element 
has an element \code{$id}, representing the subject id and the output object 
also has an element \code{$subject_ids} representing the subject ids in order.
}
\description{
Given a coxph model fit on multi-state data (prepared with \code{\link[mstate:msprep]{msprep}}),
determine transition probabilities for subjects in \code{newdata}.
}
\details{
When using this function for \code{newdata} with many subjects, consider running the 
function multiple times for parts of \code{newdata} to negate the risk of 
running our of memory.
}
\examples{
#Example from the mstate vignette
#We determine the subject specific transition probabilities for subjects
#in the ebmt3 data-set 
if(require("mstate")){
  data(ebmt3)
  n <- nrow(ebmt3)
  tmat <- transMat(x = list(c(2, 3), c(3), c()), names = c("Tx",
                                                           "PR", "RelDeath"))
  ebmt3$prtime <- ebmt3$prtime/365.25
  ebmt3$rfstime <- ebmt3$rfstime/365.25
  covs <- c("dissub", "age", "drmatch", "tcd", "prtime")
  msbmt <- msprep(time = c(NA, "prtime", "rfstime"), status = c(NA,
                  "prstat", "rfsstat"), data = ebmt3, trans = tmat, keep = covs)
  #Expand covariates so that we can have transition specific covariates
  msbmt <- expand.covs(msbmt, covs, append = TRUE, longnames = FALSE)
  
  #Create extra variable to allow gender mismatch to have the same effect 
  #for transitions 2 and 3.
  msbmt$drmatch.2.3 <- msbmt$drmatch.2 + msbmt$drmatch.3
  
  #Introduce pr covariate for proportionality assumption of transitions 2 and 3
  #(only used in models 2 and 4)
  msbmt$pr <- 0
  msbmt$pr[msbmt$trans == 3] <- 1
  
  
  
  #-------------Models---------------------#
  
  #Simple model, transition specific covariates, each transition own baseline hazard
  c1 <- coxph(Surv(Tstart, Tstop, status) ~ dissub1.1 + dissub2.1 +
                age1.1 + age2.1 + drmatch.1 + tcd.1 + dissub1.2 + dissub2.2 +
                age1.2 + age2.2 + drmatch.2 + tcd.2 + dissub1.3 + dissub2.3 +
                age1.3 + age2.3 + drmatch.3 + tcd.3 + strata(trans), data = msbmt,
                method = "breslow")
  
  #Model with same baseline hazards for transitions 2 (1->3) and 3(2->3)
  #pr then gives the ratio of the 2 hazards for these transitions
  c2 <- coxph(Surv(Tstart, Tstop, status) ~ dissub1.1 + dissub2.1 +
                age1.1 + age2.1 + drmatch.1 + tcd.1 + dissub1.2 + dissub2.2 +
                age1.2 + age2.2 + drmatch.2 + tcd.2 + dissub1.3 + dissub2.3 +
                age1.3 + age2.3 + drmatch.3 + tcd.3 + pr + strata(to), data = msbmt,
                method = "breslow")
  
  #Same as c2, but now Gender mismatch has the same effect for both 
  c3 <- coxph(Surv(Tstart, Tstop, status) ~ dissub1.1 + dissub2.1 +
                age1.1 + age2.1 + drmatch.1 + tcd.1 + dissub1.2 + dissub2.2 +
                age1.2 + age2.2 + drmatch.2.3 + tcd.2 + dissub1.3 + dissub2.3 +
                age1.3 + age2.3  + tcd.3 + pr + strata(to), data = msbmt,
              method = "breslow")
              
  
  #Predict for first 30 people in ebmt data
  tmat2 <- to.trans2(tmat)[, c(2,3,1)]
  names(tmat2)[3] <- "trans"
  n_transitions <- nrow(tmat2)
  
  newdata <- ebmt3[rep(seq_len(30), each = nrow(tmat2)),]
  newdata <- cbind(tmat2[rep(seq_len(nrow(tmat2)), times = 30), ], newdata)
  #Make of class "msdata"
  attr(newdata, "trans") <- tmat
  class(newdata) <- c("msdata", "data.frame")
  #Covariate names to expand
  covs <- names(newdata)[5:ncol(newdata)]
  newdata <- expand.covs(newdata, covs, append = TRUE, longnames = FALSE)
  
  newdata$drmatch.2.3 <- newdata$drmatch.2 + newdata$drmatch.3
  
  newdata$pr <- 0
  newdata$pr[newdata$trans == 3] <- 1
  
  #Calculate transition probabilities for the Cox fits
  icmstate_pt1 <- probtrans_coxph(c1, predt = 0, direction = "forward", 
                                  newdata = newdata, trans = tmat)
  icmstate_pt2 <- probtrans_coxph(c2, predt = 0, direction = "forward", 
                                  newdata = newdata, trans = tmat)
  icmstate_pt3 <- probtrans_coxph(c3, predt = 0, direction = "forward", 
                                  newdata = newdata, trans = tmat)
                                  
  #Now we can plot the transition probabilities for each subject separately:
  plot(icmstate_pt1[[1]])
  #icmstate_pt has length number of subjects in newdata
  #And icmstate_pt1[[i]] is an object of class "probtrans", so you can 
  #use all probtrans functions: summary, plot etc.
  
  #Alternatively, use the plotting function directly:
  plot(icmstate_pt2, id = 2)
}


}
