% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/assume.R
\name{assume}
\alias{assume}
\title{Define a theoretical distribution}
\usage{
assume(x, distribution, df = NULL, ...)
}
\arguments{
\item{x}{The output of \code{\link[=specify]{specify()}} or \code{\link[=hypothesize]{hypothesize()}}, giving the
observed data, variable(s) of interest, and (optionally) null hypothesis.}

\item{distribution}{The distribution in question, as a string. One of
\code{"F"}, \code{"Chisq"}, \code{"t"}, or \code{"z"}.}

\item{df}{Optional. The degrees of freedom parameter(s) for the \code{distribution}
supplied, as a numeric vector. For \code{distribution = "F"}, this should have
length two (e.g. \code{c(10, 3)}). For \code{distribution = "Chisq"} or
\code{distribution = "t"}, this should have length one. For
\code{distribution = "z"}, this argument is not required. The package
will supply a message if the supplied \code{df} argument is different from
recognized values. See the Details section below for more information.}

\item{...}{Currently ignored.}
}
\value{
An infer theoretical distribution that can be passed to helpers
like \code{\link[=visualize]{visualize()}}, \code{\link[=get_p_value]{get_p_value()}}, and \code{\link[=get_confidence_interval]{get_confidence_interval()}}.
}
\description{
This function allows the user to define a null distribution based on
theoretical methods. In many infer pipelines, \code{assume()} can be
used in place of \code{\link[=generate]{generate()}} and \code{\link[=calculate]{calculate()}} to create a null
distribution. Rather than outputting a data frame containing a
distribution of test statistics calculated from resamples of the observed
data, \code{assume()} outputs a more abstract type of object just containing
the distributional details supplied in the \code{distribution} and \code{df} arguments.
However, \code{assume()} output can be passed to \code{\link[=visualize]{visualize()}}, \code{\link[=get_p_value]{get_p_value()}},
and \code{\link[=get_confidence_interval]{get_confidence_interval()}} in the same way that simulation-based
distributions can.

To define a theoretical null distribution (for use in hypothesis testing),
be sure to provide a null hypothesis via \code{\link[=hypothesize]{hypothesize()}}. To define a
theoretical sampling distribution (for use in confidence intervals),
provide the output of \code{\link[=specify]{specify()}}. Sampling distributions (only
implemented for \code{t} and \code{z}) lie on the scale of the data, and will be
recentered and rescaled to match the corresponding \code{stat} given in
\code{\link[=calculate]{calculate()}} to calculate the observed statistic.
}
\details{
Note that the assumption being expressed here, for use in theory-based
inference, only extends to \emph{distributional} assumptions: the null
distribution in question and its parameters. Statistical inference with
infer, whether carried out via simulation (i.e. based on pipelines
using \code{\link[=generate]{generate()}} and \code{\link[=calculate]{calculate()}}) or theory (i.e. with \code{assume()}),
always involves the condition that observations are independent of
each other.

\code{infer} only supports theoretical tests on one or two means via the
\code{t} distribution and one or two proportions via the \code{z}.

For tests comparing two means, if \code{n1} is the group size for one level of
the explanatory variable, and \code{n2} is that for the other level, \code{infer}
will recognize the following degrees of freedom (\code{df}) arguments:
\itemize{
\item \code{min(n1 - 1, n2 - 1)}
\item \code{n1 + n2 - 2}
\item The \code{"parameter"} entry of the analogous \code{stats::t.test()} call
\item The \code{"parameter"} entry of the analogous \code{stats::t.test()} call with \code{var.equal = TRUE}
}

By default, the package will use the \code{"parameter"} entry of the analogous
\code{stats::t.test()} call with \code{var.equal = FALSE} (the default).
}
\examples{
# construct theoretical distributions ---------------------------------

# F distribution
# with the `partyid` explanatory variable
gss |>
  specify(age ~ partyid) |>
  assume(distribution = "F")

# Chi-squared goodness of fit distribution
# on the `finrela` variable
gss |>
  specify(response = finrela) |>
  hypothesize(null = "point",
              p = c("far below average" = 1/6,
                    "below average" = 1/6,
                    "average" = 1/6,
                    "above average" = 1/6,
                    "far above average" = 1/6,
                    "DK" = 1/6)) |>
  assume("Chisq")

# Chi-squared test of independence
# on the `finrela` and `sex` variables
gss |>
  specify(formula = finrela ~ sex) |>
  assume(distribution = "Chisq")

# T distribution
gss |>
  specify(age ~ college) |>
  assume("t")

# Z distribution
gss |>
  specify(response = sex, success = "female") |>
  assume("z")

\dontrun{
# each of these distributions can be passed to infer helper
# functions alongside observed statistics!

# for example, a 1-sample t-test -------------------------------------

# calculate the observed statistic
obs_stat <- gss |>
  specify(response = hours) |>
  hypothesize(null = "point", mu = 40) |>
  calculate(stat = "t")

# construct a null distribution
null_dist <- gss |>
  specify(response = hours) |>
  assume("t")

# juxtapose them visually
visualize(null_dist) +
  shade_p_value(obs_stat, direction = "both")

# calculate a p-value
get_p_value(null_dist, obs_stat, direction = "both")

# or, an F test ------------------------------------------------------

# calculate the observed statistic
obs_stat <- gss |>
  specify(age ~ partyid) |>
  hypothesize(null = "independence") |>
  calculate(stat = "F")

# construct a null distribution
null_dist <- gss |>
  specify(age ~ partyid) |>
  assume(distribution = "F")

# juxtapose them visually
visualize(null_dist) +
  shade_p_value(obs_stat, direction = "both")

# calculate a p-value
get_p_value(null_dist, obs_stat, direction = "both")
}

}
