
#define R_NO_REMAP

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <stdbool.h>
#include <unistd.h>

#include <R.h>
#include <Rinternals.h>
#include <Rdefines.h>

#include "utils.h"


//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// REturn a data.frame of information about the per-face visibliity of
// voxels which are viaible at all
//
// @param x,y,z voxel coordinates
//
// Steps:
// 1. calculate the hashed coordinates of each voxel
//    This renders (x,y,z) to (xc,yc) which are basically the isometric
//    coordinates of the voxel.  It differs from the actual coordinates in 
//    that I've rigged the xc/yc calculation so that they're integers only
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
SEXP visibility_(SEXP x_, SEXP y_, SEXP z_) {
  
  int nprotect = 0;
  
  int N = Rf_length(x_);
  if (Rf_length(y_) != N || Rf_length(z_) != N) {
    Rf_error("All of x,y,z must the same length");
  }
  
  int *x = INTEGER(x_);
  int *y = INTEGER(y_);
  int *z = INTEGER(z_);
  
  //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  // Keep track of the ranges of the hashed coordinates
  //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  int xc_max = -INT_MAX;
  int yc_max = -INT_MAX;
  int xc_min =  INT_MAX;
  int yc_min =  INT_MAX;
  
  
  //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  // Allocate and calculate the hashed coordinates
  //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  int *xc = malloc(N * sizeof(int));
  int *yc = malloc(N * sizeof(int));
  if (xc == NULL || yc == NULL) {
    Rf_error("xc/yc malloc");
  }
  for (int i = 0; i < N; i++) {
    xc[i] = x[i] - y[i];
    if (xc[i] > xc_max) { xc_max = xc[i]; }
    if (xc[i] < xc_min) { xc_min = xc[i]; }
    yc[i] = 2 * z[i] + x[i] + y[i];
    if (yc[i] > yc_max) { yc_max = yc[i]; }
    if (yc[i] < yc_min) { yc_min = yc[i]; }
  }
  
  
  //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  // Determine the size of the hashed coordinate matrix
  // This matrix will be used to track if a voxel is being drawn at the same
  // position on screen.   Resolution of which voxel to draw is governed by 
  // the relative sizes of the original (x,y,z) coordinates.
  //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  int cwidth  = xc_max - xc_min + 1;
  int cheight = yc_max - yc_min + 1;
  
  int **mat = malloc(cheight * sizeof(int *));
  if (mat == NULL) {
    Rf_error("**mat malloc");
  }
  
  for (int row = 0; row < cheight; row++) {
    mat[row] = malloc(cwidth * sizeof(int));
    if (mat[row] == NULL) {
      Rf_error("*mat malloc");
    }
    for (int col = 0; col < cwidth; col++) {
      mat[row][col] = -1; // Sentinel value: -1 = empty
    }
  }
  
  //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  // For each hashed coordinate
  //   - locate its slot in the matrix
  //   - if the slot is empty: insert this index
  //   - if the slot already contains an index, work out whether this new
  //       voxel is in front of it, and if so, replace the index with this
  //       new voxel's index
  //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  int nvisible = 0;
  for (int i = 0; i < N; i++) {
    int row = yc[i] - yc_min;
    int col = xc[i] - xc_min;
    if (col >= cwidth || row >= cheight || col < 0 || row < 0) {
      Rf_error("[%i, %i] / [%i, %i]\n", col, row, cheight, cwidth);
    }
    int val = mat[row][col];
    if (val < 0) {
      mat[row][col] = i;
      nvisible++;
    } else {
      int cidx = mat[row][col];
      if (z[i] > z[cidx] && x[i] < x[cidx] && y[i] < y[cidx]) {
        mat[row][col] = i;
      }
    }
  }

  //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  // visibility debugging
  //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  
  if (false) {
    for (int row = cheight - 1; row >= 0; row--) {
      Rprintf("[%i] ", row);
      for (int col = 0; col < cwidth; col++) {
        if (mat[row][col] >= 0) {
          Rprintf("%4i",mat[row][col] + 1);
        } else {
          Rprintf("   .");
        }
      }
      Rprintf("\n");
    }
  }
  
  //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  // Create a data.frame of
  //  - idx: indices of the 'coords' data.frame which are visible
  //  - type: bitset of which faces are visible 001 = top, 010 = left, 100 = right
  //
  // 'nvisible' is the *MAXIMUM* number of visible cubes.
  //   some cubes might not have a cube *directly* in front of them, but 
  //   its 3 faces may be blocked by three different cubes
  //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  
  int *visible_idx = malloc(nvisible * sizeof(int));
  int *vis_type    = malloc(nvisible * sizeof(int));
  
  if (visible_idx == NULL || vis_type == NULL) {
    Rf_error("visibility_(): Couldn't allocate internally");
  }
  
  int vidx = 0;
  for (int row = 0; row < cheight; row++) {
    for (int col = 0; col < cwidth; col++) {
      int this_idx = mat[row][col];
      if (this_idx >= 0) {
        
        bool visible_top   = true;
        bool visible_left  = true;
        bool visible_right = true;
        
        // Top visibility
        if (row < (cheight - 2)) {
          int above_idx = mat[row + 2][col];
          if (above_idx >= 0) {
            if (z[above_idx] > z[this_idx]) {
              visible_top = false;
            }
          }
        }
        
        // Left visibility
        if (row > 0 && col > 0) {
          int left_idx = mat[row - 1][col - 1];
          if (left_idx >= 0) {
            if (x[left_idx] < x[this_idx]) {
              visible_left = false;
            }
          }
        }
        
        // right visibility
        if (row > 0 && col < (cwidth - 1)) {
          int right_idx = mat[row - 1][col + 1];
          if (right_idx >= 0) {
            if (y[right_idx] < y[this_idx]) {
              visible_right = false;
            }
          }
        }
        
        int visibility = 
          visible_top        |
          visible_left  << 1 |
          visible_right << 2;
        
        if (visibility > 0) {
          visible_idx[vidx] = this_idx + 1; // convert to R 1-indexing
          vis_type[vidx] = visibility;
          vidx++;
        }
        
      }
    }
  }
  
  SEXP visible_idx_ = PROTECT(Rf_allocVector(INTSXP, vidx)); nprotect++;
  SEXP vis_type_    = PROTECT(Rf_allocVector(INTSXP, vidx)); nprotect++;
  
  memcpy(INTEGER(visible_idx_), visible_idx, vidx * sizeof(int));
  memcpy(INTEGER(vis_type_   ), vis_type   , vidx * sizeof(int));
  
  free(visible_idx);
  free(vis_type);
  
  //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  // Create data.frame to return
  //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  SEXP res_ = PROTECT(
    create_named_list(2, "idx", visible_idx_, "type", vis_type_)
  ); nprotect++;
  set_df_attributes(res_);
  
  
  //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  // Tidy and return
  //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  for (int i = 0; i < cheight; i++) {
    free(mat[i]);
  }
  free(mat);
  
  free(xc);
  free(yc);
  UNPROTECT(nprotect);
  return res_;
}

