% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot.l2boost.R
\name{plot.l2boost}
\alias{plot.l2boost}
\title{Plotting for \code{\link{l2boost}} objects.}
\usage{
\method{plot}{l2boost}(
  x,
  type = c("rho", "coef"),
  standardize = TRUE,
  active.set = NULL,
  xvar = c("step", "norm"),
  xlab = NULL,
  ylab = NULL,
  trim = TRUE,
  clip = NULL,
  col = NULL,
  ylim = NULL,
  xlim = NULL,
  ...
)
}
\arguments{
\item{x}{l2boost or cv.l2boost object}

\item{type}{which type of plot. \emph{rho} plots gradient-correlation, \emph{coef} regression (beta) 
coefficients vs the step number m along the x-axis}

\item{standardize}{Should we plot standardized gradient-correlation (default: TRUE)}

\item{active.set}{Vector of indices of the coordinates for highlighting with 
color=col (default: NULL shows all active coordinates)}

\item{xvar}{what measure do we plot on the x-axis? \emph{step} plots the step m, \emph{norm} plots the 
normalized distance (1-nu)^(m-1)}

\item{xlab}{specific x-axis label (NULL results in default value depending on xvar)}

\item{ylab}{specific y-axis label (NULL results in default value depending on type)}

\item{trim}{(default: TRUE)}

\item{clip}{Do we want to c}

\item{col}{Color to highlight active.set coordinates (NULL indicates default all active set at 
step M in blue, changes to red after selection}

\item{ylim}{Control plotted y-values (default: NULL for auto range)}

\item{xlim}{Control plotted x-values (default: NULL for auto domain )}

\item{...}{other arguments passed to plot functions}
}
\value{
\code{NULL}
}
\description{
plotting methods for \code{\link{l2boost}} objects (\code{\link{l2boost}} and \code{\link{cv.l2boost}}). 

By default, plotting an \code{\link{l2boost}} object produces a gradient-correlation vs iteration steps (m) plot.
Plotting a \code{\link{cv.l2boost}} object produces a cross-validation error plot, and prints the minimal CV MSE value
and optimal step opt.step to the R console.

Many generic arguments to \code{\link{plot}} are passed through the \code{\link{plot.l2boost}} function.
}
\details{
Gradient-correlation plots are created by tracing out the boosting coefficient (rho) for each candidate
direction. The coefficient and gradient-correlation are equivalent under standard scaling (zero intercept with 
design matrix columns scaled to have mean=0 and variance=1).  

Unless explicitly set using \emph{col} argument, the plot function colors the gradient-correlation paths along each
direction by the following criteria: 
\itemize{
\item Red: indicates the coordinate direction has been selected in the boosting path at some step <= m. 
\item Blue: indicates the coordinate will be selected within the specified number of steps M (and switch to 
red upon selection).
\item Grey: indicates coordinates have not and will not be selected by the algorithm over all iterations. 
}
The colors are set using the \emph{l.crit} return value from the \code{\link{l2boost}} object.
}
\examples{
#--------------------------------------------------------------------------
# Example: Diabetes 
#  
# See Efron B., Hastie T., Johnstone I., and Tibshirani R. 
# Least angle regression. Ann. Statist., 32:407-499, 2004.
data(diabetes, package = "l2boost")

l2.object <- l2boost(diabetes$x,diabetes$y, M=1000, nu=.01)

# Plot the gradient-correlation, and regression beta coefficients as a function of
# boosting steps m
par(mfrow=c(2,2))
plot(l2.object)
abline(v=500, lty=2, col="grey")
plot(l2.object, type="coef")
abline(v=500, lty=2, col="grey")

# limit the plot to only the first 500 steps of the algorithm 
# (grey vertical line in previous plots).
plot(l2.object, xlim=c(0,500))
plot(l2.object, type="coef", xlim=c(0,500))

\dontrun{
#--------------------------------------------------------------------------
# Example: Plotting cross-validation objects
dta <- elasticNetSim(n=100)
# Set the boosting parameters
Mtarget = 1000
nuTarget = 1.e-2

cv.l2 <- cv.l2boost(dta$x,dta$y,M=Mtarget, nu=nuTarget, lambda=NULL)

# Show the CV MSE plot, with a marker at the "optimal iteration"
plot(cv.l2)
abline(v=cv.l2$opt.step, lty=2, col="grey")

# Show the l2boost object plots.
plot(cv.l2$fit)
abline(v=cv.l2$opt.step, lty=2, col="grey")
 
plot(cv.l2$fit, type="coef")
abline(v=cv.l2$opt.step, lty=2, col="grey")

# Create a color vector of length p=40 (from elasticNetSim defaults)
clr <- rep("black", 40)
# Set coordinates in the boosting path to color red.
clr[unique(cv.l2$fit$l.crit)] = "red"

# Show the "optimal" coefficient values, 
# red points are selected in boosting algorithm.
plot(coef(cv.l2$fit, m=cv.l2$opt.step), col=clr, ylab=expression(beta))
}

}
\seealso{
\code{\link{l2boost}}, \code{\link{print.l2boost}}, \code{\link{predict.l2boost}} methods of l2boost 
and \code{\link{cv.l2boost}}
}
