#' Plot Mark Correlation Function Objects
#'
#' @description Plots objects of class `mc` (mark correlation functions).
#' Supports global and local types, multiple marks, and fully customizable appearance via `layers` or `...`.
#'
#' @param x An object of class `mc`.
#' @param which_marks For global or local multi-mark objects: character or numeric vector specifying which marks to plot. Default `NULL` (all marks).
#' @param which_points For local MC objects: numeric or character vector specifying which points to overlay. Default `NULL` (all points).
#' @param layers A list of additional ggplot2 layers (themes, labs, scales, etc.) to add to the plot.
#' @param xlab Label for x-axis (default "Distance r").
#' @param ylab Label for y-axis (default "Mark correlation").
#' @param ... Additional arguments passed to `geom_line()`.
#' @section Additional graphical customization:
#' The object returned by \code{plot.testmc()} is a \code{ggplot} object. This means
#' that all standard graphical components from the \pkg{ggplot2} package (such as
#' layers, scales, and themes) can be added to the plot using the \code{+} operator.
#' This allows users to further customize labels, colors, axes, and the overall
#' appearance of the plot.
#' @return Invisibly returns the ggplot object.
#' @author Mehdi Moradi \email{m2.moradi@yahoo.com}
#' @examples
#' \donttest{
#'
#'  library(spatstat.geom)
#'  library(spatstat.random)
#'  library(spatstat.explore)
#'  library(spatstat.linnet)
#'  library(ggplot2)
#'
#' # --- Example 1: Single mark --------------------------------------------
#'
#' X1 <- rpoispp(100)
#' marks(X1) <- cbind(m1 = runif(npoints(X1), 1, 10))
#'
#' mc1 <- mcorr.ppp(X1, ftype = "stoyan", method = "density")
#'
#' plot(mc1) +
#'   labs(
#'     title = "Mark correlation for a single mark",
#'     x = expression(r),
#'     y = "Correlation"
#'   ) +
#'   theme_minimal() +
#'   theme(
#'     plot.title = element_text(size = 14, face = "bold"),
#'     axis.title = element_text(size = 12),
#'     axis.text  = element_text(size = 10),
#'     legend.title = element_text(size = 11),
#'     legend.text  = element_text(size = 10)
#'   )
#'
#'
#' # --- Example 2: Two marks ----------------------------------------------
#'
#' X2 <- rpoispp(100)
#' marks(X2) <- cbind(
#'   m1 = runif(npoints(X2), 1, 10),
#'   m2 = runif(npoints(X2), 1, 10)
#' )
#'
#' mc2 <- mcorr.ppp(X2, ftype = "stoyan", method = "density")
#'
#' # Plot both marks
#' plot(mc2) +
#'   labs(
#'     title = "Mark correlation for two marks",
#'     x = expression(r),
#'     y = "Correlation"
#'   ) +
#'   theme_bw() +
#'   theme(
#'     plot.title = element_text(size = 14, face = "bold"),
#'     axis.title = element_text(size = 12),
#'     axis.text  = element_text(size = 10),
#'     legend.title = element_text(size = 11),
#'     legend.text  = element_text(size = 10),
#'     legend.position = "top"
#'   )
#'
#' # Plot only mark m1
#' plot(mc2, which_marks = "m1") +
#'   labs(
#'     title = "Mark correlation for mark m1 only",
#'     x = expression(r),
#'     y = "Correlation"
#'   ) +
#'   theme_minimal() +
#'   theme(
#'     plot.title = element_text(size = 14, face = "bold"),
#'     axis.title = element_text(size = 12),
#'     axis.text  = element_text(size = 10),
#'     legend.position = "none"
#'   )
#'
#'
#' # --- Example 3: Local mark correlation ---------------------------------
#'
#' lc1 <- lmcorr.ppp(X2, ftype = "stoyan", method = "density")
#'
#' # Two selected points, mark m1
#' plot(lc1, which_points = 1:2, which_marks = "m1") +
#'   labs(
#'     title = "Local mark correlation (points 1–2, mark m1)",
#'     x = expression(r),
#'     y = "Local correlation"
#'   ) +
#'   theme_classic() +
#'   theme(
#'     plot.title = element_text(size = 13, face = "bold"),
#'     axis.title = element_text(size = 12),
#'     axis.text  = element_text(size = 10),
#'     legend.title = element_text(size = 11),
#'     legend.text  = element_text(size = 10),
#'     legend.position = "right"
#'   )
#'
#' # Single selected point
#' plot(lc1, which_points = 1) +
#'   labs(
#'     title = "Local mark correlation (single selected point)",
#'     x = expression(r),
#'     y = "Local correlation"
#'   ) +
#'   theme_light() +
#'   theme(
#'     plot.title = element_text(size = 13, face = "bold"),
#'     axis.title = element_text(size = 12),
#'     axis.text  = element_text(size = 10),
#'     legend.position = "bottom"
#'   )
#'
#' }
#' @export
plot.mc <- function(x,
                    which_marks = NULL,
                    which_points = NULL,
                    layers = list(),
                    xlab = "Distance r",
                    ylab = "Mark correlation",
                    ...) {
  
  if (!requireNamespace("ggplot2", quietly = TRUE)) stop("plot.mc requires ggplot2")
  if (!requireNamespace("tidyr", quietly = TRUE)) stop("plot.mc requires tidyr")
  if (!requireNamespace("dplyr", quietly = TRUE)) stop("plot.mc requires dplyr")
  
  mctype <- attr(x, "type")
  if (is.null(mctype)) stop("Object must have attribute 'type' ('global' or 'local').")
  add_hline <- !is.null(attr(x, "normalise")) && attr(x, "normalise") == TRUE
  p <- NULL  # initialize plot object
  
  # ------------------ GLOBAL MC ------------------
  if (mctype == "global") {
    df <- as.data.frame(x)
    if (!"r" %in% names(df)) stop("as.data.frame(x) must contain an 'r' column.")
    value_cols <- setdiff(names(df), "r")
    if (!is.null(which_marks)) value_cols <- intersect(value_cols, which_marks)
    if (length(value_cols) == 0) stop("No mark columns to plot for global MC.")
    
    df_long <- if (length(value_cols) == 1) {
      df$key <- value_cols
      df$value <- df[[value_cols]]
      df
    } else {
      tidyr::pivot_longer(df, cols = dplyr::all_of(value_cols),
                          names_to = "key", values_to = "value")
    }
    
    aes_map <- ggplot2::aes(x = r, y = value, group = if(length(value_cols) > 1) key else 1)
    if (length(value_cols) > 1) aes_map$colour <- quote(key)
    
    p <- ggplot2::ggplot(df_long, aes_map) +
      ggplot2::geom_line(...)
  }
  
  # ------------------ LOCAL MC ------------------
  else if (mctype == "local") {
    df_local <- if (inherits(x, "mc")) as.data.frame(x) else x
    if (is.list(df_local) && all(sapply(df_local, is.data.frame))) {
      marks_list <- df_local
      mark_names <- names(marks_list)
      if (is.null(mark_names)) mark_names <- paste0("mark", seq_along(marks_list))
      if (!is.null(which_marks)) {
        marks_list <- marks_list[which_marks]
        # mark_names <- mark_names[which_marks]
        mark_names <- which_marks
      }
      
      df_long <- do.call(rbind, lapply(seq_along(marks_list), function(i) {
        dfm <- as.data.frame(marks_list[[i]])
        point_cols <- setdiff(names(dfm), "r")
        if (!is.null(which_points)) point_cols <- intersect(point_cols, which_points)
        df_long_i <- tidyr::pivot_longer(dfm,
                                         cols = dplyr::all_of(point_cols),
                                         names_to = "point",
                                         values_to = "value")
        df_long_i$mark <- mark_names[i]
        df_long_i
      }))
      
      aes_map <- ggplot2::aes(x = r, y = value, group = interaction(mark, point))
      if (length(unique(df_long$mark)) > 1) aes_map$colour <- quote(mark)
      
      p <- ggplot2::ggplot(df_long, aes_map) +
        ggplot2::geom_line(...)
      
    } else if (is.data.frame(df_local)) {
      df <- as.data.frame(df_local)
      value_cols <- setdiff(names(df), "r")
      if (!is.null(which_points)) value_cols <- intersect(value_cols, which_points)
      df_long <- tidyr::pivot_longer(df, cols = dplyr::all_of(value_cols),
                                     names_to = "point", values_to = "value")
      
      aes_map <- ggplot2::aes(x = r, y = value, group = point)
      p <- ggplot2::ggplot(df_long, aes_map) +
        ggplot2::geom_line(...)
    } else {
      stop("Local MC object must be a data frame or a list of data frames after conversion.")
    }
  } else {
    stop("Unknown type of MC function: must be 'global' or 'local'.")
  }
  
  # ------------------ horizontal line if normalized ------------------
  if (add_hline) p <- p + ggplot2::geom_hline(yintercept = 1, linetype = "dashed", color = "black")
  
  # ------------------ labels and theme ------------------
  p <- p + ggplot2::labs(x = xlab, y = ylab) + ggplot2::theme_bw()
  
  # ------------------ add user layers ------------------
  if (length(layers) > 0) for (ly in layers) p <- p + ly
  
  print(p)
  invisible(p)
}

if (getRversion() >= "2.15.1") {
  utils::globalVariables(c("r", "value", "key", "mark", "point"))
}