#' Create request for RIA API
#'
#' Create the request for ria based on paths
#'
#' This function creates the request, deal with errors and retries if needed
#' and access the data of the response.
#'
#' @param paths list of character vectors with the paths as obtained by
#'   \link{\code{.create_ria_path}}
#'
#' @return a tibble converted from the response json.
#'
#' @noRd
.create_ria_request <- function(paths) {

  ria_requests <- purrr::map(
    paths,
    \(path) {
      httr2::request("https://www.juntadeandalucia.es") |>
        httr2::req_url_path_append(path) |>
        httr2::req_user_agent(
          "meteospain R package (https://emf.creaf.cat/software/meteospain/)"
        ) |>
        httr2::req_error(
          body = \(resp) {
            # fallback
            message <- "Unknown error, response has no body"
            if (httr2::resp_has_body(resp)) {
              message <- httr2::resp_body_string(resp)
            }
            # more verbose known errors
            if (httr2::resp_status(resp) == 404L) {
              message <- c(
                message,
                httr2::resp_url(resp)
              )
            }
            # if (httr2::resp_status(resp) == 400L) {
            #   message <-
            #     httr2::resp_body_json(resp, simplifyVector = TRUE)$message
            # }

            message
          }
        ) |>
        httr2::req_retry(
          max_tries = 3,
          retry_on_failure = TRUE,
          is_transient = \(resp) {
            httr2::resp_status(resp) %in% c(429, 500, 503)
          },
          backoff = \(resp) {
            60
          },
          after = \(resp) {
            NA
          }
        )
    }
  )

  # now we use req_perform_sequential to get the results.
  # Here there is catch. As we are using sequential perform, but some
  # API endpoints have only one request, the on_error argument
  # must be "continue" when more than one request, but "stop" when
  # only one request
  on_error_value <- "stop"
  if (length(ria_requests) > 1) {
    on_error_value <- "continue"
  }
  ria_responses <- httr2::req_perform_sequential(
    ria_requests, on_error = on_error_value, progress = TRUE
  )

  res <- ria_responses |>
    httr2::resps_successes() |>
    httr2::resps_data(
      \(resp) {
        httr2::resp_body_json(resp, simplifyVector = TRUE) |>
          dplyr::as_tibble() |>
          dplyr::mutate(url = httr2::resp_url(resp))
      }
    )

  # Check if any failure happened
  failures <- ria_responses |>
    httr2::resps_failures()

  # TODO: do something with the errors (cli warning or similar)
  if (length(failures) > 0) {
    urls <- purrr::map(httr2::resps_requests(failures), "url")
    messages <- purrr::map(failures, "message")

    # If all is failures, stop
    if (identical(length(failures), length(ria_responses))) {
      cli::cli_abort(c(
        "x" = "No data was retrieved",
        purrr::map2_chr(urls, messages, \(u, m) { paste0(u, ": ", m) })
      ))
    }

    cli::cli_inform(c(
      "x" = "Some stations didn't return data:",
      "i" = purrr::map2_chr(urls, messages, \(u, m) { paste0(u, ": ", m) })
    ))
  }

  return(res)
}

#' Create the path elements for RIA API
#'
#' Path vectors for RIA API to use with httr2
#'
#' @section Stations
#' In this case as RIA is capped to one station per query, so we need to loop by stations provided, or, if
#' NULL, by all the stations available.
#'
#' @param api_options Option list as generated by \link{\code{ria_options}}
#'
#' @noRd
.create_ria_path <- function(api_options) {

  # we need the resolution to create the corresponding path
  resolution <- api_options$resolution

  ria_stamp <- lubridate::stamp("2001-12-25", orders = "Ymd0", quiet = TRUE)

  month_and_years <- dplyr::tibble(
    year = lubridate::year(seq(api_options$start_date, api_options$end_date, 'months')),
    month = lubridate::month(seq(api_options$start_date, api_options$end_date, 'months'))
  ) |>
    dplyr::group_by(.data$year) |>
    dplyr::mutate(min_month = min(.data$month), max_month = max(.data$month)) |>
    dplyr::select(-"month") |>
    dplyr::distinct() |>
    as.list()

  provinces_and_stations <- stringr::str_split(api_options$stations, '-', n = 2, simplify = TRUE)

  # now the path vectors for the resolutions
  paths_resolution <- switch(
    resolution,
    # for daily and monthly, stations are paths.
    'daily' = purrr::map2(
      provinces_and_stations[,1], provinces_and_stations[,2],
      function(province, station) {
        c(
          'agriculturaypesca', 'ifapa', 'riaws', 'datosdiarios', 'forceEt0', province, station,
          ria_stamp(api_options$start_date), ria_stamp(api_options$end_date)
        )
      }
    ),
    'monthly' = purrr::flatten(purrr::map2(
      provinces_and_stations[,1], provinces_and_stations[,2],
      function(province, station) {
        province_station_path <-
          c('agriculturaypesca', 'ifapa', 'riaws', 'datosmensuales', province, station)
        purrr::pmap(
          month_and_years,
          function(year, min_month, max_month) {
            c(province_station_path, year, min_month, max_month)
          }
        )
      }
    )),
    list()
  )

  # not recognised resolution
  if (length(paths_resolution) < 1) {
    cli::cli_abort(c(
      "{.arg {resolution}} is not a valid temporal resolution for ria.\nPlease see ria_options help for more information."
    ))
  }

  return(paths_resolution)
}

#' Get info for the ria stations
#'
#' Get info for the ria stations
#'
#' @noRd

.get_info_ria <- function() {

  # GET parts needed --------------------------------------------------------------------------------------
  # path
  path_resolution <- list(c('agriculturaypesca', 'ifapa', 'riaws', 'estaciones'))
  # cache
  cache_ref <- rlang::hash(path_resolution)

  # get data from cache or from API if new
  info_ria <- .get_cached_result(cache_ref, {

    .create_ria_request(path_resolution) |>
      unnest_safe("provincia", names_sep = "_") |>
      # add service name, to identify the data if joining with other services
      dplyr::mutate(service = 'ria') |>
      dplyr::select(
        "service", station_id = "codigoEstacion", station_name = "nombre",
        station_province = "provincia_nombre", province_id = "provincia_id",
        altitude = "altitud", "longitud", "latitud", under_plastic = "bajoplastico"
      ) |>
      dplyr::distinct() |>
      dplyr::mutate(
        station_id = as.character(glue::glue("{province_id}-{station_id}")),
        altitude = units::set_units(.data$altitude, 'm'),
        latitud = .parse_coords_dmsh(.data$latitud),
        longitud = .parse_coords_dmsh(.data$longitud),
      ) |>
      sf::st_as_sf(coords = c('longitud', 'latitud'), crs = 4326)
  })

  return(info_ria)
}

#' Get data from RIA
#'
#' Get data from RIA service
#'
#' For all resolutions, if no stations are provided all stations will be retrieved
#'
#' @param api_options Option list as generated by \link{\code{ria_options}}
#'
#' @noRd
.get_data_ria <- function(api_options) {

  # All necessary things for the GET ----------------------------------------------------------------------
  # stations_info and update api_options
  # we need the stations id and their province
  stations_info <- .get_info_ria()
  if (is.null(api_options$stations)) {
    api_options$stations <- stations_info[['station_id']]
  }
  # create api paths
  paths_resolution <- .create_ria_path(api_options)
  # cache
  # NOTE: in RIA, only resolutions available are daily and monthly, so no
  # cache invalidation is needed for less than daily as in other APIs
  cache_ref <- rlang::hash(paths_resolution)

  # get data from cache or from API if new
  .get_cached_result(cache_ref, {

    # Resolution specific carpentry -------------------------------------------------------------------------
    # Now, instant/hourly and daily/monthly/yearly differs in the unnest step, as the column names are called
    # differently. It also differs in the select step as in the latter group there is no repetition of column
    # names after the unnest step.
    resolution_specific_select_quos <- switch(
      api_options$resolution,
      'daily' = .ria_daily_select_quos,
      'monthly' = .ria_monthly_select_quos
    )

    resolution_specific_mutate_quos <- switch(
      api_options$resolution,
      'daily' = .ria_daily_mutate_quos,
      'monthly' = .ria_monthly_mutate_quos
    )

    # Data transformation -----------------------------------------------------------------------------------
    res <- .create_ria_request(paths_resolution) |>
      dplyr::mutate(station_id = .ria_url2station(url)) |>
      dplyr::select(
        !!! resolution_specific_select_quos(), "station_id",
        mean_temperature = "tempMedia", min_temperature = "tempMin", max_temperature = "tempMax",
        mean_relative_humidity = "humedadMedia", min_relative_humidity = "humedadMin",
        max_relative_humidity = "humedadMax",
        mean_wind_speed = "velViento", mean_wind_direction = "dirViento",
        max_wind_speed = "velVientoMax", max_wind_direction = "dirVientoVelMax",
        precipitation = "precipitacion",
        solar_radiation = "radiacion"
      ) |>
      dplyr::mutate(
        !!! resolution_specific_mutate_quos(),
        mean_temperature = units::set_units(.data$mean_temperature, "degree_C"),
        min_temperature = units::set_units(.data$min_temperature, "degree_C"),
        max_temperature = units::set_units(.data$max_temperature, "degree_C"),
        mean_relative_humidity = units::set_units(.data$mean_relative_humidity, "%"),
        min_relative_humidity = units::set_units(.data$min_relative_humidity, "%"),
        max_relative_humidity = units::set_units(.data$max_relative_humidity, "%"),
        mean_wind_speed = units::set_units(.data$mean_wind_speed, 'm/s'),
        max_wind_speed = units::set_units(.data$max_wind_speed, 'm/s'),
        mean_wind_direction = units::set_units(.data$mean_wind_direction, 'degree'),
        max_wind_direction = units::set_units(.data$max_wind_direction, 'degree'),
        precipitation = units::set_units(.data$precipitation, "L/m^2"),
        solar_radiation = units::set_units(.data$solar_radiation, "MJ/d/m^2"),
        timestamp = lubridate::as_datetime(.data$timestamp),
        station_id = as.character(.data$station_id)
      ) |>
      dplyr::left_join(stations_info, by = 'station_id') |>
      dplyr::select(!dplyr::any_of(c('month', 'year', 'province_id'))) |>
      # reorder variables to be consistent among all services
      relocate_vars() |>
      # ensure we have an sf
      sf::st_as_sf()

    # Copyright message -------------------------------------------------------------------------------------
    cli::cli_inform(c(
      i = copyright_style("Data provided by Red de Informaci\u00F3n Agroclim\u00E1tica de Andaluc\u00EDa (RIA)"),
      legal_note_style("https://www.juntadeandalucia.es/agriculturaypesca/ifapa/riaweb/web/")
    ))

    res
  })
}



# resolution specific carpentry -------------------------------------------------------------------------

.ria_monthly_select_quos <- function() {
  return(rlang::quos(year = "anyo", month = "mes"))
}

.ria_daily_select_quos <- function() {
  return(rlang::quos(timestamp = "fecha"))
}

.ria_monthly_mutate_quos <- function() {
  return(rlang::quos(timestamp = as.Date(glue::glue("{year}-{month}-01"))))
}

.ria_daily_mutate_quos <- function() {
  return(rlang::quos())
}
