% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/computeInformation.R
\name{computeMutualInfo}
\alias{computeMutualInfo}
\title{Compute (conditional) mutual information}
\usage{
computeMutualInfo(
  x,
  y,
  df_conditioning = NULL,
  maxbins = NULL,
  cplx = c("nml", "bic"),
  n_eff = -1,
  sample_weights = NULL,
  is_continuous = NULL,
  plot = FALSE
)
}
\arguments{
\item{x}{[a vector]
The \eqn{X} vector that contains the observational data of the first variable.}

\item{y}{[a vector]
The \eqn{Y} vector that contains the observational data of the second variable.}

\item{df_conditioning}{[a data frame]
The data frame of the observations of the conditioning variables.}

\item{maxbins}{[an integer]
When the data contain continuous variables, the maximum number of bins
allowed during the discretization. A smaller number makes the computation
faster, a larger number allows finer discretization.}

\item{cplx}{[a string]
The complexity model:
\itemize{
\item["bic"] Bayesian Information Criterion
\item["nml"] Normalized Maximum Likelihood, more accurate complexity cost
compared to BIC, especially on small sample size.
}}

\item{n_eff}{[an integer]
The effective number of samples. When there is significant autocorrelation
between successive samples, you may want to specify an effective number of
samples that is lower than the total number of samples.}

\item{sample_weights}{[a vector of floats]
Individual weights for each sample, used for the same reason as the effective
number of samples but with individual weights.}

\item{is_continuous}{[a vector of booleans]
Specify if each variable is to be treated as continuous (TRUE) or discrete
(FALSE), must be of length `ncol(df_conditioning) + 2`, in the order
\eqn{X, Y, U1, U2, ...}. If not specified, factors and character vectors are
considered as discrete, and numerical vectors as continuous.}

\item{plot}{[a boolean]
Specify whether the resulting XY optimum discretization is to be plotted
(requires `ggplot2` and `gridExtra`).}
}
\value{
A list that contains :
\itemize{
\item cutpoints1: Only when \eqn{X} is continuous, a vector containing
  the cutpoints for the partitioning of \eqn{X}.
\item cutpoints2: Only when \eqn{Y} is continuous, a vector containing
  the cutpoints for the partitioning of \eqn{Y}.
\item n_iterations: Only when at least one of the input variables is
  continuous, the number of iterations it takes to reach the convergence of
  the estimated information.
\item iteration1, iteration2, ... Only when at least one of the input
  variables is continuous, the list of vectors of cutpoints of each
  iteration.
\item info: The estimation of (conditional) mutual information without the
complexity cost.
\item infok: The estimation of (conditional) mutual information with the
complexity cost (\eqn{Ik = I - cplx}).
\item plot: Only when `plot == TRUE`, the plot object.
}
}
\description{
For discrete or categorical variables, the (conditional)
mutual information is computed using the empirical frequencies minus a
complexity cost (computed as BIC or with the Normalized Maximum Likelihood).
When continuous variables are present, each continuous variable is
discretized for each mutual information estimate so as to maximize the
mutual information minus the complexity cost (see Cabeli 2020).
}
\details{
For a pair of continuous variables \eqn{X} and \eqn{Y}, the mutual
information \eqn{I(X;Y)} will be computed iteratively. In each iteration, the
algorithm optimizes the partitioning of \eqn{X} and then of \eqn{Y},
in order to maximize
\deqn{Ik(X_{d};Y_{d}) = I(X_{d};Y_{d}) - cplx(X_{d};Y_{d})}
where \eqn{cplx(X_{d}; Y_{d})} is the complexity cost of the corresponding
partitioning (see Cabeli 2020).
Upon convergence, the information terms \eqn{I(X_{d};Y_{d})}
and \eqn{Ik(X_{d};Y_{d})}, as well as the partitioning of \eqn{X_{d}}
and \eqn{Y_{d}} in terms of cutpoints, are returned.

For conditional mutual information with a conditioning set \eqn{U}, the
computation is done based on
\deqn{
  Ik(X;Y|U) = 0.5*(Ik(X_{d};Y_{d},U_{d}) - Ik(X_{d};U_{d})
                 + Ik(Y_{d};X_{d},U_{d}) - Ik(Y_{d};U_{d})),
}
where each of the four summands is estimated separately.
}
\examples{
library(miic)
N <- 1000
# Dependence, conditional independence : X <- Z -> Y
Z <- runif(N)
X <- Z * 2 + rnorm(N, sd = 0.2)
Y <- Z * 2 + rnorm(N, sd = 0.2)
res <- computeMutualInfo(X, Y, plot = FALSE)
message("I(X;Y) = ", res$info)
res <- computeMutualInfo(X, Y, df_conditioning = matrix(Z, ncol = 1), plot = FALSE)
message("I(X;Y|Z) = ", res$info)

\donttest{
# Conditional independence with categorical conditioning variable : X <- Z -> Y
Z <- sample(1:3, N, replace = TRUE)
X <- -as.numeric(Z == 1) + as.numeric(Z == 2) + 0.2 * rnorm(N)
Y <- as.numeric(Z == 1) + as.numeric(Z == 2) + 0.2 * rnorm(N)
res <- miic::computeMutualInfo(X, Y, cplx = "nml")
message("I(X;Y) = ", res$info)
res <- miic::computeMutualInfo(X, Y, matrix(Z, ncol = 1), is_continuous = c(TRUE, TRUE, FALSE))
message("I(X;Y|Z) = ", res$info)


# Independence, conditional dependence : X -> Z <- Y
X <- runif(N)
Y <- runif(N)
Z <- X + Y + rnorm(N, sd = 0.1)
res <- computeMutualInfo(X, Y, plot = TRUE)
message("I(X;Y) = ", res$info)
res <- computeMutualInfo(X, Y, df_conditioning = matrix(Z, ncol = 1), plot = TRUE)
message("I(X;Y|Z) = ", res$info)
}
}
\references{
\itemize{
\item Cabeli \emph{et al.}, PLoS Comput. Biol. 2020, \href{https://journals.plos.org/ploscompbiol/article?id=10.1371/journal.pcbi.1007866}{Learning clinical networks from medical records based on information estimates in mixed-type data}
\item Affeldt \emph{et al.}, UAI 2015, \href{https://auai.org/uai2015/proceedings/papers/293.pdf}{Robust Reconstruction of Causal Graphical Models based on Conditional 2-point and 3-point Information}
}
}
