% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/discretizeMutual.R
\name{discretizeMutual}
\alias{discretizeMutual}
\title{Iterative dynamic programming for (conditional) mutual information through
optimized discretization.}
\usage{
discretizeMutual(
  x,
  y,
  matrix_u = NULL,
  maxbins = NULL,
  cplx = "nml",
  n_eff = NULL,
  sample_weights = NULL,
  is_continuous = NULL,
  plot = TRUE
)
}
\arguments{
\item{x}{[a vector]
The \eqn{X} vector that contains the observational data of the first variable.}

\item{y}{[a vector]
The \eqn{Y} vector that contains the observational data of the second variable.}

\item{matrix_u}{[a numeric matrix]
The matrix with the observations of as many columns as conditioning variables.}

\item{maxbins}{[an int]
The maximum number of bins desired in the discretization. A lower number makes the computation faster, a higher
number allows finer discretization (by default : 5 * cubic root of N).}

\item{cplx}{[a string]
The complexity used in the dynamic programming:
\itemize{
\item["bic"] Bayesian Information Criterion
\item["nml"] Normalized Maximum Likelihood, more accurate complexity cost
compared to BIC, especially on small sample size.
}}

\item{n_eff}{[an integer]
The effective number of samples. When there is significant autocorrelation
between successive samples, you may want to specify an effective number of
samples that is lower than the total number of samples.}

\item{sample_weights}{[a vector of floats]
Individual weights for each sample, used for the same reason as the effective
number of samples but with individual weights.}

\item{is_continuous}{[a vector of booleans]
Specify if each variable is to be treated as continuous (TRUE)
or discrete (FALSE) in a logical vector of length ncol(matrix_u) + 2,
in the order [X, Y, U1, U2...]. By default, factors and character vectors
are treated as discrete, and numerical vectors as continuous.}

\item{plot}{[a boolean]
Specify whether the resulting XY optimum discretization is to be plotted
(requires `ggplot2` and `gridExtra`).}
}
\value{
A list that contains :
\itemize{
\item{two vectors containing the cutpoints for each variable :
\emph{cutpoints1} corresponds to \emph{x},
\emph{cutpoints2} corresponds to \emph{y}.}
\item{\emph{n_iterations} is the number of iterations performed before
convergence of the (C)MI estimation.}
\item{\emph{iteration1, iteration2, ...}, lists containing
the cutpoint vectors for each iteration.}
\item{\emph{info} and \emph{infok}, the estimated (C)MI value
and (C)MI minus the complexity cost.}
\item{if \emph{plot} == TRUE, a plot object (requires ggplot2 and gridExtra).}
}
}
\description{
This function chooses cutpoints in the input distributions by
maximizing the mutual information minus a complexity cost
(computed as BIC or with the Normalized Maximum Likelihood).
The (conditional) mutual information computed on the optimized discretized
distributions effectively estimates the mutual information of the original
continuous variables.
}
\details{
For a pair of continuous variables \eqn{X} and \eqn{Y},
the algorithm will iteratively choose cutpoints on \eqn{X} then on \eqn{Y},
maximizing \eqn{I(X_{d};Y_{d}) - cplx(X_{d};Y_{d})} where
\eqn{cplx(X_{d};Y_{d})} is the complexity cost of the considered
discretizations of \eqn{X} and \eqn{Y} (see Cabeli 2020).
Upon convergence, the discretization scheme of \eqn{X_{d}} and \eqn{Y_{d}}
is returned as well as \eqn{I(X_{d};Y_{d})}
and \eqn{I(X_{d};Y_{d})-cplx(X_{d};Y_{d})}.

With a set of conditioning variables \eqn{U}, the discretization scheme
maximizes each term of the sum
\eqn{I(X;Y|U) \sim 0.5*(I(X_{d};Y_{d}, U_{d}) - I(X_{d};U_{d}) + I(Y_{d};X_{d}, U_{d}) - I(Y_{d};U_{d}))}.

Discrete variables can be passed as factors and will be used "as is" to maximize each term.
}
\examples{
library(miic)
N <- 1000
# Dependence, conditional independence : X <- Z -> Y
Z <- runif(N)
X <- Z * 2 + rnorm(N, sd = 0.2)
Y <- Z * 2 + rnorm(N, sd = 0.2)
res <- discretizeMutual(X, Y, plot = FALSE)
message("I(X;Y) = ", res$info)
res <- discretizeMutual(X, Y, matrix_u = matrix(Z, ncol = 1), plot = FALSE)
message("I(X;Y|Z) = ", res$info)

\donttest{
# Conditional independence with categorical conditioning variable : X <- Z -> Y
Z <- sample(1:3, N, replace = TRUE)
X <- -as.numeric(Z == 1) + as.numeric(Z == 2) + 0.2 * rnorm(N)
Y <- as.numeric(Z == 1) + as.numeric(Z == 2) + 0.2 * rnorm(N)
res <- miic::discretizeMutual(X, Y, cplx = "nml")
message("I(X;Y) = ", res$info)
res <- miic::discretizeMutual(X, Y, matrix(Z, ncol = 1), is_continuous = c(TRUE, TRUE, FALSE))
message("I(X;Y|Z) = ", res$info)


# Independence, conditional dependence : X -> Z <- Y
X <- runif(N)
Y <- runif(N)
Z <- X + Y + rnorm(N, sd = 0.1)
res <- discretizeMutual(X, Y, plot = TRUE)
message("I(X;Y) = ", res$info)
res <- discretizeMutual(X, Y, matrix_u = matrix(Z, ncol = 1), plot = TRUE)
message("I(X;Y|Z) = ", res$info)
}
}
\references{
\itemize{
\item Cabeli \emph{et al.}, PLoS Comput. Biol. 2020, \href{https://journals.plos.org/ploscompbiol/article?id=10.1371/journal.pcbi.1007866}{Learning clinical networks from medical records based on information estimates in mixed-type data}
}
}
