#' Estimate Time-Varying Case Reproduction Number Using Wallinga-Lipsitch Method
#'
#' Estimates the time-varying case reproduction number (R_c) from daily incidence data
#' using the method developed by Wallinga and Lipsitch (2007). The case reproduction
#' number represents the average number of secondary infections generated by cases
#' with symptom onset at time t, making it useful for retrospective outbreak analysis.
#'
#' The method calculates the relative likelihood that each earlier case infected each
#' later case based on their time differences and the serial interval distribution, then
#' aggregates these likelihoods to estimate reproduction numbers. The approach makes
#' minimal assumptions beyond specifying the serial interval distribution.
#'
#' Key features:
#' \itemize{
#'   \item \strong{Pairwise likelihood approach}: Considers all epidemiologically plausible transmission pairs (earlier to later cases)
#'   \item \strong{Right-truncation correction}: Adjusts for unobserved future cases (see \code{\link{calculate_truncation_correction}})
#'   \item \strong{Bootstrap confidence intervals}: Quantifies estimation uncertainty
#'   \item \strong{Temporal shifting}: Optional alignment with instantaneous R estimates
#'   \item \strong{Flexible smoothing}: User-controlled temporal smoothing of estimates
#' }
#'
#' @param incidence numeric vector; daily case counts. Must be non-negative integers
#'                   or counts. Length must match \code{dates}
#' @param dates vector; dates corresponding to each incidence count. Must be the same
#'              length as \code{incidence}. Can be Date objects or anything coercible
#'              to dates
#' @param si_mean numeric; mean of the serial interval distribution in days. Must be
#'                positive. Typically estimated from contact tracing data or literature
#' @param si_sd numeric; standard deviation of the serial interval distribution in days.
#'              Must be positive
#' @param si_dist character; distribution family for the serial interval. Options:
#' \itemize{
#'   \item \code{"gamma"} (default): Recommended for most applications as it naturally
#'         restricts to positive values
#'   \item \code{"normal"}: Allows negative serial intervals, useful when co-primary
#'         infections are suspected
#' }
#' @param smoothing integer; window size for temporal smoothing of R estimates. Use 0
#'                  for no smoothing (default), or positive integers for moving average
#'                  smoothing over the specified number of days
#' @param bootstrap logical; whether to calculate bootstrap confidence intervals.
#'                  Defaults to \code{FALSE}. Setting to \code{TRUE} increases
#'                  computation time but provides uncertainty quantification
#' @param n_bootstrap integer; number of bootstrap samples when \code{bootstrap = TRUE}.
#'                    More samples provide more stable intervals but increase computation
#'                    time. Defaults to 1000
#' @param conf_level numeric; confidence level for bootstrap intervals, between 0 and 1.
#'                   Defaults to 0.95 (95% confidence intervals)
#' @param shift logical; whether to shift R estimates forward by one mean serial interval.
#'              When \code{TRUE}, adds a \code{shifted_date} column for comparison with
#'              instantaneous reproduction number estimates. Defaults to \code{FALSE}
#'
#' @return A data frame with the following columns:
#' \itemize{
#'   \item \code{date}: Original input dates
#'   \item \code{incidence}: Original input case counts
#'   \item \code{R}: Estimated case reproduction number
#'   \item \code{R_corrected}: Case reproduction number with right-truncation correction
#'   \item \code{R_lower, R_upper}: Bootstrap confidence intervals for R (if \code{bootstrap = TRUE})
#'   \item \code{R_corrected_lower, R_corrected_upper}: Bootstrap confidence intervals for
#'         R_corrected (if \code{bootstrap = TRUE})
#'   \item \code{shifted_date}: Dates shifted forward by mean serial interval (if \code{shift = TRUE})
#' }
#'
#' @details
#' The Wallinga-Lipsitch method estimates the case reproduction number by:
#' \enumerate{
#'   \item Computing transmission likelihoods from each earlier case to each later case based on the serial interval distribution
#'   \item Normalizing these likelihoods so they sum to 1 for each potential infectee
#'   \item Aggregating normalized likelihoods to estimate expected secondary cases per primary case
#'   \item Applying corrections for right-truncation bias
#' }
#'
#' \strong{Right-truncation correction} accounts for secondary cases that may occur
#' after the observation period ends (see \code{\link{calculate_truncation_correction}}). This correction is particularly important for recent cases in the time series.
#'
#' \strong{Bootstrap confidence intervals} are calculated by resampling individual
#' cases with replacement, providing non-parametric uncertainty estimates that
#' account for both Poisson sampling variation and method uncertainty.
#'
#' @note The case reproduction number differs from the instantaneous reproduction
#'       number in timing: R_c reflects the reproductive potential of cases by their
#'       symptom onset date, while instantaneous R reflects transmission potential
#'       at the time of infection. Use \code{shift = TRUE} for comparisons with
#'       instantaneous R estimates.
#'
#' @seealso \code{\link{si_estim}} for serial interval estimation,
#'          \code{\link{generate_synthetic_epidemic}} for testing data,
#'          \code{\link{calculate_truncation_correction}} for right-truncation correction
#'
#' @references
#' Wallinga J, Lipsitch M (2007). How generation intervals shape the relationship between growth rates and reproductive numbers. Proceedings of the Royal Society B: Biological Sciences, 274(1609), 599-604. \doi{10.1098/rspb.2006.3754}
#'
#' @export
#' @examples
#' # Example 1: Basic usage with synthetic data
#' set.seed(123)
#' dates <- seq(as.Date("2023-01-01"), by = "day", length.out = 30)
#' incidence <- c(1, 2, 4, 7, 12, 15, 18, 20, 22, 19,
#'                16, 14, 11, 9, 7, 5, 4, 3, 2, 1,
#'                rep(0, 10))
#'
#' # Estimate reproduction number
#' result <- wallinga_lipsitch(
#'   incidence = incidence,
#'   dates = dates,
#'   si_mean = 7,
#'   si_sd = 3,
#'   si_dist = "gamma"
#' )
#'
#' # View results
#' head(result)
#'
#' # Example 2: With bootstrap confidence intervals
#' result_ci <- wallinga_lipsitch(
#'   incidence = incidence,
#'   dates = dates,
#'   si_mean = 7,
#'   si_sd = 3,
#'   si_dist = "gamma",
#'   bootstrap = TRUE,
#'   n_bootstrap = 500  # Reduced for faster example
#' )
#'
#' # Plot results with confidence intervals
#' if (require(ggplot2)) {
#'   library(ggplot2)
#'   ggplot(result_ci, aes(x = date)) +
#'     geom_ribbon(aes(ymin = R_corrected_lower, ymax = R_corrected_upper),
#'                 alpha = 0.3, fill = "blue") +
#'     geom_line(aes(y = R_corrected), color = "blue", size = 1) +
#'     geom_hline(yintercept = 1, linetype = "dashed", color = "red") +
#'     labs(x = "Date", y = "Reproduction Number",
#'          title = "Time-varying Reproduction Number") +
#'     theme_minimal()
#' }
#'
#' # Example 3: With smoothing and shifting
#' result_smooth <- wallinga_lipsitch(
#'   incidence = incidence,
#'   dates = dates,
#'   si_mean = 7,
#'   si_sd = 3,
#'   si_dist = "gamma",
#'   smoothing = 7,  # 7-day smoothing window
#'   shift = TRUE    # Shift for comparison with instantaneous R
#' )
#'
#' # Example 4: Using normal distribution for serial interval
#' result_normal <- wallinga_lipsitch(
#'   incidence = incidence,
#'   dates = dates,
#'   si_mean = 6,
#'   si_sd = 2,
#'   si_dist = "normal",
#'   smoothing = 5
#' )
#'
wallinga_lipsitch <- function(
  incidence,
  dates,
  si_mean,
  si_sd,
  si_dist = "gamma",
  smoothing = 0,
  bootstrap = FALSE,
  n_bootstrap = 1000,
  conf_level = 0.95,
  shift = FALSE
) {
  # Input validation
  stopifnot(length(incidence) == length(dates))
  stopifnot(is.numeric(incidence))
  stopifnot(all(incidence >= 0))
  stopifnot(si_mean > 0)
  stopifnot(si_sd > 0)
  stopifnot(si_dist %in% c("gamma", "normal"))

  # Create matrix of day differences and calculate serial interval probabilities
  day_diffs <- create_day_diff_matrix(dates)
  si_prob <- calculate_si_probability_matrix(
    day_diffs,
    si_mean,
    si_sd,
    si_dist
  )

  # Calculate point estimates
  point_estimates <- calculate_r_estimates(
    incidence,
    si_prob,
    dates,
    si_mean,
    si_sd,
    si_dist,
    smoothing
  )

  # Initialize result data frame
  result <- data.frame(
    date = dates,
    incidence = incidence,
    R = point_estimates$r,
    R_corrected = point_estimates$r_corrected
  )

  # Add bootstrap confidence intervals if requested
  if (bootstrap) {
    ci_estimates <- calculate_bootstrap_ci(
      incidence,
      si_prob,
      dates,
      si_mean,
      si_sd,
      si_dist,
      smoothing,
      n_bootstrap,
      conf_level
    )

    # Add to result data frame
    result$R_lower <- ci_estimates$r_lower
    result$R_upper <- ci_estimates$r_upper
    result$R_corrected_lower <- ci_estimates$r_corrected_lower
    result$R_corrected_upper <- ci_estimates$r_corrected_upper
  }

  # Apply shift if requested
  if (shift) {
    # Calculate the number of days to shift based on the mean serial interval
    shift_days <- round(si_mean)

    # Create new date vector that's shifted forward by the serial interval
    shifted_dates <- dates + shift_days

    # Add the shifted dates to the results dataframe
    result$shifted_date <- shifted_dates

    # Warn if shift extends beyond available data
    if (max(shifted_dates) > max(dates) + shift_days) {
      warning(
        "Shifted dates extend beyond the available data range. Interpret with caution."
      )
    }
  }

  return(result)
}
