% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate_means.R
\name{estimate_means}
\alias{estimate_means}
\title{Estimate Marginal Means (Model-based average at each factor level)}
\usage{
estimate_means(
  model,
  by = "auto",
  predict = NULL,
  ci = 0.95,
  estimate = NULL,
  transform = NULL,
  keep_iterations = FALSE,
  backend = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{model}{A statistical model.}

\item{by}{The (focal) predictor variable(s) at which to evaluate the desired
effect / mean / contrasts. Other predictors of the model that are not
included here will be collapsed and "averaged" over (the effect will be
estimated across them). \code{by} can be a character (vector) naming the focal
predictors, optionally including representative values or levels at which
focal predictors are evaluated (e.g., \code{by = "x = c(1, 2)"}). When \code{estimate}
is \emph{not} \code{"average"}, the \code{by} argument is used to create a "reference grid"
or "data grid" with representative values for the focal predictors. In this
case, \code{by} can also be list of named elements. See details in
\code{\link[insight:get_datagrid]{insight::get_datagrid()}} to learn more about how to create data grids for
predictors of interest.}

\item{predict}{Is passed to the \code{type} argument in \code{emmeans::emmeans()} (when
\code{backend = "emmeans"}) or in \code{marginaleffects::avg_predictions()} (when
\code{backend = "marginaleffects"}). Valid options for \code{predict} are:
\itemize{
\item \code{backend = "marginaleffects"}: \code{predict} can be \code{"response"}, \code{"link"},
\code{"inverse_link"} or any valid \code{type} option supported by model's class
\code{predict()} method (e.g., for zero-inflation models from package
\strong{glmmTMB}, you can choose \code{predict = "zprob"} or \code{predict = "conditional"}
etc., see \link[glmmTMB:predict.glmmTMB]{glmmTMB::predict.glmmTMB}). By default, when \code{predict = NULL},
the most appropriate transformation is selected, which usually returns
predictions or contrasts on the response-scale. The \code{"inverse_link"} is a
special option, comparable to \emph{marginaleffects}' \code{invlink(link)} option. It
will calculate predictions on the link scale and then back-transform to the
response scale.
\item \code{backend = "emmeans"}: \code{predict} can be \code{"response"}, \code{"link"}, \code{"mu"},
\code{"unlink"}, or \code{"log"}. If \code{predict = NULL} (default), the most appropriate
transformation is selected (which usually is \code{"response"}). See also
\href{https://CRAN.R-project.org/package=emmeans/vignettes/transformations.html}{this vignette}.
}

See also section \emph{Predictions on different scales}.}

\item{ci}{Confidence Interval (CI) level. Default to \code{0.95} (\verb{95\%}).}

\item{estimate}{The \code{estimate} argument determines how predictions are
averaged ("marginalized") over variables not specified in \code{by} or \code{contrast}
(non-focal predictors). It controls whether predictions represent a "typical"
individual, an "average" individual from the sample, or an "average"
individual from a broader population.
\itemize{
\item \code{"typical"} (Default): Calculates predictions for a balanced data grid
representing all combinations of focal predictor levels (specified in \code{by}).
For non-focal numeric predictors, it uses the mean; for non-focal
categorical predictors, it marginalizes (averages) over the levels. This
represents a "typical" observation based on the data grid and is useful for
comparing groups. It answers: "What would the average outcome be for a
'typical' observation?". This is the default approach when estimating
marginal means using the \emph{emmeans} package.
\item \code{"average"}: Calculates predictions for each observation in the sample and
then averages these predictions within each group defined by the focal
predictors. This reflects the sample's actual distribution of non-focal
predictors, not a balanced grid. It answers: "What is the predicted value
for an average observation in my data?"
\item \code{"population"}: "Clones" each observation, creating copies with all
possible combinations of focal predictor levels. It then averages the
predictions across these "counterfactual" observations (non-observed
permutations) within each group. This extrapolates to a hypothetical
broader population, considering "what if" scenarios. It answers: "What is
the predicted response for the 'average' observation in a broader possible
target population?" This approach entails more assumptions about the
likelihood of different combinations, but can be more apt to generalize.
This is also the option that should be used for \strong{G-computation}
(causal inference, see \emph{Chatton and Rohrer 2024}). \code{"counterfactual"} is
an alias for \code{"population"}.
}

You can set a default option for the \code{estimate} argument via \code{options()},
e.g. \code{options(modelbased_estimate = "average")}.

Note following limitations:
\itemize{
\item When you set \code{estimate} to \code{"average"}, it calculates the average based
only on the data points that actually exist. This is in particular
important for two or more focal predictors, because it doesn't generate a
\emph{complete} grid of all theoretical combinations of predictor values.
Consequently, the output may not include all the values.
\item Filtering the output at values of continuous predictors, e.g.
\code{by = "x=1:5"}, in combination with \code{estimate = "average"} may result in
returning an empty data frame because of what was described above. In such
case, you can use \code{estimate = "typical"} or use the \code{newdata} argument to
provide a data grid of predictor values at which to evaluate predictions.
\item \code{estimate = "population"} is not available for \code{estimate_slopes()}.
}}

\item{transform}{A function applied to predictions and confidence intervals
to (back-) transform results, which can be useful in case the regression
model has a transformed response variable (e.g., \code{lm(log(y) ~ x)}). For
Bayesian models, this function is applied to individual draws from the
posterior distribution, before computing summaries. Can also be \code{TRUE}, in
which case \code{insight::get_transformation()} is called to determine the
appropriate transformation-function. Note that no standard errors are returned
when transformations are applied.}

\item{keep_iterations}{If \code{TRUE}, will keep all iterations (draws) of
bootstrapped or Bayesian models. They will be added as additional columns
named \code{iter_1}, \code{iter_2}, and so on. If \code{keep_iterations} is a positive
number, only as many columns as indicated in \code{keep_iterations} will be added
to the output. You can reshape them to a long format by running
\code{\link[bayestestR:reshape_iterations]{bayestestR::reshape_iterations()}}.}

\item{backend}{Whether to use \code{"marginaleffects"} (default) or \code{"emmeans"} as
a backend. Results are usually very similar. The major difference will be
found for mixed models, where \code{backend = "marginaleffects"} will also average
across random effects levels, producing "marginal predictions" (instead of
"conditional predictions", see Heiss 2022).

Another difference is that \code{backend = "marginaleffects"} will be slower than
\code{backend = "emmeans"}. For most models, this difference is negligible. However,
in particular complex models or large data sets fitted with \emph{glmmTMB} can be
significantly slower.

You can set a default backend via \code{options()}, e.g. use
\code{options(modelbased_backend = "emmeans")} to use the \strong{emmeans} package or
\code{options(modelbased_backend = "marginaleffects")} to set \strong{marginaleffects} as
default backend.}

\item{verbose}{Use \code{FALSE} to silence messages and warnings.}

\item{...}{Other arguments passed, for instance, to \code{\link[insight:get_datagrid]{insight::get_datagrid()}},
to functions from the \strong{emmeans} or \strong{marginaleffects} package, or to process
Bayesian models via \code{\link[bayestestR:describe_posterior]{bayestestR::describe_posterior()}}. Examples:
\itemize{
\item \code{insight::get_datagrid()}: Argument such as \code{length}, \code{digits} or \code{range}
can be used to control the (number of) representative values. For integer
variables, \code{protect_integers} modulates whether these should also be
treated as numerics, i.e. values can have fractions or not.
\item \strong{marginaleffects}: Internally used functions are \code{avg_predictions()} for
means and contrasts, and \code{avg_slope()} for slopes. Therefore, arguments for
instance like \code{vcov}, \code{equivalence}, \code{df}, \code{slope}, \code{hypothesis} or even
\code{newdata} can be passed to those functions. A \code{weights} argument is passed
to the \code{wts} argument in \code{avg_predictions()} or \code{avg_slopes()}, however,
weights can only be applied when \code{estimate} is \code{"average"} or
\code{"population"} (i.e. for those marginalization options that do not use data
grids). Other arguments, such as \code{re.form} or \code{allow.new.levels}, may be
passed to \code{predict()} (which is internally used by \emph{marginaleffects}) if
supported by that model class.
\item \strong{emmeans}: Internally used functions are \code{emmeans()} and \code{emtrends()}.
Additional arguments can be passed to these functions.
\item Bayesian models: For Bayesian models, parameters are cleaned using
\code{describe_posterior()}, thus, arguments like, for example, \code{centrality},
\code{rope_range}, or \code{test} are passed to that function.
\item Especially for \code{estimate_contrasts()} with integer focal predictors, for
which contrasts should be calculated, use argument \code{integer_as_continuous}
to set the maximum number of unique values in an integer predictor to treat
that predictor as "discrete integer" or as numeric. For the first case,
contrasts are calculated between values of the predictor, for the latter,
contrasts of slopes are calculated. If the integer has more than
\code{integer_as_continuous} unique values, it is treated as numeric. Defaults
to \code{5}. Set to \code{TRUE} to always treat integer predictors as continuous.
\item For count regression models that use an offset term, use \verb{offset = <value>}
to fix the offset at a specific value. Or use \code{estimate = "average"}, to
average predictions over the distribution of the offset (if appropriate).
}}
}
\value{
A data frame of estimated marginal means.
}
\description{
Estimate average values of the response variable at each factor level or
at representative values, respectively at values defined in a "data grid" or
"reference grid". For plotting, check the examples in
\code{\link[=visualisation_recipe]{visualisation_recipe()}}. See also other related functions such as
\code{\link[=estimate_contrasts]{estimate_contrasts()}} and \code{\link[=estimate_slopes]{estimate_slopes()}}.
}
\details{
The \code{\link[=estimate_slopes]{estimate_slopes()}}, \code{\link[=estimate_means]{estimate_means()}} and \code{\link[=estimate_contrasts]{estimate_contrasts()}}
functions are forming a group, as they are all based on \emph{marginal}
estimations (estimations based on a model). All three are built on the
\strong{emmeans} or \strong{marginaleffects} package (depending on the \code{backend}
argument), so reading its documentation (for instance \code{\link[emmeans:emmeans]{emmeans::emmeans()}},
\code{\link[emmeans:emtrends]{emmeans::emtrends()}} or this \href{https://marginaleffects.com/}{website}) is
recommended to understand the idea behind these types of procedures.
\itemize{
\item Model-based \strong{predictions} is the basis for all that follows. Indeed, the
first thing to understand is how models can be used to make predictions
(see \code{\link[=estimate_relation]{estimate_relation()}}). This corresponds to the predicted response (or
"outcome variable") given specific predictor values of the predictors
(i.e., given a specific data configuration). This is why the concept of
the \link[insight:get_datagrid]{reference grid} is so important for direct
predictions.
\item \strong{Marginal "means"}, obtained via \code{\link[=estimate_means]{estimate_means()}}, are an extension of
such predictions, allowing to "average" (collapse) some of the predictors,
to obtain the average response value at a specific predictors
configuration. This is typically used when some of the predictors of
interest are factors. Indeed, the parameters of the model will usually give
you the intercept value and then the "effect" of each factor level (how
different it is from the intercept). Marginal means can be used to directly
give you the mean value of the response variable at all the levels of a
factor. Moreover, it can also be used to control, or average over
predictors, which is useful in the case of multiple predictors with or
without interactions.
\item \strong{Marginal contrasts}, obtained via \code{\link[=estimate_contrasts]{estimate_contrasts()}}, are themselves
at extension of marginal means, in that they allow to investigate the
difference (i.e., the contrast) between the marginal means. This is, again,
often used to get all pairwise differences between all levels of a factor.
It works also for continuous predictors, for instance one could also be
interested in whether the difference at two extremes of a continuous
predictor is significant.
\item Finally, \strong{marginal effects}, obtained via \code{\link[=estimate_slopes]{estimate_slopes()}}, are
different in that their focus is not values on the response variable, but
the model's parameters. The idea is to assess the effect of a predictor at
a specific configuration of the other predictors. This is relevant in the
case of interactions or non-linear relationships, when the effect of a
predictor variable changes depending on the other predictors. Moreover,
these effects can also be "averaged" over other predictors, to get for
instance the "general trend" of a predictor over different factor levels.
}

\strong{Example:} Let's imagine the following model \code{lm(y ~ condition * x)} where
\code{condition} is a factor with 3 levels A, B and C and \code{x} a continuous
variable (like age for example). One idea is to see how this model performs,
and compare the actual response y to the one predicted by the model (using
\code{\link[=estimate_expectation]{estimate_expectation()}}). Another idea is evaluate the average mean at each of
the condition's levels (using \code{\link[=estimate_means]{estimate_means()}}), which can be useful to
visualize them. Another possibility is to evaluate the difference between
these levels (using \code{\link[=estimate_contrasts]{estimate_contrasts()}}). Finally, one could also estimate
the effect of x averaged over all conditions, or instead within each
condition (using \code{\link[=estimate_slopes]{estimate_slopes()}}).
}
\section{Predictions and contrasts at meaningful values (data grids)}{


To define representative values for focal predictors (specified in \code{by},
\code{contrast}, and \code{trend}), you can use several methods. These values are
internally generated by \code{insight::get_datagrid()}, so consult its
documentation for more details.
\itemize{
\item You can directly specify values as strings or lists for \code{by}, \code{contrast},
and \code{trend}.
\itemize{
\item For numeric focal predictors, use examples like \code{by = "gear = c(4, 8)"},
\code{by = list(gear = c(4, 8))} or \code{by = "gear = 5:10"}
\item For factor or character predictors, use \code{by = "Species = c('setosa', 'virginica')"}
or \code{by = list(Species = c('setosa', 'virginica'))}
}
\item You can use "shortcuts" within square brackets, such as \code{by = "Sepal.Width = [sd]"}
or \code{by = "Sepal.Width = [fivenum]"}
\item For numeric focal predictors, if no representative values are specified
(i.e., \code{by = "gear"} and \emph{not} \code{by = "gear = c(4, 8)"}), \code{length} and
\code{range} control the number and type of representative values for the focal
predictors:
\itemize{
\item \code{length} determines how many equally spaced values are generated.
\item \code{range} specifies the type of values, like \code{"range"} or \code{"sd"}.
\item \code{length} and \code{range} apply to all numeric focal predictors.
\item If you have multiple numeric predictors, \code{length} and \code{range} can accept
multiple elements, one for each predictor (see 'Examples').
}
\item For integer variables, only values that appear in the data will be included
in the data grid, independent from the \code{length} argument. This behaviour
can be changed by setting \code{protect_integers = FALSE}, which will then treat
integer variables as numerics (and possibly produce fractions).
}

See also \href{https://easystats.github.io/modelbased/articles/visualisation_matrix.html}{this vignette}
for some examples.
}

\section{Predictions on different scales}{


The \code{predict} argument allows to generate predictions on different scales of
the response variable. The \code{"link"} option does not apply to all models, and
usually not to Gaussian models. \code{"link"} will leave the values on scale of
the linear predictors. \code{"response"} (or \code{NULL}) will transform them on scale
of the response variable. Thus for a logistic model, \code{"link"} will give
estimations expressed in log-odds (probabilities on logit scale) and
\code{"response"} in terms of probabilities.

To predict distributional parameters (called "dpar" in other packages), for
instance when using complex formulae in \code{brms} models, the \code{predict} argument
can take the value of the parameter you want to estimate, for instance
\code{"sigma"}, \code{"kappa"}, etc.

\code{"response"} and \code{"inverse_link"} both return predictions on the response
scale, however, \code{"response"} first calculates predictions on the response
scale for each observation and \emph{then} aggregates them by groups or levels
defined in \code{by}. \code{"inverse_link"} first calculates predictions on the link
scale for each observation, then aggregates them by groups or levels defined
in \code{by}, and finally back-transforms the predictions to the response scale.
Both approaches have advantages and disadvantages. \code{"response"} usually
produces less biased predictions, but confidence intervals might be outside
reasonable bounds (i.e., for instance can be negative for count data). The
\code{"inverse_link"} approach is more robust in terms of confidence intervals,
but might produce biased predictions. However, you can try to set
\code{bias_correction = TRUE}, to adjust for this bias.

In particular for mixed models, using \code{"response"} is recommended, because
averaging across random effects groups is then more accurate.
}

\section{Finite mixture models}{


For finite mixture models (currently, only the \code{\link[brms:mixture]{brms::mixture()}} family
from package \emph{brms} is supported), use \code{predict = "link"} to return predicted
values stratified by class membership. To predict the class membership, use
\code{predict = "classification"}. See also
\href{https://easystats.github.io/modelbased/articles/practical_growthmixture.html}{this vignette}.
}

\section{Equivalence tests (smallest effect size of interest)}{


There are two ways of performing equivalence tests with \strong{modelbased}.
\itemize{
\item Using the \emph{marginaleffects} machinery

The first is by specifying the \code{equivalence} argument. It takes a numeric
vector of length two, defining the lower and upper bounds of the region of
equivalence (ROPE). The output then includes an additional column
\code{p_Equivalence}. A high p-value (non-significant result) means we reject the
assumption of practical equivalence (and that a minimal important difference
can be assumed, or that the estimate of the predicted value, slope or
contrast is likely outside the ROPE).
\item Using the \code{equivalence_test()} function

The second option is to use the \code{\link[parameters:equivalence_test.lm]{parameters::equivalence_test.lm()}}
function from the \strong{parameters} package on the output of
\code{estimate_means()}, \code{estimate_slopes()} or \code{estimate_contrasts()}. This
method is more flexible and implements different "rules" to calculate
practical equivalence. Furthermore, the rule decisions of accepting,
rejecting, or undecided regarding the null hypothesis of the equivalence
test are also provided. Thus, resulting p-values may differ from those
p-values returned when using the \code{equivalence} argument.
}

The output from \code{equivalence_test()} returns a column \code{SGPV}, the "second
generation p-value", which is equivalent to the \code{p (Equivalence)} column when
using the \code{equivalence} argument. It is basically representative of the ROPE coverage
from the confidence interval of the estimate (i.e. the proportion of the
confidence intervals that lies within the region of practical equivalence).
}

\section{Global Options to Customize Estimation of Marginal Means}{

\itemize{
\item \code{modelbased_backend}: \verb{options(modelbased_backend = <string>)} will set a
default value for the \code{backend} argument and can be used to set the package
used by default to calculate marginal means. Can be \code{"marginaleffects"} or
\code{"emmeans"}.
\item \code{modelbased_estimate}: \verb{options(modelbased_estimate = <string>)} will
set a default value for the \code{estimate} argument.
\item \code{modelbased_integer}: \verb{options(modelbased_integer = <value>)} will set the
minimum number of unique values in an integer predictor to treat that
predictor as a "discrete integer" or as continuous. If the integer has more
than \code{modelbased_integer} unique values, it is treated as continuous. Set
to \code{TRUE} to always treat integer predictors as continuous.
}
}

\examples{
\dontshow{if (all(insight::check_if_installed(c("marginaleffects", "see", "lme4"), quietly = TRUE))) withAutoprint(\{ # examplesIf}
library(modelbased)

# Frequentist models
# -------------------
model <- lm(Petal.Length ~ Sepal.Width * Species, data = iris)

estimate_means(model)

# the `length` argument is passed to `insight::get_datagrid()` and modulates
# the number of representative values to return for numeric predictors
estimate_means(model, by = c("Species", "Sepal.Width"), length = 2)

# an alternative way to setup your data grid is specify the values directly
estimate_means(model, by = c("Species", "Sepal.Width = c(2, 4)"))

# or use one of the many predefined "tokens" that help you creating a useful
# data grid - to learn more about creating data grids, see help in
# `?insight::get_datagrid`.
estimate_means(model, by = c("Species", "Sepal.Width = [fivenum]"))

\dontrun{
# same for factors: filter by specific levels
estimate_means(model, by = "Species = c('versicolor', 'setosa')")
estimate_means(model, by = c("Species", "Sepal.Width = 0"))

# estimate marginal average of response at values for numeric predictor
estimate_means(model, by = "Sepal.Width", length = 5)
estimate_means(model, by = "Sepal.Width = c(2, 4)")

# or provide the definition of the data grid as list
estimate_means(
  model,
  by = list(Sepal.Width = c(2, 4), Species = c("versicolor", "setosa"))
)

# equivalence test: the null-hypothesis is that the estimate is outside
# the equivalence bounds [-4.5, 4.5]
estimate_means(model, by = "Species", equivalence = c(-4.5, 4.5))

# Methods that can be applied to it:
means <- estimate_means(model, by = c("Species", "Sepal.Width = 0"))

plot(means) # which runs visualisation_recipe()
standardize(means)

# grids for numeric predictors, combine range and length
model <- lm(Sepal.Length ~ Sepal.Width * Petal.Length, data = iris)

# create a "grid": value range for first numeric predictor, mean +/-1 SD
# for remaining numeric predictors.
estimate_means(model, c("Sepal.Width", "Petal.Length"), range = "grid")

# range from minimum to maximum spread over four values,
# and mean +/- 1 SD (a total of three values)
estimate_means(
  model,
  by = c("Sepal.Width", "Petal.Length"),
  range = c("range", "sd"),
  length = c(4, 3)
)

data <- iris
data$Petal.Length_factor <- ifelse(data$Petal.Length < 4.2, "A", "B")

model <- lme4::lmer(
  Petal.Length ~ Sepal.Width + Species + (1 | Petal.Length_factor),
  data = data
)
estimate_means(model)
estimate_means(model, by = "Sepal.Width", length = 3)
}
\dontshow{\}) # examplesIf}
}
\references{
Chatton, A. and Rohrer, J.M. 2024. The Causal Cookbook: Recipes for
Propensity Scores, G-Computation, and Doubly Robust Standardization. Advances
in Methods and Practices in Psychological Science. 2024;7(1).
\doi{10.1177/25152459241236149}

Dickerman, Barbra A., and Miguel A. Hernán. 2020. Counterfactual Prediction
Is Not Only for Causal Inference. European Journal of Epidemiology 35 (7):
615–17. \doi{10.1007/s10654-020-00659-8}

Heiss, A. (2022). Marginal and conditional effects for GLMMs with
{marginaleffects}. Andrew Heiss. \doi{10.59350/xwnfm-x1827}
}
